<?php
/*
 * ----------------------------------------------------------------------------
 * Va-Stats.inc.php                                                  11/13/2006
 * This script contains functions used to read and write to the statistical
 * data files of VADV.
 * ----------------------------------------------------------------------------
 * Copyright (C) 2002-2005 Steve Winn. All Rights Reserved.
 *
 * The source code contains a fair amount of comments, which may aid anyone
 * attempting to follow the program or to modify it.
 *
 * For licensing and copyright information, view LICENSE.TXT.
 * ----------------------------------------------------------------------------
 */

require_once('include_dir.php');                    // Get include directory.
require_once($includeDir . 'va-functions.inc.php'); // A VADV-PHP script.


/**
* Va-Stats.inc.php :: GetConnectionStats()
*
* This function will retrieve the VADV connection statistics.
*
* @return array
*/
function GetConnectionStats() {
    $fileString = ReadConnectionStats();

    $connectStats = array();
    $connectStats['1200']  = GetLongInt($fileString, 5);
    $connectStats['2400']  = GetLongInt($fileString, 9);
    $connectStats['9600']  = GetLongInt($fileString, 13);
    $connectStats['14400'] = GetLongInt($fileString, 17);
    $connectStats['19200'] = GetLongInt($fileString, 21);
    $connectStats['24000'] = GetLongInt($fileString, 25);
    $connectStats['28800'] = GetLongInt($fileString, 29);
    
    // Get the total...
    $connectStats['total'] = $connectStats['1200']  + $connectStats['2400'] +
                             $connectStats['9600']  + $connectStats['14400'] +
                             $connectStats['19200'] + $connectStats['24000'] +
                             $connectStats['24000'] + $connectStats['28800'];

    return $connectStats;                      // Return the statistics.
}


/**
* Va-Stats.inc.php :: GetDOWStats()
*
* This function will retrieve the VADV Day of Week statistics.
*
* @return array
*/
function GetDOWStats() {
    $fileString = ReadDOWStats();

    $dowStats = array();
    $dowStats['sunday']    = GetLongInt($fileString, 1);
    $dowStats['monday']    = GetLongInt($fileString, 5);
    $dowStats['tuesday']   = GetLongInt($fileString, 9);
    $dowStats['wednesday'] = GetLongInt($fileString, 13);
    $dowStats['thursday']  = GetLongInt($fileString, 17);
    $dowStats['friday']    = GetLongInt($fileString, 21);
    $dowStats['saturday']  = GetLongInt($fileString, 25);

    // Calculate the total...
    $dowStats['total'] = $dowStats['sunday'] +
                         $dowStats['monday'] +
                         $dowStats['tuesday'] +
                         $dowStats['wednesday'] +
                         $dowStats['thursday'] +
                         $dowStats['friday'] +
                         $dowStats['saturday'];

    return $dowStats;
}


/**
* Va-Stats.inc.php :: GetHourStats()
*
* This function will retrieve the VADV hour statistics.
*
* @return array
*/
function GetHourStats() {
    $fileString = ReadHourStats();

    $hourStats = array();
    $hourStats['12am'] = GetLongInt($fileString, 1);
    $hourStats['1am']  = GetLongInt($fileString, 5);
    $hourStats['2am']  = GetLongInt($fileString, 9);
    $hourStats['3am']  = GetLongInt($fileString, 13);
    $hourStats['4am']  = GetLongInt($fileString, 17);
    $hourStats['5am']  = GetLongInt($fileString, 21);
    $hourStats['6am']  = GetLongInt($fileString, 25);
    $hourStats['7am']  = GetLongInt($fileString, 29);
    $hourStats['8am']  = GetLongInt($fileString, 33);
    $hourStats['9am']  = GetLongInt($fileString, 37);
    $hourStats['10am'] = GetLongInt($fileString, 41);
    $hourStats['11am'] = GetLongInt($fileString, 45);
    $hourStats['12pm'] = GetLongInt($fileString, 49);
    $hourStats['1pm']  = GetLongInt($fileString, 53);
    $hourStats['2pm']  = GetLongInt($fileString, 57);
    $hourStats['3pm']  = GetLongInt($fileString, 61);
    $hourStats['4pm']  = GetLongInt($fileString, 65);
    $hourStats['5pm']  = GetLongInt($fileString, 69);
    $hourStats['6pm']  = GetLongInt($fileString, 73);
    $hourStats['7pm']  = GetLongInt($fileString, 77);
    $hourStats['8pm']  = GetLongInt($fileString, 81);
    $hourStats['9pm']  = GetLongInt($fileString, 85);
    $hourStats['10pm'] = GetLongInt($fileString, 89);
    $hourStats['11pm'] = GetLongInt($fileString, 93);
    
    // Calculate the total...
    $hourStats['total'] = $hourStats['12am'] + $hourStats['1am'] +
                          $hourStats['2am']  + $hourStats['3am'] +
                          $hourStats['4am']  + $hourStats['5am'] +
                          $hourStats['6am']  + $hourStats['7am'] +
                          $hourStats['8am']  + $hourStats['9am'] +
                          $hourStats['10am'] + $hourStats['11am'] +
                          $hourStats['12pm'] + $hourStats['1pm'] +
                          $hourStats['2pm']  + $hourStats['3pm'] +
                          $hourStats['4pm']  + $hourStats['5pm'] +
                          $hourStats['6pm']  + $hourStats['7pm'] +
                          $hourStats['8pm']  + $hourStats['9pm'] +
                          $hourStats['10pm'] + $hourStats['11pm'];

    return $hourStats;                      // Return the statistics.
}


/**
* Va-Stats.inc.php :: GetStats()
*
* This function will retrieve the VADV statistics.
*
* @return array
*/
function GetStats($fileString = '') {
    if (!is_string($fileString)) {
        $fileString = (string) $fileString;
    }

    if ($fileString == '') {
        $fileString = ReadStats();
        if ($fileString === FALSE) {
            trigger_error(VA_Error(3));
        }
    }

    $stats = array();
    $stats['callstotal']     = GetLongInt($fileString, 1);
    $stats['callstoday']     = GetLongInt($fileString, 5);
    $stats['netcallstotal']  = GetLongInt($fileString, 9);
    $stats['netcallstoday']  = GetLongInt($fileString, 13);
    $stats['activemintotal'] = GetLongInt($fileString, 17);
    $stats['activemintoday'] = GetLongInt($fileString, 21);
    $stats['idlemintotal']   = GetLongInt($fileString, 25);
    $stats['idlemintoday']   = GetLongInt($fileString, 29);
    $stats['omrtotal']       = GetLongInt($fileString, 33);
    $stats['omrtoday']       = GetLongInt($fileString, 37);
    $stats['daysonline']     = GetLongInt($fileString, 41);
    $stats['downloadstotal'] = GetLongInt($fileString, 45);
    $stats['downloadstoday'] = GetLongInt($fileString, 49);
    $stats['uploadstotal']   = GetLongInt($fileString, 53);
    $stats['uploadstoday']   = GetLongInt($fileString, 57);
    $stats['poststotal']     = GetLongInt($fileString, 61);
    $stats['poststoday']     = GetLongInt($fileString, 65);
    $stats['emailstotal']    = GetLongInt($fileString, 69);
    $stats['emailstoday']    = GetLongInt($fileString, 73);
    // The next two variables are not standard VADV entries.. they are created
    // by VADV-PHP!
    $stats['webcallstotal']  = GetLongInt($fileString, 77, 1);
    $stats['webhitstotal']   = GetLongInt($fileString, 81, 1);
    $stats['webmaxusers']    = GetLongInt($fileString, 85, 1);
    
    return $stats;                          // Return the statistics.
}


/**
* Va-Stats.inc.php :: PutConnectionStats()
*
* This function will write a new file string based on the configuration variable
* passed to it. The new file string can be used to create a new configuration
* file, but you must be careful that the configuration variable is valid.
*
* @param string $fileString
* @param array $configVariable
* @return string
*/
function PutConnectionStats($fileString, $configVariable) {
    if (!is_string($fileString)) {
        $fileString = (string) $fileString;
    }
    if (!is_array($configVariable)) {
        trigger_error(VA_Error(1));
    }

    if (isset($configVariable['1200']))  $fileString = PutLongInt($fileString, $configVariable['1200'], 5);
    if (isset($configVariable['2400']))  $fileString = PutLongInt($fileString, $configVariable['2400'], 9);
    if (isset($configVariable['9600']))  $fileString = PutLongInt($fileString, $configVariable['9600'], 13);
    if (isset($configVariable['14400'])) $fileString = PutLongInt($fileString, $configVariable['14400'], 17);
    if (isset($configVariable['19200'])) $fileString = PutLongInt($fileString, $configVariable['19200'], 21);
    if (isset($configVariable['24000'])) $fileString = PutLongInt($fileString, $configVariable['24000'], 25);
    if (isset($configVariable['28800'])) $fileString = PutLongInt($fileString, $configVariable['28800'], 29);

    return $fileString;                     // Return the new file string.

}


/**
* Va-Stats.inc.php :: PutDOWStats()
*
* This function will write a new file string based on the configuration variable
* passed to it. The new file string can be used to create a new configuration
* file, but you must be careful that the configuration variable is valid.
*
* @param string $fileString
* @param array $configVariable
* @return string
*/
function PutDOWStats($fileString, $configVariable) {
    if (!is_string($fileString)) {
        $fileString = (string) $fileString;
    }
    if (!is_array($configVariable)) {
        trigger_error(VA_Error(1));
    }

    if (isset($configVariable['sunday']))    $fileString = PutLongInt($fileString, $configVariable['sunday'], 1);
    if (isset($configVariable['monday']))    $fileString = PutLongInt($fileString, $configVariable['monday'], 5);
    if (isset($configVariable['tuesday']))   $fileString = PutLongInt($fileString, $configVariable['tuesday'], 9);
    if (isset($configVariable['wednesday'])) $fileString = PutLongInt($fileString, $configVariable['wednesday'], 13);
    if (isset($configVariable['thursday']))  $fileString = PutLongInt($fileString, $configVariable['thursday'], 17);
    if (isset($configVariable['friday']))    $fileString = PutLongInt($fileString, $configVariable['friday'], 21);
    if (isset($configVariable['saturday']))  $fileString = PutLongInt($fileString, $configVariable['saturday'], 25);

    return $fileString;                     // Return the new file string.

}


/**
* Va-Stats.inc.php :: PutHourStats()
*
* This function will write a new file string based on the configuration variable
* passed to it. The new file string can be used to create a new configuration
* file, but you must be careful that the configuration variable is valid.
*
* @param string $fileString
* @param array $configVariable
* @return string
*/
function PutHourStats($fileString, $configVariable) {
    if (!is_string($fileString)) {
        $fileString = (string) $fileString;
    }
    if (!is_array($configVariable)) {
        trigger_error(VA_Error(1));
    }

    if (isset($configVariable['12am'])) $fileString = PutLongInt($fileString, $configVariable['12am'], 1);
    if (isset($configVariable['1am']))  $fileString = PutLongInt($fileString, $configVariable['1am'], 5);
    if (isset($configVariable['2am']))  $fileString = PutLongInt($fileString, $configVariable['2am'], 9);
    if (isset($configVariable['3am']))  $fileString = PutLongInt($fileString, $configVariable['3am'], 13);
    if (isset($configVariable['4am']))  $fileString = PutLongInt($fileString, $configVariable['4am'], 17);
    if (isset($configVariable['5am']))  $fileString = PutLongInt($fileString, $configVariable['5am'], 21);
    if (isset($configVariable['6am']))  $fileString = PutLongInt($fileString, $configVariable['6am'], 25);
    if (isset($configVariable['7am']))  $fileString = PutLongInt($fileString, $configVariable['7am'], 29);
    if (isset($configVariable['8am']))  $fileString = PutLongInt($fileString, $configVariable['8am'], 33);
    if (isset($configVariable['9am']))  $fileString = PutLongInt($fileString, $configVariable['9am'], 37);
    if (isset($configVariable['10am'])) $fileString = PutLongInt($fileString, $configVariable['10am'], 41);
    if (isset($configVariable['11am'])) $fileString = PutLongInt($fileString, $configVariable['11am'], 45);
    if (isset($configVariable['12pm'])) $fileString = PutLongInt($fileString, $configVariable['12pm'], 49);
    if (isset($configVariable['1pm']))  $fileString = PutLongInt($fileString, $configVariable['1pm'], 53);
    if (isset($configVariable['2pm']))  $fileString = PutLongInt($fileString, $configVariable['2pm'], 57);
    if (isset($configVariable['3pm']))  $fileString = PutLongInt($fileString, $configVariable['3pm'], 61);
    if (isset($configVariable['4pm']))  $fileString = PutLongInt($fileString, $configVariable['4pm'], 65);
    if (isset($configVariable['5pm']))  $fileString = PutLongInt($fileString, $configVariable['5pm'], 69);
    if (isset($configVariable['6pm']))  $fileString = PutLongInt($fileString, $configVariable['6pm'], 73);
    if (isset($configVariable['7pm']))  $fileString = PutLongInt($fileString, $configVariable['7pm'], 77);
    if (isset($configVariable['8pm']))  $fileString = PutLongInt($fileString, $configVariable['8pm'], 81);
    if (isset($configVariable['9pm']))  $fileString = PutLongInt($fileString, $configVariable['9pm'], 85);
    if (isset($configVariable['10pm'])) $fileString = PutLongInt($fileString, $configVariable['10pm'], 89);
    if (isset($configVariable['11pm'])) $fileString = PutLongInt($fileString, $configVariable['11pm'], 93);

    return $fileString;                     // Return the new file string.

}


/**
* Va-Stats.inc.php :: PutStats()
*
* This function will write a new file string based on the configuration variable
* passed to it. The new file string can be used to create a new configuration
* file, but you must be careful that the configuration variable is valid.
*
* @param string $fileString
* @param array $configVariable
* @return string
*/
function PutStats($fileString, $configVariable) {
    if (!is_string($fileString)) {
        $fileString = (string) $fileString;
    }
    if (!is_array($configVariable)) {
        trigger_error(VA_Error(1));
    }

    if (isset($configVariable['callstotal']))     $fileString = PutLongInt($fileString, $configVariable['callstotal'], 1);
    if (isset($configVariable['callstoday']))     $fileString = PutLongInt($fileString, $configVariable['callstoday'], 5);
    if (isset($configVariable['netcallstotal']))  $fileString = PutLongInt($fileString, $configVariable['netcallstotal'], 9);
    if (isset($configVariable['netcallstoday']))  $fileString = PutLongInt($fileString, $configVariable['netcallstoday'], 13);
    if (isset($configVariable['activemintotal'])) $fileString = PutLongInt($fileString, $configVariable['activemintotal'], 17);
    if (isset($configVariable['activemintoday'])) $fileString = PutLongInt($fileString, $configVariable['activemintoday'], 21);
    if (isset($configVariable['idlemintotal']))   $fileString = PutLongInt($fileString, $configVariable['idlemintotal'], 25);
    if (isset($configVariable['idlemintoday']))   $fileString = PutLongInt($fileString, $configVariable['idlemintoday'], 29);
    if (isset($configVariable['omrtotal']))       $fileString = PutLongInt($fileString, $configVariable['omrtotal'], 33);
    if (isset($configVariable['omrtoday']))       $fileString = PutLongInt($fileString, $configVariable['omrtoday'], 37);
    if (isset($configVariable['daysonline']))     $fileString = PutLongInt($fileString, $configVariable['daysonline'], 41);
    if (isset($configVariable['downloadstotal'])) $fileString = PutLongInt($fileString, $configVariable['downloadstotal'], 45);
    if (isset($configVariable['downloadstoday'])) $fileString = PutLongInt($fileString, $configVariable['downloadstoday'], 49);
    if (isset($configVariable['uploadstotal']))   $fileString = PutLongInt($fileString, $configVariable['uploadstotal'], 53);
    if (isset($configVariable['uploadstoday']))   $fileString = PutLongInt($fileString, $configVariable['uploadstoday'], 57);
    if (isset($configVariable['poststotal']))     $fileString = PutLongInt($fileString, $configVariable['poststotal'], 61);
    if (isset($configVariable['poststoday']))     $fileString = PutLongInt($fileString, $configVariable['poststoday'], 65);
    if (isset($configVariable['emailstotal']))    $fileString = PutLongInt($fileString, $configVariable['emailstotal'], 69);
    if (isset($configVariable['emailstoday']))    $fileString = PutLongInt($fileString, $configVariable['emailstoday'], 73);
    // The next two variables are not standard VADV entries.. they are created
    // by VADV-PHP!
    if (isset($configVariable['webcallstotal']))  $fileString = PutLongInt($fileString, $configVariable['webcallstotal'], 77, 1);
    if (isset($configVariable['webhitstotal']))   $fileString = PutLongInt($fileString, $configVariable['webhitstotal'], 81, 1);
    if (isset($configVariable['webmaxusers']))    $fileString = PutLongInt($fileString, $configVariable['webmaxusers'], 85, 1);
    
    return $fileString;                     // Return the new file string.
}


/**
* Va-Stats.inc.php :: ReadConnectionStats()
*
* This function will read the connect.dat file.
*
* @return string
*/
function ReadConnectionStats() {
    global $VA_SYSPATHS;                    // Include the global variable.

    // Set the filename.
    $file = $VA_SYSPATHS['data'] . '/CONNECT.DAT';
    $fileString = ReadCfgFile($file);       // Read the file.

    return $fileString;                     // Return the file string.
}


/**
* Va-Stats.inc.php :: ReadDOWStats()
*
* This function will read the dow.dat file.
*
* @return string
*/
function ReadDOWStats() {
    global $VA_SYSPATHS;                    // Include the global variable.

    // Set the filename.
    $file = $VA_SYSPATHS['data'] . '/DOW.DAT';
    $fileString = ReadCfgFile($file);       // Read the file.

    return $fileString;                     // Return the file string.
}


/**
* Va-Stats.inc.php :: ReadHourStats()
*
* This function will read the hour.dat file.
*
* @return string
*/
function ReadHourStats() {
    global $VA_SYSPATHS;                    // Include the global variable.

    // Set the filename.
    $file = $VA_SYSPATHS['data'] . '/HOURS.DAT';
    $fileString = ReadCfgFile($file);       // Read the file.

    return $fileString;                     // Return the file string.
}


/**
* Va-Stats.inc.php :: ReadStats()
*
* This function will read the statsvbbs.dat file.
*
* @return string
*/
function ReadStats() {
    global $VA_SYSPATHS;                    // Include the global variable.

    // Set the filename.
    $file = $VA_SYSPATHS['data'] . '/STATVBBS.DAT';
    $fileString = ReadCfgFile($file);       // Read the file.
    
    return $fileString;                     // Return the file string.
}


/**
* Va-Stats.inc.php :: UserStats()
*
* This function will retrieve user statistics and sort them highest to lowest.
* If securitylevel is greater than zero, then it will only include users with
* a securitylevel lower or equal to the value. If set to zero, then it will
* include all non-deleted users.
*
* The returned array is in the format $userStats['statistic'][position]['value']
* The position starts with 1 which is the largest value. It returns the value,
* the user number, user handle and user's real name to which it belongs.
*
* @param integer $securitylevel
* @return array
*/
function UserStats($securitylevel = 0) {
    if (!is_int($securitylevel)) {
        $securitylevel = (integer) $securitylevel;
    }

    global $VA_SYSPATHS;                    // Include the global variable.

    // Set the filename.
    $file = $VA_SYSPATHS['data'] . '/USERFILE.DAT';
    $fileString = ReadCfgFile($file);       // Read the file.

    $userStats = array();
    $offset = 0;

    // Set the default values.
    $userStats['totalcalls'] = array();
    $userStats['totaltime']  = array();
    $userStats['uplfiles']   = array();
    $userStats['uplkb']      = array();
    $userStats['dlfiles']    = array();
    $userStats['dlkb']       = array();
    $userStats['totalposts'] = array();
    $userStats['totalemail'] = array();

    // Set the counting values.
    $pos[0] = 0;
    $pos['totalcalls'] = 0;
    $pos['totaltime']  = 0;
    $pos['uplfiles']   = 0;
    $pos['uplkb']      = 0;
    $pos['dlfiles']    = 0;
    $pos['dlkb']       = 0;
    $pos['totalposts'] = 0;
    $pos['totalemail'] = 0;

    for ($x = 1; $offset < strlen($fileString); $x++) {
        $user['securitylevel'] = GetInteger($fileString, 374 + $offset);
        if ($user['securitylevel'] > 0) {
            if (($securitylevel == 0) || ($user['securitylevel'] <= $securitylevel)) {
                // Get the user's stats.
                $user['handle']     = GetString($fileString, 1 + $offset, 40);
                $user['realname']   = GetString($fileString, 41 + $offset, 40);
                $user['totalcalls'] = GetLongInt($fileString, 386 + $offset);
                $user['totaltime']  = GetLongInt($fileString, 390 + $offset);
                $user['uplfiles']   = GetLongInt($fileString, 823 + $offset);
                $user['uplkb']      = GetLongInt($fileString, 827 + $offset);
                $user['dlfiles']    = GetLongInt($fileString, 831 + $offset);
                $user['dlkb']       = GetLongInt($fileString, 835 + $offset);
                $user['totalposts'] = GetLongInt($fileString, 839 + $offset);
                $user['totalemail'] = GetLongInt($fileString, 843 + $offset);

                if ($user['totalcalls'] > 0) {
                    $userStats['totalcalls'][$pos['totalcalls']]['userno']        = $x;
                    $userStats['totalcalls'][$pos['totalcalls']]['handle']        = $user['handle'];
                    $userStats['totalcalls'][$pos['totalcalls']]['realname']      = $user['realname'];
                    $userStats['totalcalls'][$pos['totalcalls']]['securitylevel'] = $user['securitylevel'];
                    $userStats['totalcalls'][$pos['totalcalls']]['value']         = $user['totalcalls'];
                    $pos['totalcalls']++;
                }

                if ($user['totaltime'] > 0) {
                    $userStats['totaltime'][$pos['totaltime']]['userno']        = $x;
                    $userStats['totaltime'][$pos['totaltime']]['handle']        = $user['handle'];
                    $userStats['totaltime'][$pos['totaltime']]['realname']      = $user['realname'];
                    $userStats['totaltime'][$pos['totaltime']]['securitylevel'] = $user['securitylevel'];
                    $userStats['totaltime'][$pos['totaltime']]['value']         = $user['totaltime'];
                    $pos['totaltime']++;
                }

                if ($user['uplfiles'] > 0) {
                    $userStats['uplfiles'][$pos['uplfiles']]['userno']        = $x;
                    $userStats['uplfiles'][$pos['uplfiles']]['handle']        = $user['handle'];
                    $userStats['uplfiles'][$pos['uplfiles']]['realname']      = $user['realname'];
                    $userStats['uplfiles'][$pos['uplfiles']]['securitylevel'] = $user['securitylevel'];
                    $userStats['uplfiles'][$pos['uplfiles']]['value']         = $user['uplfiles'];
                    $pos['uplfiles']++;
                }

                if ($user['uplkb'] > 0) {
                    $userStats['uplkb'][$pos['uplkb']]['userno']        = $x;
                    $userStats['uplkb'][$pos['uplkb']]['handle']        = $user['handle'];
                    $userStats['uplkb'][$pos['uplkb']]['realname']      = $user['realname'];
                    $userStats['uplkb'][$pos['uplkb']]['securitylevel'] = $user['securitylevel'];
                    $userStats['uplkb'][$pos['uplkb']]['value']         = $user['uplkb'];
                    $pos['uplkb']++;
                }

                if ($user['dlfiles'] > 0) {
                    $userStats['dlfiles'][$pos['dlfiles']]['userno']        = $x;
                    $userStats['dlfiles'][$pos['dlfiles']]['handle']        = $user['handle'];
                    $userStats['dlfiles'][$pos['dlfiles']]['realname']      = $user['realname'];
                    $userStats['dlfiles'][$pos['dlfiles']]['securitylevel'] = $user['securitylevel'];
                    $userStats['dlfiles'][$pos['dlfiles']]['value']         = $user['dlfiles'];
                    $pos['dlfiles']++;
                }

                if ($user['dlkb'] > 0) {
                    $userStats['dlkb'][$pos['dlkb']]['userno']        = $x;
                    $userStats['dlkb'][$pos['dlkb']]['handle']        = $user['handle'];
                    $userStats['dlkb'][$pos['dlkb']]['realname']      = $user['realname'];
                    $userStats['dlkb'][$pos['dlkb']]['securitylevel'] = $user['securitylevel'];
                    $userStats['dlkb'][$pos['dlkb']]['value']         = $user['dlkb'];
                    $pos['dlkb']++;
                }

                if ($user['totalposts'] > 0) {
                    $userStats['totalposts'][$pos['totalposts']]['userno']        = $x;
                    $userStats['totalposts'][$pos['totalposts']]['handle']        = $user['handle'];
                    $userStats['totalposts'][$pos['totalposts']]['realname']      = $user['realname'];
                    $userStats['totalposts'][$pos['totalposts']]['securitylevel'] = $user['securitylevel'];
                    $userStats['totalposts'][$pos['totalposts']]['value']         = $user['totalposts'];
                    $pos['totalposts']++;
                }

                if ($user['totalemail'] > 0) {
                    $userStats['totalemail'][$pos['totalemail']]['userno']        = $x;
                    $userStats['totalemail'][$pos['totalemail']]['handle']        = $user['handle'];
                    $userStats['totalemail'][$pos['totalemail']]['realname']      = $user['realname'];
                    $userStats['totalemail'][$pos['totalemail']]['securitylevel'] = $user['securitylevel'];
                    $userStats['totalemail'][$pos['totalemail']]['value']         = $user['totalemail'];
                    $pos['totalemail']++;
                }

                $pos[0]++;                  // Increase the main counter.
            }
        }
        $offset = 1048 * $x;
    }

    // Sort...
    if ($pos[0] > 0) {
        foreach ($userStats['totalcalls'] as $row) {
            $sortarr[] = $row['value'];
        }
        array_multisort($sortarr, SORT_DESC, SORT_NUMERIC, $userStats['totalcalls']);
        array_unshift($userStats['totalcalls'], $pos['totalcalls']);

        unset($sortarr);
        $sortarr = array();
        foreach ($userStats['totaltime'] as $row) {
            $sortarr[] = $row['value'];
        }
        array_multisort($sortarr, SORT_DESC, SORT_NUMERIC, $userStats['totaltime']);
        array_unshift($userStats['totaltime'], $pos['totaltime']);

        unset($sortarr);
        $sortarr = array();
        foreach ($userStats['uplfiles'] as $row) {
            $sortarr[] = $row['value'];
        }
        array_multisort($sortarr, SORT_DESC, SORT_NUMERIC, $userStats['uplfiles']);
        array_unshift($userStats['uplfiles'], $pos['uplfiles']);

        unset($sortarr);
        $sortarr = array();
        foreach ($userStats['uplkb'] as $row) {
            $sortarr[] = $row['value'];
        }
        array_multisort($sortarr, SORT_DESC, SORT_NUMERIC, $userStats['uplkb']);
        array_unshift($userStats['uplkb'], $pos['uplkb']);

        unset($sortarr);
        $sortarr = array();
        foreach ($userStats['dlfiles'] as $row) {
            $sortarr[] = $row['value'];
        }
        array_multisort($sortarr, SORT_DESC, SORT_NUMERIC, $userStats['dlfiles']);
        array_unshift($userStats['dlfiles'], $pos['dlfiles']);

        unset($sortarr);
        $sortarr = array();
        foreach ($userStats['dlkb'] as $row) {
            $sortarr[] = $row['value'];
        }
        array_multisort($sortarr, SORT_DESC, SORT_NUMERIC, $userStats['dlkb']);
        array_unshift($userStats['dlkb'], $pos['dlkb']);

        unset($sortarr);
        $sortarr = array();
        foreach ($userStats['totalposts'] as $row) {
            $sortarr[] = $row['value'];
        }
        array_multisort($sortarr, SORT_DESC, SORT_NUMERIC, $userStats['totalposts']);
        array_unshift($userStats['totalposts'], $pos['totalposts']);

        unset($sortarr);
        $sortarr = array();
        foreach ($userStats['totalemail'] as $row) {
            $sortarr[] = $row['value'];
        }
        array_multisort($sortarr, SORT_DESC, SORT_NUMERIC, $userStats['totalemail']);
        array_unshift($userStats['totalemail'], $pos['totalemail']);
    } else {
        $userStats['totalcalls'][0] = 0;
        $userStats['totaltime'][0] = 0;
        $userStats['uplfiles'][0] = 0;
        $userStats['uplkb'][0] = 0;
        $userStats['dlfiles'][0] = 0;
        $userStats['dlkb'][0] = 0;
        $userStats['totalposts'][0] = 0;
        $userStats['totalemail'][0] = 0;
    }

    $userStats[0] = $pos[0];

    return $userStats;                      // Return the array.
}


/**
* Va-Stats.inc.php :: WriteConnectionStats()
*
* This function will write a new connect.dat file based on the file string passed
* to it. If the file string is not in the correct format, then your connect.dat
* file will become corrupt and the BBS could stop functioning!
*
* @param string $fileString
* @return mixed
*/
function WriteConnectionStats($fileString) {
    if (!is_string($fileString)) {
        $fileString = (string) $fileString;
    }

    global $VA_SYSPATHS;                    // Include the global variable.

    // Set the filename.
    $file = $VA_SYSPATHS['data'] . '/CONNECT.DAT';
    $status = WriteCfgFile($file, $fileString);

    return $status;                         // Return the result status.
}


/**
* Va-Stats.inc.php :: WriteDOWStats()
*
* This function will write a new dow.dat file based on the file string passed
* to it.
*
* @param string $fileString
* @return mixed
*/
function WriteDOWStats($fileString) {
    if (!is_string($fileString)) {
        $fileString = (string) $fileString;
    }

    global $VA_SYSPATHS;                    // Include the global variable.

    // Set the filename.
    $file = $VA_SYSPATHS['data'] . '/DOW.DAT';
    $status = WriteCfgFile($file, $fileString);

    return $status;                         // Return the result status.
}


/**
* Va-Stats.inc.php :: WriteHourStats()
*
* This function will write a new hours.dat file based on the file string passed
* to it. If the file string is not in the correct format, then your hours.dat
* file will become corrupt and the BBS could stop functioning!
*
* @param string $fileString
* @return mixed
*/
function WriteHourStats($fileString) {
    if (!is_string($fileString)) {
        $fileString = (string) $fileString;
    }

    global $VA_SYSPATHS;                    // Include the global variable.

    // Set the filename.
    $file = $VA_SYSPATHS['data'] . '/HOURS.DAT';
    $status = WriteCfgFile($file, $fileString);

    return $status;                         // Return the result status.
}


/**
* Va-Stats.inc.php :: WriteStats()
*
* This function will write a new statvbbs.dat file based on the file string
* passed to it. If the file string is not in the correct format, then your
* statvbbs.dat file will become corrupt and the BBS could stop functioning!
*
* @param string $fileString
* @return mixed
*/
function WriteStats($fileString) {
    if (!is_string($fileString)) {
        $fileString = (string) $fileString;
    }

    global $VA_SYSPATHS;                    // Include the global variable.

    // Set the filename.
    $file = $VA_SYSPATHS['data'] . '/STATVBBS.DAT';
    $status = WriteCfgFile($file, $fileString);

    return $status;                         // Return the result status.
}

?>