<?php
/*
 * ----------------------------------------------------------------------------
 * Va-MultiFBX.inc.php                                               11/24/2006
 * This script will retrieve all of the informaton related to the configured
 * multi-feedback settings that are found in MULTIFBX.CFG.
 * ----------------------------------------------------------------------------
 * Copyright (C) 2002-2006 Steve Winn. All Rights Reserved.
 *
 * The source code contains a fair amount of comments, which may aid anyone
 * attempting to follow the program or to modify it.
 *
 * For licensing and copyright information, view LICENSE.TXT.
 * ----------------------------------------------------------------------------
 */

require_once('include_dir.php');                    // Get include directory.
require_once($includeDir . 'va-functions.inc.php'); // A VADV-PHP script.


/**
* Va-MultiFBX.inc.php :: GetMultiFBX()
*
* Retrieve all of the multi-feedback information and store it into a
* multidimensional array.
*
* Array format: $multifbx[1]['setting']
*               $multifbx[0] = Total number of multi-feeback settings.
*
* @return array
*/
function GetMultiFBX() {
    $fileString = ReadMultiFBX();
    if ($fileString === FALSE) {
        trigger_error(VA_Error(3));
    }

    $multifbx = array();                    // Create the array.
    $offset   = 0;

    for ($x = 1; $offset < strlen($fileString); $x++) {
        $multifbx[$x]['desc']       = GetString($fileString, 1 + $offset, 40);
        $multifbx[$x]['handle']     = GetString($fileString, 41 + $offset, 40);
        $multifbx[$x]['userno']     = GetLongInt($fileString, 81 + $offset);
        $multifbx[$x]['netaddress'] = GetString($fileString, 85 + $offset, 80);
        $multifbx[$x]['netno']      = GetInteger($fileString, 165 + $offset);
        $offset = 166 * $x;
    }

    $multifbx[0] = $x - 1;                  // Put the number of configured users in position 0.

    return $multifbx;                       // Return the multi-feedback settings.
}


/**
* Va-MultiFBX.inc.php :: PutMultiFBX()
*
* This function will change the multifbx.cfg to match the configuration variable
* passed. It then returns the new file string. The configuration variable MUST
* be in the format of $variable[1]['setting'].
*
* @param string $fileString
* @param array $configVariable
* @return string
*/
function PutMultiFBX($fileString, $configVariable) {
    if (!is_string($fileString)) {
        $fileString = (string) $fileString;
    }
    if (!is_array($configVariable)) {
        trigger_error(VA_Error(1));
    }

    $length  = count($configVariable);      // Get the number of elements in the config variable.
    $pointer = 0;                           // Reset the pointer.
    $offset  = 0;

    if (isset($configVariable[0])) {
        $length--;
    }

    for ($x = 1; $pointer < $length; $x++) {
        $total = intval(strlen($fileString) / 166);
        if ($x > $total) {
            $fileString = $fileString . str_pad('', 166, chr(0));
        }

        if (isset($configVariable[$x])) {
            $pointer++;
            if (isset($configVariable[$x]['desc']))       $fileString = PutString($fileString, $configVariable[$x]['desc'], 1 + $offset, 40);
            if (isset($configVariable[$x]['handle']))     $fileString = PutString($fileString, $configVariable[$x]['handle'], 41 + $offset, 40);
            if (isset($configVariable[$x]['userno']))     $fileString = PutLongInt($fileString, $configVariable[$x]['userno'], 81 + $offset);
            if (isset($configVariable[$x]['netaddress'])) $fileString = PutString($fileString, $configVariable[$x]['netaddress'], 85 + $offset, 80);
            if (isset($configVariable[$x]['netno']))      $fileString = PutInteger($fileString, $configVariable[$x]['netno'], 165 + $offset);
        }
        $offset = 166 * $x;
    }

    return $fileString;                     // Return the new file string.
}


/**
* Va-MultiFBX.inc.php :: ReadMultiFBX()
*
* This function will read the multifbx.cfg file.
*
* @return string
*/
function ReadMultiFBX() {
    global $VA_SYSPATHS;                    // Include the global variable.

    // Set the filename.
    $file = $VA_SYSPATHS['main'] . '/MULTIFBX.CFG';
    $fileString = ReadCfgFile($file);       // Read the file.

    return $fileString;                     // Return the file string.
}


/**
* Va-MultiFBX.inc.php :: WriteMultiFBX()
*
* This function will write a new multifbx.cfg file based on the file string
* passed to it. If the file string is not in the correct format, then your
* multifbx.cfg file will become corrupt and the BBS could stop functioning!
*
* @param string $fileString
* @return mixed
*/
function WriteMultiFBX($fileString) {
    if (!is_string($fileString)) {
        $fileString = (string) $fileString;
    }

    global $VA_SYSPATHS;                    // Include the global variable.

    // Set the filename.
    $file = $VA_SYSPATHS['main'] . '/MULTIFBX.CFG';
    $status = WriteCfgFile($file, $fileString);

    return $status;                         // Return the result status.
}


// Retrieve the muli-feedback settings automatically.
$VA_MULTIFBX = GetMultiFBX();

?>