<?php
/*
 * ----------------------------------------------------------------------------
 * Va-Download.inc.php                                               11/24/2006
 * This script will retrieve all of the informaton related to the configured
 * download protocols that are found in DOWNLOAD.CFG.
 * ----------------------------------------------------------------------------
 * Copyright (C) 2002-2006 Steve Winn. All Rights Reserved.
 *
 * The source code contains a fair amount of comments, which may aid anyone
 * attempting to follow the program or to modify it.
 *
 * For licensing and copyright information, view LICENSE.TXT.
 * ----------------------------------------------------------------------------
 */

require_once('include_dir.php');                    // Get include directory.
require_once($includeDir . 'va-functions.inc.php'); // A VADV-PHP script.


/**
* Va-Download.inc.php :: GetDownload()
*
* Retrieve all of the download protocol information and store it into a
* multidimensional array.
*
* Array format: $download[1]['setting']
*               $download[0] = Total number of download protocols.
*
* @return array
*/
function GetDownload() {
    $fileString = ReadDownload();
    if ($fileString === FALSE) {
        trigger_error(VA_Error(3));
    }

    $download = array();                    // Create the array.
    $offset   = 0;

    for ($x = 1; $offset < strlen($fileString); $x++) {
        $download[$x]['hotkey']  = GetString($fileString, 1 + $offset, 1);
        $download[$x]['desc']    = GetString($fileString, 2 + $offset, 23);
        $download[$x]['batch']   = GetChar($fileString, 25 + $offset);
        $download[$x]['command'] = GetString($fileString, 26 + $offset, 48);
        $offset = 73 * $x;
    }

    $download[0] = $x - 1;                  // Put the number of download protocols in position 0.

    return $download;                       // Return the download protocols.

}


/**
* Va-Download.inc.php :: PutDownload()
*
* This function will change the download.cfg to match the configuration variable
* passed. It then returns the new file string. The configuration variable MUST
* be in the format of $variable[1]['setting'].
*
* @param string $fileString
* @param array $configVariable
* @return string
*/
function PutDownload($fileString, $configVariable) {
    if (!is_string($fileString)) {
        $fileString = (string) $fileString;
    }
    if (!is_array($configVariable)) {
        trigger_error(VA_Error(1));
    }

    $length  = count($configVariable);      // Get the number of elements in the config variable.
    $pointer = 0;                           // Reset the pointer.
    $offset  = 0;

    if (isset($configVariable[0])) {
        $length--;
    }

    for ($x = 1; $pointer < $length; $x++) {
        $total = intval(strlen($fileString) / 73);
        if ($x > $total) {
            $fileString = $fileString . str_pad('', 73, chr(0));
        }

        if (isset($configVariable[$x])) {
            $pointer++;
            if (isset($configVariable[$x]['hotkey']))  $fileString = PutString($fileString, $configVariable[$x]['hotkey'], 1 + $offset, 1);
            if (isset($configVariable[$x]['desc']))    $fileString = PutString($fileString, $configVariable[$x]['desc'], 2 + $offset, 23);
            if (isset($configVariable[$x]['batch']))   $fileString = PutChar($fileString, $configVariable[$x]['batch'], 25 + $offset);
            if (isset($configVariable[$x]['command'])) $fileString = PutString($fileString, $configVariable[$x]['command'], 26 + $offset, 48);
        }
        $offset = 73 * $x;
    }

    return $fileString;                     // Return the new file string.
}


/**
* Va-Download.inc.php :: ReadDownload()
*
* This function will read the download.cfg file.
*
* @return string
*/
function ReadDownload() {
    global $VA_SYSPATHS;                    // Include the global variable.

    // Set the filename.
    $file = $VA_SYSPATHS['main'] . '/DOWNLOAD.CFG';
    $fileString = ReadCfgFile($file);       // Read the file.

    return $fileString;                     // Return the file string.
}


/**
* Va-Download.inc.php :: WriteDownload()
*
* This function will write a new download.cfg file based on the file string
* passed to it. If the file string is not in the correct format, then your
* download.cfg file will become corrupt and the BBS could stop functioning!
*
* @param string $fileString
* @return mixed
*/
function WriteDownload($fileString) {
    if (!is_string($fileString)) {
        $fileString = (string) $fileString;
    }

    global $VA_SYSPATHS;                    // Include the global variable.

    // Set the filename.
    $file = $VA_SYSPATHS['main'] . '/DOWNLOAD.CFG';
    if (!is_file($file)) {
        return FALSE;
    }

    $status = WriteCfgFile($file, $fileString);

    return $status;                         // Return the result status.
}


// Retrieve the download protocols automatically.
$VA_DOWNLOAD = GetDownload();

?>