<?php
/*
 * ----------------------------------------------------------------------------
 * Va-Doors.inc.php                                                  11/24/2006
 * This script will retrieve all of the informaton related to the configured
 * doors that are found in DOORS.CFG.
 * ----------------------------------------------------------------------------
 * Copyright (C) 2002-2006 Steve Winn. All Rights Reserved.
 *
 * The source code contains a fair amount of comments, which may aid anyone
 * attempting to follow the program or to modify it.
 *
 * For licensing and copyright information, view LICENSE.TXT.
 * ----------------------------------------------------------------------------
 */

require_once('include_dir.php');                    // Get include directory.
require_once($includeDir . 'va-functions.inc.php'); // A VADV-PHP script.


/**
* Va-Doors.inc.php :: GetDoors()
*
* Retrieve all of the doors information and store it into a multidimensional
* array.
*
* Array format: $door[1]['setting']
*               $door[0] = Total number of doors.
*
* @return array
*/
function GetDoors() {
    $fileString = ReadDoors();
    if ($fileString === FALSE) {
        trigger_error(VA_Error(3));
    }

    $doors  = array();                      // Create the array.
    $offset = 0;

    for ($x = 1; $offset < strlen($fileString); $x++) {
        $doors[$x]['desc']       = GetString($fileString, 1 + $offset, 32);
        $doors[$x]['command']    = GetString($fileString, 33 + $offset, 32);
        $doors[$x]['minsl']      = GetInteger($fileString, 65 + $offset);
        $doors[$x]['accflag']    = GetString($fileString, 67 + $offset, 1);
        $doors[$x]['singleuser'] = GetInteger($fileString, 68 + $offset);
        $doors[$x]['credits']    = GetInteger($fileString, 70 + $offset);
        $doors[$x]['extra']      = GetString($fileString, 72 + $offset, 3);
        $offset = 74 * $x;
    }

    $doors[0] = $x - 1;                     // Put the number of doors in position 0.

    return $doors;                          // Return the doors.

}


/**
* Va-Doors.inc.php :: PutDoors()
*
* This function will change the doors.cfg to match the configuration variable
* passed. It then returns the new file string. The configuration variable MUST
* be in the format of $variable[1]['setting'].
*
* @param string $fileString
* @param array $configVariable
* @return string
*/
function PutDoors($fileString, $configVariable) {
    if (!is_string($fileString)) {
        $fileString = (string) $fileString;
    }
    if (!is_array($configVariable)) {
        trigger_error(VA_Error(1));
    }

    $length  = count($configVariable);      // Get the number of elements in the config variable.
    $pointer = 0;                           // Reset the pointer.
    $offset  = 0;

    if (isset($configVariable[0])) {
        $length--;
    }

    for ($x = 1; $pointer < $length; $x++) {
        $total = intval(strlen($fileString) / 74);
        if ($x > $total) {
            $fileString = $fileString . str_pad('', 74, chr(0));
        }
        
        if (isset($configVariable[$x])) {
            $pointer++;
            if (isset($configVariable[$x]['desc']))       $fileString = PutString($fileString, $configVariable[$x]['desc'], 1 + $offset, 32);
            if (isset($configVariable[$x]['command']))    $fileString = PutString($fileString, $configVariable[$x]['command'], 33 + $offset, 32);
            if (isset($configVariable[$x]['minsl']))      $fileString = PutInteger($fileString, $configVariable[$x]['minsl'], 65 + $offset);
            if (isset($configVariable[$x]['accflag']))    $fileString = PutString($fileString, $configVariable[$x]['accflag'], 67 + $offset, 1);
            if (isset($configVariable[$x]['singleuser'])) $fileString = PutInteger($fileString, $configVariable[$x]['singleuser'], 68 + $offset);
            if (isset($configVariable[$x]['credits']))    $fileString = PutInteger($fileString, $configVariable[$x]['credits'], 70 + $offset);
            if (isset($configVariable[$x]['extra']))      $fileString = PutString($fileString, $configVariable[$x]['extra'], 72 + $offset, 3);
        }
        $offset = 74 * $x;
    }

    return $fileString;                     // Return the new file string.
}


/**
* Va-Doors.inc.php :: ReadDoors()
*
* This function will read the doors.cfg file.
*
* @return string
*/
function ReadDoors() {
    global $VA_SYSPATHS;                    // Include the global variable.

    // Set the filename.
    $file = $VA_SYSPATHS['main'] . '/DOORS.CFG';
    $fileString = ReadCfgFile($file);       // Read the file.

    return $fileString;                     // Return the file string.
}


/**
* Va-Doors.inc.php :: WriteDoors()
*
* This function will write a new doors.cfg file based on the file string
* passed to it. If the file string is not in the correct format, then your
* doors.cfg file will become corrupt and the BBS could stop functioning!
*
* @param string $fileString
* @return mixed
*/
function WriteDoors($fileString) {
    if (!is_string($fileString)) {
        $fileString = (string) $fileString;
    }

    global $VA_SYSPATHS;                    // Include the global variable.

    // Set the filename.
    $file = $VA_SYSPATHS['main'] . '/DOORS.CFG';
    $status = WriteCfgFile($file, $fileString);

    return $status;                         // Return the result status.
}


// Retrieve the doors automatically.
$VA_DOORS = GetDoors();

?>