{
  MANNSOCK.PAS
  Win32 socket communications routines

    Written By:   Rick Parrish
    Last Updated: September 07, 2002

  CHANGE LOG

    09/07/02 - First official release

  NOTES

    Untested with Non-Blocking sockets.  Theoretically there should be
    no problems though.
}
unit MannSock;

interface

uses
  Windows, Winsock;

type
  TTelnetState = (tnsDATA, tnsDO, tnsDONT, tnsWILL, tnsWONT, tnsIAC);

  TMannSock = Object
    private
      FBuffer: String;
      FSocket: TSocket;
      FTelnetCommands: Array[0..255] of Boolean;
      FTelnetState: TTelnetState;

      procedure DoRX;
      function Negotiate(ALine: String): String;
    public
      DontClose: Boolean;
      StripLF: Boolean;

      constructor Create;
      destructor Destroy;

      procedure Close(ADestroy: Boolean);
      function Connect(AHost: String; APort: Word): Boolean;
      function Connected: Boolean;
      function KeyPressed: Boolean;
      function Open(ASocket: TSocket): Boolean;
      function ReadBuf: String;
      function ReadKey: Char;
      procedure SendDO(ACh: Char);
      procedure SendDONT(ACh: Char);
      procedure SendWILL(ACh: Char);
      procedure SendWONT(ACh: Char);
      procedure SetBlocking(ABlocking: Boolean);
      procedure Write(ALine: String);
      procedure WriteLn(ALine: String);
  end;

const
  TNO_BINARY  = #0;      // Binary Transmission
  TNO_ECHO    = #1;      // Echo
  TNC_WILL    = #251;    // Indicates the desire to begin
  TNC_WONT    = #252;    // Indicates the refusal to perform,
  TNC_DO      = #253;    // Indicates the request that the
  TNC_DONT    = #254;    // Indicates the demand that the
  TNC_IAC     = #255;    // Data Byte 255.

implementation

constructor TMannSock.Create;
var
   WSAData: TWSAData;
begin
     WSAStartup(MakeWord(2, 0), WSAData);

     DontClose := True;
     FBuffer := '';
     FSocket := INVALID_SOCKET;
     FillChar(FTelnetCommands, SizeOf(FTelnetCommands), 0);
     FTelnetState := tnsData;
     StripLF := True;
end;

destructor TMannSock.Destroy;
begin
     Close(True);
     WSACleanUp;
end;

procedure TMannSock.Close(ADestroy: Boolean);
begin
     if (ADestroy) then
     begin
          Shutdown(FSocket, 2);
          CloseSocket(FSocket);
     end;
     FSocket := INVALID_SOCKET;
end;

function TMannSock.Connect(AHost: String; APort: Word): Boolean;
var
   HostEnt: PHostEnt;
   Remote: TSockAddr;
begin
     if (Connected) then
        Close(True);

     FBuffer := '';
     FillChar(FTelnetCommands, SizeOf(FTelnetCommands), 0);

     HostEnt := GetHostByName(PChar(AHost));
     if (HostEnt = nil) then
        Result := False
     else
     begin
          FillChar(Remote, SizeOf(Remote), 0);
          Remote.Sin_Family := AF_INET;
          Remote.Sin_Port := HToNS(APort);
          Remote.Sin_Addr.S_Addr := LongInt(PLongInt(HostEnt^.H_Addr_List^)^);

          FSocket := Winsock.Socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
          Result := Winsock.Connect(FSocket, Remote, SizeOf(Remote)) <> SOCKET_ERROR;
     end;
end;

function TMannSock.Connected: Boolean;
begin
     Result := FSocket <> INVALID_SOCKET;
end;

{ Reading byte by byte sucks.  But my previous code which called RECV()
  using a large array of Char only worked 99.9% of the time.  EleBBS
  running on FreeBSD seemed to break it somehow and lots of text was lost.
  The code worked fine for every other BBS I tested with, and when compiled
  using Delphi it even worked for EleBBS/FreeBSD, so I am blaming VPascal.
  Hopefully the next build will fix this problem, if it is ever released }
procedure TMannSock.DoRX;
var
   FDSet: TFDSet;
   InBuf: Array[0..1] of Char;
   NumRead: Integer;
   Res: Integer;
   S: String;
   TimeOut: TTImeVal;
begin
     FDSet.FD_Count := 1;
     FDSet.FD_Array[0] := FSocket;
     TimeOut.TV_Sec := 0;
     TimeOut.TV_USec := 0;

     Res := Select(0, @FDSet, nil, nil, @TimeOut);
     if (Res = SOCKET_ERROR) then { Use WSAGetLastError }
        Close(Not(DontClose))
     else
     if (Res = 0) then { No Data Available }
        //Nothing There
     else
     if (Res > 0) then { Data Available }
     begin
          S := '';
          repeat
                NumRead := Recv(FSocket, InBuf, 1, 0);
                if (NumRead = 1) then { Successfully Read Data }
                begin
                     if (InBuf[0] <> #10) or Not(StripLF) then
                        S := S + InBuf;
                end else
                if (NumRead = 0) then { Connection Closed Gracefully }
                   Close(Not(DontClose))
                else
                if (NumRead = SOCKET_ERROR) then { Use WSAGetLastError }
                   Close(Not(DontClose));
          until Not(Connected) or (Select(0, @FDSet, nil, nil, @TimeOut) < 1);
          if (S <> '') then
             FBuffer := FBuffer + Negotiate(S);
     end;
end;

function TMannSock.KeyPressed: Boolean;
begin
     if (Length(FBuffer) = 0) then
        DoRX;
     Result := Length(FBuffer) > 0;
end;

function TMannSock.Negotiate(ALine: String): String;
var
   I: Integer;
   Res: String;
begin
     Res := '';
     for I := 1 to Length(ALine) do
     begin
          case FTelnetState of
               tnsData: begin
                             if (ALine[I] = TNC_IAC) then
                                FTelnetState := tnsIAC
                             else
                                 Res := Res + ALine[I];
                        end;
               tnsIAC: begin
                            case ALine[I] of
                                 TNC_IAC: begin
                                               FTelnetState := tnsData;
                                               Res := Res + TNC_IAC;
                                          end;
                                 TNC_WILL: FTelnetState := tnsWILL;
                                 TNC_WONT: FTelnetState := tnsWONT;
                                 TNC_DO: FTelnetState := tnsDO;
                                 TNC_DONT: FTelnetState := tnsDONT;
                            else
                                FTelnetState := tnsData;
                            end;
                       end;
               tnsWILL: begin
                             if (ALine[I] = TNO_BINARY) or (ALine[I] = TNO_ECHO) then
                                SendDO(ALine[I])
                             else
                                 SendDONT(ALine[I]);
                             FTelnetState := tnsData;
                        end;
               tnsWONT: begin
                             if (ALine[I] = TNO_BINARY) or (ALine[I] = TNO_ECHO) then
                                SendDO(ALine[I])
                             else
                                 SendDONT(ALine[I]);
                             FTelnetState := tnsData;
                        end;
               tnsDO: begin
                           if (ALine[I] = TNO_BINARY) or (ALine[I] = TNO_ECHO) then
                              SendWILL(ALine[I])
                           else
                               SendWONT(ALine[I]);
                           FTelnetState := tnsData;
                      end;
               tnsDONT: begin
                             if (ALine[I] = TNO_BINARY) or (ALine[I] = TNO_ECHO) then
                                SendWILL(ALine[I])
                             else
                                 SendWONT(ALine[I]);
                             FTelnetState := tnsData;
                        end;
          else
              FTelnetState := tnsData;
          end;
     end;
     Result := Res;
end;

function TMannSock.Open(ASocket: TSocket): Boolean;
begin
     if (Connected) then
        Close(True);

     FBuffer := '';
     FSocket := ASocket;
     FillChar(FTelnetCommands, SizeOf(FTelnetCommands), 0);

     DoRX;
     Result := Connected;
end;

function TMannSock.ReadBuf: String;
begin
     DoRX;
     Result := FBuffer;
     FBuffer := '';
end;

function TMannSock.ReadKey: Char;
begin
     while (Connected) and Not(KeyPressed) do
           Sleep(1);

     if (KeyPressed) then
     begin
          Result := FBuffer[1];
          Delete(FBuffer, 1, 1);
     end else
         Result := #0;
end;

procedure TMannSock.SendDO(ACh: Char);
begin
     if (FTelnetCommands[Ord(ACh)] = False) then
        Write(TNC_IAC + TNC_DO + ACh);
     FTelnetCommands[Ord(ACh)] := True;
end;

procedure TMannSock.SendDONT(ACh: Char);
begin
     if (FTelnetCommands[Ord(ACh)] = False) then
        Write(TNC_IAC + TNC_DONT + ACh);
     FTelnetCommands[Ord(ACh)] := True;
end;

procedure TMannSock.SendWILL(ACh: Char);
begin
     if (FTelnetCommands[Ord(ACh)] = False) then
        Write(TNC_IAC + TNC_WILL + ACh);
     FTelnetCommands[Ord(ACh)] := True;
end;

procedure TMannSock.SendWONT(ACh: Char);
begin
     if (FTelnetCommands[Ord(ACh)] = False) then
        Write(TNC_IAC + TNC_WONT + ACh);
     FTelnetCommands[Ord(ACh)] := True;
end;

procedure TMannSock.SetBlocking(ABlocking: Boolean);
var
   I: Integer;
begin
     I := Integer(Not(ABlocking));
     IOCtlSocket(FSocket, FIONBIO, @I);
end;

procedure TMannSock.Write(ALine: String);
begin
     Send(FSocket, ALine[1], Length(ALine), 0);
end;

procedure TMannSock.WriteLn(ALine: String);
begin
     Write(ALine + #13#10);
end;

end.
