/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "lcmd/LCmdMakeDir.h"
#include "lcmd/LCmdFilePanel.h"
#include "lcmd/LCmdDlgCreateDirectory.h"
#include "lcmd/LCmdDirCache.h"
#include "lcmd/LCmd.h"

#include "glib/GProgram.h"
#include "glib/vfs/GFile.h"
#include "glib/sys/GSystem.h"
#include "glib/primitives/GVArgs.h"

LCmdMakeDir::LCmdMakeDir ()
            :makeAsZip(false) 
{
}

LCmdMakeDir::~LCmdMakeDir ()
{
}

bool LCmdMakeDir::doMakeDir ( GVfs& vfs, const GString& dir, bool asZip )
{
   GError err;

   if (asZip)
   {
      GVfs::FileHandle hfile = vfs.openFile(dir, &err,
                                            GVfs::Mode_ReadWrite,
                                            GVfs::Create_IfNew);
      if (err == GError::Ok)
      {
         char data[] = {
            0x50, 0x4B, 0x05, 0x06, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
         };
         vfs.writeToFile(hfile, data, sizeof(data));
         vfs.closeFile(hfile);

         // Add the new "directory" to the Dynamic Directory Cache.
         if (lcmd->options.dirCache.addDirsDuringWalk)
         {
            GString walkedDir;
            vfs.getWalkedPath(dir, walkedDir);
            GString fullPath = vfs.getFullVirtualPathTo(walkedDir);
            lcmd->dirCache->addDirectoryToCache(fullPath, true);
         }
      }
   }

   else
   {
      GString fullDir(".");
      for (int i=0; err==GError::Ok; i++)
      {
         GString curDir(dir.getIndexedPart(i, "\\/"));
         if (curDir == "")
            break;
         fullDir += GFile::SlashChar;
         fullDir += curDir;
         err = vfs.createDirectory(fullDir);
         if (err == GError::Ok)
         {
            // Add the new "directory" to the Dynamic Directory Cache.
            if (lcmd->options.dirCache.addDirsDuringWalk)
            {
               GString walkedDir;
               vfs.getWalkedPath(fullDir, walkedDir);
               GString fullPath = vfs.getFullVirtualPathTo(walkedDir);
               lcmd->dirCache->addDirectoryToCache(fullPath, true);
            }
         }
      }
   }

   if (err != GError::Ok)
   {
      GStringl msg("%Txt_Err_CreateDirX", GVArgs(dir).add(err.getErrorMessage()));
      GProgram& prg = GProgram::GetProgram();
      GWindow& mwin = prg.getMainWindow();
      mwin.showMessageBox(msg, GMessageBox::TYPE_ERROR);
      return false;
   }

   return true;
}

void LCmdMakeDir::cmdMakeDir ( LCmdFilePanel& panel )
{
   GProgram& prg = GProgram::GetProgram();
   GWindow& mwin = prg.getMainWindow();

   // Open the dialogbox of where user must specify a name of the new directory.
   LCmdDlgCreateDirectory dlg(*this);
   if (!dlg.execute(mwin))
      return;

   GVfs& vf = panel.vfs.peek();
   if (panel.vfs.isRootOnly())
   {
      // Set current disk and directory.
      GString dir = vf.getCurrentDirectory(false); // Remember which directory was active.
      if (!panel.activatePanelDriveAndDir(true))
         return;

      // activatePanelDriveAndDir() may have changed the current directory
      // even if it returns OK. In that case, do nothing but fill panel with
      // filenames from that directory set (don't create requested directory).
      if (!vf.getCurrentDirectory(false).equalsIgnoreCase(dir))
      {
         // Fill in the filenames of the current dir in selected drive.
         panel.reRead();
         return;
      }
   }

   // Create the directory.
   if (doMakeDir(vf, dirStr, makeAsZip))
   {
      // Reread filenames into panel, including the newly created directory
      if (!panel.reRead())
         return;

      // Select the newly created directory item
      panel.selectItem(dirStr);
   }
}

