/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "lcmd/LCmd.h"
#include "lcmd/LCmdDirCache.h"
#include "lcmd/LCmdDlgFileAttributes.h"
#include "lcmd/LCmdDlgFileTime.h"
#include "lcmd/LCmdDlgFNameCase.h"
#include "lcmd/LCmdDlgCalcDirSize.h"
#include "lcmd/LCmdDlgEAEditor.h"
#include "lcmd/LCmdDelete.h"
#include "lcmd/LCmdExitWorker.h"
#include "lcmd/LCmdDlgStartupTip.h"
#include "lcmd/LCmdDynaFileSearch.h"
#include "lcmd/LCmdProcess.h"
#include "lcmd/LCmdVfsFilePreparator.h"
#include "lcmd/LCmdSubClient.h"
#include "lcmd/LCmdDlgDebugOptions.h"

#include "glib/gui/GMenu.h"
#include "glib/gui/GKeyBar.h"
#include "glib/gui/GStatusbar.h"
#include "glib/gui/GStaticText.h"
#include "glib/gui/GPushButton.h"
#include "glib/gui/GUserStringDlg.h"
#include "glib/gui/event/GKeyMessage.h"
#include "glib/gui/event/GUserMessage.h"
#include "glib/gui/layout/GBorderLayout.h"
#include "glib/sys/GSystem.h"
#include "glib/util/GToken.h"
#include "glib/util/GStringUtil.h"
#include "glib/exceptions/GIllegalStateException.h"

LCmdMainWindow::LCmdMainWindow ( LCmd& lcmd )
               :GFrameWindow("AppWin"),
                lcmd(lcmd),
                subClient(null),
                strCountChildProcesses("%StrStatusbar_ChildProcessCount"),
                strConLineCurrent("%StrStatusbar_ConsoleLineCurrent"),
                strConLineCount("%StrStatusbar_ConsoleLineCount"),
                strPhysMemFree("%StrStatusbar_PhysMemFree"),
                strSysCountProcesses("%StrStatusbar_ProcessCount"),
                strSysCountThreads("%StrStatusbar_ThreadCount"),
                strDriveFree("%StrStatusbar_DriveFree"),
                strDirCacheCount("%StrStatusbar_DirCacheCount")
{
   setOS2Y(true);
   winStyle2 |= WS2_IGNORE_COLORS_AND_FONT_PROFILE;
   subClient = new LCmdSubClient(*this);

   // ---
   bool fancyMenbar = lcmd.isUseFancyMenubars();
   setAccelTable("MainAccel");
   setMenubar("MainMenu", fancyMenbar, false);
   setToolbar("MainToolbar", "ToolBar");
   setKeybarVisible(true);
   setStatusbarVisible(true);
   setAutoLayoutClientWin(subClient);
   setAutoManageWindowPos(false);

   // ---
   GStatusbar* sbar = getStatusbar();
   sbar->setTooltipPosition(GTooltip::PosAbove);
   sbar->getTextWindow().setTooltipText("%TxtStatbarTooltipTextWindow");
   sbar->addTextCell("CountChildProcesses").setTooltipText("%TxtStatbarTooltipCountChildProcesses");
   sbar->addTextCell("ConLineCurrent").setTooltipText("%TxtStatbarTooltipConLineCurrent");
   sbar->addTextCell("ConLineCount").setTooltipText("%TxtStatbarTooltipConLineCount");
   sbar->addTextCell("PhysMemFree").setTooltipText("%TxtStatbarTooltipPhysMemFree");
   sbar->addTextCell("SysCountProcesses").setTooltipText("%TxtStatbarTooltipSysCountProcesses");
   sbar->addTextCell("SysCountThreads").setTooltipText("%TxtStatbarTooltipSysCountThreads");
   sbar->addTextCell("DriveFree").setTooltipText("%TxtStatbarTooltipDriveFree");
   sbar->addTextCell("DirCacheCount").setTooltipText("%TxtStatbarTooltipDirCacheCount");

   // Create and activate the popup menu of which to display when the
   // user selects the <i>cmdUserMenu</i> command (F2).
   bool fancyPopMen = lcmd.isUseFancyPopupMenues();
   setPopupMenu("UserMenu", fancyPopMen);

   // Start the 1 Hz timer used to update the statusbar and info panels.
   startTimer("INFOUPDATE", 1000);
}

LCmdMainWindow::~LCmdMainWindow ()
{
   stopTimer("INFOUPDATE");
   delete subClient;
}

LCmdMainWindow& LCmdMainWindow::GetMainWindow ()
{
   LCmd* lc = ::lcmd;
   return lc->mainWin;
}

void LCmdMainWindow::queryProfile ( const GString& sectName )
{
   GSectionBag& ini = lcmd.getIniProperties();
   LCmdOptions& opt = lcmd.options;

   // Set the style to be used as the common default for all
   // {@link GStaticText} that doesn't use a specific style other
   // than {@link GWindow#LeaderStyle_DEFAULT}.
   GWindow::SetCommonLeaderStyle(GWindow::LeaderStyle_FANCY1);

   // Save Settings On Exit
   GString sn = sectName + ".SaveSettingsOnExit";
   opt.saveOnExit.everything = ini.getBool(sn, "All", opt.saveOnExit.everything);
   opt.saveOnExit.currentDirs = ini.getBool(sn, "CurDir", opt.saveOnExit.currentDirs);
   opt.saveOnExit.storedDirs = ini.getBool(sn, "StoredDirs", opt.saveOnExit.storedDirs);
   opt.saveOnExit.dirHist = ini.getBool(sn, "DirHistory", opt.saveOnExit.dirHist);
   opt.saveOnExit.cmdHist = ini.getBool(sn, "CmdHistory", opt.saveOnExit.cmdHist);
   opt.saveOnExit.dirCache = ini.getBool(sn, "DirCache", opt.saveOnExit.dirCache);
   opt.saveOnExit.curSelFiles = ini.getBool(sn, "CurSelectedFileItem", opt.saveOnExit.curSelFiles);
   opt.saveOnExit.visibleState = ini.getBool(sn, "VisibilityStates", opt.saveOnExit.visibleState);
   opt.saveOnExit.panelModeSort = ini.getBool(sn, "PanelModesAndSortOptions", opt.saveOnExit.panelModeSort);
   opt.saveOnExit.frameWinPos = ini.getBool(sn, "MainWinPos", opt.saveOnExit.frameWinPos);
   opt.saveOnExit.colors = ini.getBool(sn, "Colors", opt.saveOnExit.colors);
   opt.saveOnExit.fonts = ini.getBool(sn, "Fonts", opt.saveOnExit.fonts);
   opt.saveOnExit.otherOptions = ini.getBool(sn, "AllOther", opt.saveOnExit.otherOptions);

   // Global
   sn = sectName + ".Global";
   opt.fancyMenuBars = ini.getBool(sn, "FancyMenuBars", opt.fancyMenuBars);
   opt.fancyPopupMenues = ini.getBool(sn, "FancyPopupMenues", opt.fancyPopupMenues);
   lcmd.mainWin.getMenubar()->setUseFancyMenues(opt.fancyMenuBars);
   lcmd.mainWin.getPopupMenu()->setUseFancyMenues(opt.fancyPopupMenues); // Update the user menu (F2)
   lcmd.fp1->getPopupMenu()->setUseFancyMenues(opt.fancyPopupMenues);
   lcmd.fp2->getPopupMenu()->setUseFancyMenues(opt.fancyPopupMenues);
   subClient->cmdcont.cmdline.entry.getPopupMenu()->setUseFancyMenues(opt.fancyPopupMenues);
   subClient->cmdcont.conmon.getPopupMenu()->setUseFancyMenues(opt.fancyPopupMenues);

   // Programs
   queryProgProfile("Programs.Editor", opt.editor);
   queryProgProfile("Programs.Viewer", opt.viewer);
   queryProgProfile("Programs.FFind", opt.ffind);
   queryProgProfile("Programs.FDiff", opt.fdiff);
   queryProgProfile("Programs.CClean", opt.cclean);
   queryProgProfile("Programs.Undel", opt.undel);
   queryProgProfile("Programs.Eas", opt.eas);

   // Confirmations
   sn = sectName + ".Confirm";
   opt.confirm.prgExit = ini.getBool(sn, "Exit", opt.confirm.prgExit);
   opt.confirm.delFile = ini.getBool(sn, "DelFile", opt.confirm.delFile);
   opt.confirm.delFileInSubDirs = ini.getBool(sn, "DelFileInSubDirs", opt.confirm.delFileInSubDirs);
   opt.confirm.delEmptyDir = ini.getBool(sn, "DelDir", opt.confirm.delEmptyDir);
   opt.confirm.delDirContainingFiles = ini.getBool(sn, "DelNonEmptyDir", opt.confirm.delDirContainingFiles);
   opt.confirm.delHidSysReadFile = ini.getBool(sn, "DelHidSysReadOnly", opt.confirm.delHidSysReadFile);
   opt.confirm.overwriteFileOrDir = ini.getBool(sn, "Overwrite", opt.confirm.overwriteFileOrDir);

   // File copy/move
   sn = sectName + ".FileCopyAndMove";
   opt.fileCopy.showStatistics = ini.getBool(sn, "ShowStatistics", opt.fileCopy.showStatistics);
   opt.fileCopy.waitOnFinish = ini.getBool(sn, "WaitOnFinish", opt.fileCopy.waitOnFinish);
   opt.fileCopy.warnIfNotEnoughTargetSpace = ini.getBool(sn, "WarnIfNotEnoughTargetSpace", opt.fileCopy.warnIfNotEnoughTargetSpace);
   opt.fileCopy.setPreallocSpaceFromInt(ini.getInt(sn, "PreallocateSpace", opt.fileCopy.preallocSpace));
   opt.fileCopy.preallocSpaceOnDrives = ini.getString(sn, "PreallocateSpaceonDrives", opt.fileCopy.preallocSpaceOnDrives);
   opt.fileCopy.maxPreallocateSize = ini.getLong(sn, "MaxPreallocateSize", opt.fileCopy.maxPreallocateSize);
   if (opt.fileCopy.maxPreallocateSize < 0)
      opt.fileCopy.maxPreallocateSize = 0;
   opt.fileCopy.maxBuffSize = ini.getInt(sn, "MaxBufferSize", opt.fileCopy.maxBuffSize);
   if (opt.fileCopy.maxBuffSize < 1)
      opt.fileCopy.maxBuffSize = 1;
   else
   if (opt.fileCopy.maxBuffSize > 65536) // 64 MBytes.
      opt.fileCopy.maxBuffSize = 65536;

   // FileDelete
   sn = sectName + ".FileDelete";
   opt.fileDel.defaultDeleteToTrashCan = ini.getBool(sn, "DeleteToTrashCanByDefault", opt.fileDel.defaultDeleteToTrashCan);
   opt.fileDel.useDelKey = ini.getBool(sn, "UseDeleteKey", opt.fileDel.useDelKey);
   opt.fileDel.showDirsProgressOnly = ini.getBool(sn, "ShowDirectoryProgressOnly", opt.fileDel.showDirsProgressOnly);

   // DirCache
   sn = sectName + ".DirCache";
   opt.dirCache.addDirsDuringWalk = ini.getBool(sn, "AddDirsDuringWalk", opt.dirCache.addDirsDuringWalk);
   opt.dirCache.autoStripDirsAfterDirDeletions = ini.getBool(sn, "AutoStripDirsAfterDirDeletions", opt.dirCache.autoStripDirsAfterDirDeletions);
   opt.dirCache.includeVfsDirs = ini.getBool(sn, "IncludeVfsDirs", opt.dirCache.includeVfsDirs);

   // VariousConsoleOptions
   sn = sectName + ".VariousConsoleOptions";
   opt.conVarious.defaultCloseConOnExit = ini.getBool(sn, "CloseConsoleOnExitIsDefault", opt.conVarious.defaultCloseConOnExit);
   opt.conVarious.startConFullScreen = ini.getBool(sn, "StartConsoleInFullScreen", opt.conVarious.startConFullScreen);
   opt.conVarious.startDosFullScreen = ini.getBool(sn, "StartDosInFullScreen", opt.conVarious.startDosFullScreen);

   // VfsZip
   sn = sectName + ".VfsZip";
   opt.vfsZip.sysLaunchZipFiles = ini.getBool(sn, "SystemLaunch", opt.vfsZip.sysLaunchZipFiles);
   opt.vfsZip.toolPath = ini.getString(sn, "ToolPath", opt.vfsZip.toolPath);
   opt.vfsZip.argsDelete = ini.getString(sn, "ArgsDelete", opt.vfsZip.argsDelete);

   // VariousOtherOptions
   sn = sectName + ".VariousOtherOptions";
   opt.various.addToItemHeight = ini.getInt(sn, "AdditionalItemHeight", opt.various.addToItemHeight);
   opt.various.addToItemWidth = ini.getInt(sn, "AdditionalItemWidth", opt.various.addToItemWidth);
   opt.various.delayShowTooltipMillis = ini.getInt(sn, "DelayShowTooltipMillis", opt.various.delayShowTooltipMillis);
   opt.various.autoHideTooltipMillis = ini.getInt(sn, "AutoHideTooltipMillis", opt.various.autoHideTooltipMillis);
   GTooltip::SetDefaultDelayShowMillis(opt.various.delayShowTooltipMillis);
   GTooltip::SetDefaultAutoHideMillis(opt.various.autoHideTooltipMillis);

   // Common File Filter
   sn = sectName + ".CommonFileFilter";
   opt.fileFilter.inclFiles = ini.getBool(sn, "Operate on files", opt.fileFilter.inclFiles);
   opt.fileFilter.inclDirs = ini.getBool(sn, "Operate on directories", opt.fileFilter.inclDirs);
   opt.fileFilter.flagArchive = ini.getInt(sn, "Archive", 0, 2, opt.fileFilter.flagArchive);
   opt.fileFilter.flagReadOnly = ini.getInt(sn, "Read Only", 0, 2, opt.fileFilter.flagReadOnly);
   opt.fileFilter.flagHidden = ini.getInt(sn, "Hidden", 0, 2, opt.fileFilter.flagHidden);
   opt.fileFilter.flagSystem = ini.getInt(sn, "System", 0, 2, opt.fileFilter.flagSystem);
   opt.fileFilter.useInclFilterStr = ini.getBool(sn, "UseIncludeFilterString", opt.fileFilter.useInclFilterStr);
   opt.fileFilter.useExclFilterStr = ini.getBool(sn, "UseExcludeFilterString", opt.fileFilter.useExclFilterStr);
   opt.fileFilter.inclFilterStr = ini.getString(sn, "Include Files", opt.fileFilter.inclFilterStr);
   opt.fileFilter.exclFilterStr = ini.getString(sn, "Exclude Files", opt.fileFilter.exclFilterStr);

   // Compare Directories
   sn = sectName + ".CompareDirectories";
   opt.cmpDirs.timeStampOpt = (LCmdDlgCompareDirs::TAGTIMESTAMPOPT) ini.getInt(sn, "Time Stamp Compare", LCmdDlgCompareDirs::TAGTMST_IGNORE, LCmdDlgCompareDirs::TAGTMST_DIFF, opt.cmpDirs.timeStampOpt);
   opt.cmpDirs.sizeOpt = (LCmdDlgCompareDirs::TAGSIZEOPT) ini.getInt(sn, "File Size Compare", LCmdDlgCompareDirs::TAGSIZE_IGNORE, LCmdDlgCompareDirs::TAGSIZE_DIFF, opt.cmpDirs.sizeOpt);
   opt.cmpDirs.tagUnmatched = ini.getBool(sn, "Tag Unmatched Files", opt.cmpDirs.tagUnmatched);
   opt.cmpDirs.compareContents = ini.getBool(sn, "Compare File Content", opt.cmpDirs.compareContents);

   // Stored Directories
   sn = sectName + ".StoredDirectories";
   opt.quickDirList.loadFromProfile(ini, sn);

   // ---
   // Recurse through all of our child windows.
   GFrameWindow::queryProfile(sectName);

   // ---
   if (!subClient->cmdcont.cmdline.isVisible())
      opt.panelsFullScreen = true;
}

void LCmdMainWindow::writeProfile ( const GString& sectName, bool force )
{
   GSectionBag& ini = lcmd.getIniProperties();
   LCmdOptions& opt = lcmd.options;
   if (opt.saveOnExit.everything)
      force = true;

   // Recurse through all of our child windows.
   GFrameWindow::writeProfile(sectName, force);

   // Store the frame window position and minimize/maximize state.
   if (force || opt.saveOnExit.frameWinPos)
      storeWindowPos(sectName, "Pos");

   // Global
   GString sn = sectName + ".Global";
   ini.putBool(sn, "FancyMenuBars", opt.fancyMenuBars, force || opt.saveOnExit.visibleState);
   ini.putBool(sn, "FancyPopupMenues", opt.fancyPopupMenues, force || opt.saveOnExit.visibleState);

   // Programs
   writeProgProfile("Programs.Editor", opt.editor, force);
   writeProgProfile("Programs.Viewer", opt.viewer, force);
   writeProgProfile("Programs.FFind", opt.ffind, force);
   writeProgProfile("Programs.FDiff", opt.fdiff, force);
   writeProgProfile("Programs.CClean", opt.cclean, force);
   writeProgProfile("Programs.Undel", opt.undel, force);
   writeProgProfile("Programs.EAs", opt.eas, force);

   // Confirmations
   sn = sectName + ".Confirm";
   ini.putBool(sn, "Exit", opt.confirm.prgExit, force || opt.saveOnExit.otherOptions);
   ini.putBool(sn, "DelFile", opt.confirm.delFile, force || opt.saveOnExit.otherOptions);
   ini.putBool(sn, "DelFileInSubDirs", opt.confirm.delFileInSubDirs, force || opt.saveOnExit.otherOptions);
   ini.putBool(sn, "DelDir", opt.confirm.delEmptyDir, force || opt.saveOnExit.otherOptions);
   ini.putBool(sn, "DelNonEmptyDir", opt.confirm.delDirContainingFiles, force || opt.saveOnExit.otherOptions);
   ini.putBool(sn, "DelHidSysReadOnly", opt.confirm.delHidSysReadFile, force || opt.saveOnExit.otherOptions);
   ini.putBool(sn, "Overwrite", opt.confirm.overwriteFileOrDir, force || opt.saveOnExit.otherOptions);

   // File copy/move
   sn = sectName + ".FileCopyAndMove";
   ini.putBool(sn, "ShowStatistics", opt.fileCopy.showStatistics, force || opt.saveOnExit.otherOptions);
   ini.putBool(sn, "WaitOnFinish", opt.fileCopy.waitOnFinish, force || opt.saveOnExit.otherOptions);
   ini.putBool(sn, "WarnIfNotEnoughTargetSpace", opt.fileCopy.warnIfNotEnoughTargetSpace, force || opt.saveOnExit.otherOptions);
   ini.putInt(sn, "PreallocateSpace", opt.fileCopy.preallocSpace, force || opt.saveOnExit.otherOptions);
   ini.putString(sn, "PreallocateSpaceonDrives", opt.fileCopy.preallocSpaceOnDrives, force || opt.saveOnExit.otherOptions);
   ini.putLong(sn, "MaxPreallocateSize", opt.fileCopy.maxPreallocateSize, force || opt.saveOnExit.otherOptions);
   ini.putInt(sn, "MaxBufferSize", opt.fileCopy.maxBuffSize, force || opt.saveOnExit.otherOptions);

   // FileDelete
   sn = sectName + ".FileDelete";
   ini.putBool(sn, "DeleteToTrashCanByDefault", opt.fileDel.defaultDeleteToTrashCan, force || opt.saveOnExit.otherOptions);
   ini.putBool(sn, "UseDeleteKey", opt.fileDel.useDelKey, force || opt.saveOnExit.otherOptions);
   ini.putBool(sn, "ShowDirectoryProgressOnly", opt.fileDel.showDirsProgressOnly, force || opt.saveOnExit.otherOptions);

   // DirCache
   sn = sectName + ".DirCache";
   ini.putBool(sn, "AddDirsDuringWalk", opt.dirCache.addDirsDuringWalk, force || opt.saveOnExit.otherOptions);
   ini.putBool(sn, "AutoStripDirsAfterDirDeletions", opt.dirCache.autoStripDirsAfterDirDeletions, force || opt.saveOnExit.otherOptions);
   ini.putBool(sn, "IncludeVfsDirs", opt.dirCache.includeVfsDirs, force || opt.saveOnExit.otherOptions);

   // VariousConsoleOptions
   sn = sectName + ".VariousConsoleOptions";
   ini.putBool(sn, "CloseConsoleOnExitIsDefault", opt.conVarious.defaultCloseConOnExit, force || opt.saveOnExit.otherOptions);
   ini.putBool(sn, "StartConsoleInFullScreen", opt.conVarious.startConFullScreen, force || opt.saveOnExit.otherOptions);
   ini.putBool(sn, "StartDosInFullScreen", opt.conVarious.startDosFullScreen, force || opt.saveOnExit.otherOptions);

   // VfsZip
   sn = sectName + ".VfsZip";
   ini.putBool(sn, "SystemLaunch", opt.vfsZip.sysLaunchZipFiles, force || opt.saveOnExit.otherOptions);
   ini.putString(sn, "ToolPath", opt.vfsZip.toolPath, force || opt.saveOnExit.otherOptions);
   ini.putString(sn, "ArgsDelete", opt.vfsZip.argsDelete, force || opt.saveOnExit.otherOptions);

   // VariousOtherOptions
   sn = sectName + ".VariousOtherOptions";
   ini.putInt(sn, "AdditionalItemHeight", opt.various.addToItemHeight, force || opt.saveOnExit.otherOptions);
   ini.putInt(sn, "AdditionalItemWidth", opt.various.addToItemWidth, force || opt.saveOnExit.otherOptions);
   ini.putInt(sn, "DelayShowTooltipMillis", opt.various.delayShowTooltipMillis, force || opt.saveOnExit.otherOptions);
   ini.putInt(sn, "AutoHideTooltipMillis", opt.various.autoHideTooltipMillis, force || opt.saveOnExit.otherOptions);

   // Common File Filter
   sn = sectName + ".CommonFileFilter";
   ini.putBool(sn, "Operate on files", opt.fileFilter.inclFiles, force || opt.saveOnExit.otherOptions);
   ini.putBool(sn, "Operate on directories", opt.fileFilter.inclDirs, force || opt.saveOnExit.otherOptions);
   ini.putInt(sn, "Archive", opt.fileFilter.flagArchive, force || opt.saveOnExit.otherOptions);
   ini.putInt(sn, "Read Only", opt.fileFilter.flagReadOnly, force || opt.saveOnExit.otherOptions);
   ini.putInt(sn, "Hidden", opt.fileFilter.flagHidden, force || opt.saveOnExit.otherOptions);
   ini.putInt(sn, "System", opt.fileFilter.flagSystem, force || opt.saveOnExit.otherOptions);
   ini.putBool(sn, "UseIncludeFilterString", opt.fileFilter.useInclFilterStr, force || opt.saveOnExit.otherOptions);
   ini.putBool(sn, "UseExcludeFilterString", opt.fileFilter.useExclFilterStr, force || opt.saveOnExit.otherOptions);
   ini.putString(sn, "Include Files", opt.fileFilter.inclFilterStr, force || opt.saveOnExit.otherOptions);
   ini.putString(sn, "Exclude Files", opt.fileFilter.exclFilterStr, force || opt.saveOnExit.otherOptions);

   // Compare Directories
   sn = sectName + ".CompareDirectories";
   ini.putInt(sn, "Time Stamp Compare", opt.cmpDirs.timeStampOpt, force || opt.saveOnExit.otherOptions);
   ini.putInt(sn, "File Size Compare", opt.cmpDirs.sizeOpt, force || opt.saveOnExit.otherOptions);
   ini.putBool(sn, "Tag Unmatched Files", opt.cmpDirs.tagUnmatched, force || opt.saveOnExit.otherOptions);
   ini.putBool(sn, "Compare File Content", opt.cmpDirs.compareContents, force || opt.saveOnExit.otherOptions);

   // Stored Directories
   sn = sectName + ".StoredDirectories";
   if (force || opt.saveOnExit.storedDirs)
      opt.quickDirList.writeToProfile(ini, sn);

   // Directory Caches.
   if (force || opt.saveOnExit.dirCache)
      lcmd.dirCache->writeToProfile();
}

void LCmdMainWindow::writeProgProfile ( const GString& sectName, LCmdOptions::ProgOption& prog, bool force )
{
   GSectionBag& ini = lcmd.getIniProperties();
   LCmdOptions& opt = lcmd.options;
   ini.putBool(sectName, "UseExternal", prog.useExternal, force || opt.saveOnExit.otherOptions);
   ini.putString(sectName, "ProgPath", prog.progName, force || opt.saveOnExit.otherOptions);
   ini.putString(sectName, "ProgParams", prog.params, force || opt.saveOnExit.otherOptions);
   ini.putString(sectName, "WorkingDir", prog.workDir, force || opt.saveOnExit.otherOptions);
}

void LCmdMainWindow::queryProgProfile ( const GString& sectName, LCmdOptions::ProgOption& prog )
{
   GSectionBag& ini = lcmd.getIniProperties();
   prog.useExternal = ini.getBool(sectName, "UseExternal", prog.useExternal);
   prog.progName = ini.getString(sectName, "ProgPath", prog.progName);
   prog.params = ini.getString(sectName, "ProgParams", prog.params);
   prog.workDir = ini.getString(sectName, "WorkingDir", prog.workDir);
}

DEFINE_COMMAND_TABLE(LCmdMainWindow);
   ADD_COMMAND("cmdActivateOppositePanel",      cmdActivateOppositePanel);
   ADD_COMMAND("cmdActivateStoredDir0",         cmdActivateStoredDir0);
   ADD_COMMAND("cmdActivateStoredDir1",         cmdActivateStoredDir1);
   ADD_COMMAND("cmdActivateStoredDir2",         cmdActivateStoredDir2);
   ADD_COMMAND("cmdActivateStoredDir3",         cmdActivateStoredDir3);
   ADD_COMMAND("cmdActivateStoredDir4",         cmdActivateStoredDir4);
   ADD_COMMAND("cmdActivateStoredDir5",         cmdActivateStoredDir5);
   ADD_COMMAND("cmdActivateStoredDir6",         cmdActivateStoredDir6);
   ADD_COMMAND("cmdActivateStoredDir7",         cmdActivateStoredDir7);
   ADD_COMMAND("cmdActivateStoredDir8",         cmdActivateStoredDir8);
   ADD_COMMAND("cmdActivateStoredDir9",         cmdActivateStoredDir9);
   ADD_COMMAND("cmdAlignPanelsHeight",          cmdAlignPanelsHeight);
   ADD_COMMAND("cmdAlignPanelsWidth",           cmdAlignPanelsWidth);
   ADD_COMMAND("cmdAppendEndOfStream",          cmdAppendEndOfStream);
   ADD_COMMAND("cmdExtendLeftPanelWidth",       cmdExtendLeftPanelWidth);
   ADD_COMMAND("cmdExtendRightPanelWidth",      cmdExtendRightPanelWidth);
   ADD_COMMAND("cmdExtendPanelsHeight",         cmdExtendPanelsHeight);
   ADD_COMMAND("cmdExtendConMonHeight",         cmdExtendConMonHeight);
   ADD_COMMAND("cmdAutoFilenameBackward",       cmdAutoFilenameBackward);
   ADD_COMMAND("cmdAutoFilenameForeward",       cmdAutoFilenameForeward);
   ADD_COMMAND("cmdBreak",                      cmdBreak);
   ADD_COMMAND("cmdEditEAs",                    cmdEditEAs);
   ADD_COMMAND("cmdCalcDirSize",                cmdCalcDirSize);
   ADD_COMMAND("cmdCalcRootDirSize",            cmdCalcRootDirSize);
   ADD_COMMAND("cmdChooseFileAttrib",           cmdChooseFileAttrib);
   ADD_COMMAND("cmdChooseFileNameCase",         cmdChooseFileNameCase);
   ADD_COMMAND("cmdChooseFileTime",             cmdChooseFileTime);
   ADD_COMMAND("cmdCloneProgramInstance",       cmdCloneProgramInstance);
   ADD_COMMAND("cmdCloseAndExit",               cmdCloseAndExit);
   ADD_COMMAND("cmdCommandHistory",             cmdCommandHistory);
   ADD_COMMAND("cmdCompareDirs",                cmdCompareDirs);
   ADD_COMMAND("cmdCompareFiles",               cmdCompareFiles);
   ADD_COMMAND("cmdCopy",                       cmdCopy);
   ADD_COMMAND("cmdCopyCurFileName",            cmdCopyCurFileName);
   ADD_COMMAND("cmdCopyCurFilePath",            cmdCopyCurFilePath);
   ADD_COMMAND("cmdCopyFile",                   cmdCopyFile);
   ADD_COMMAND("cmdCopyFileToSameDir",          cmdCopyFileToSameDir);
   ADD_COMMAND("cmdCreateSysObject",            cmdCreateSysObject);
   ADD_COMMAND("cmdCurPanelCycleIconTypes",     cmdCurPanelCycleIconTypes);
   ADD_COMMAND("cmdCurPanelGotoRecentDir",      cmdCurPanelGotoRecentDir);
   ADD_COMMAND("cmdCurPanelModeBrief",          cmdCurPanelModeBrief);
   ADD_COMMAND("cmdCurPanelModeFull",           cmdCurPanelModeFull);
   ADD_COMMAND("cmdCurPanelModeTree",           cmdCurPanelModeTree);
   ADD_COMMAND("cmdCurPanelModeWide",           cmdCurPanelModeWide);
   ADD_COMMAND("cmdCurPanelOptionsSort",        cmdCurPanelOptionsSort);
   ADD_COMMAND("cmdCurPanelReread",             cmdCurPanelReread);
   ADD_COMMAND("cmdCurPanelSortByDate",         cmdCurPanelSortByDate);
   ADD_COMMAND("cmdCurPanelSortByExtension",    cmdCurPanelSortByExtension);
   ADD_COMMAND("cmdCurPanelSortByName",         cmdCurPanelSortByName);
   ADD_COMMAND("cmdCurPanelSortBySize",         cmdCurPanelSortBySize);
   ADD_COMMAND("cmdCurPanelSortByTime",         cmdCurPanelSortByTime);
   ADD_COMMAND("cmdCurPanelSortReverse",        cmdCurPanelSortReverse);
   ADD_COMMAND("cmdCurPanelSortUnsorted",       cmdCurPanelSortUnsorted);
   ADD_COMMAND("cmdDebugOptions",               cmdDebugOptions);
   ADD_COMMAND("cmdDebugShowLogFile",           cmdDebugShowLogFile);
   ADD_COMMAND("cmdDeleteFile",                 cmdDeleteFile);
   ADD_COMMAND("cmdDirHistory",                 cmdDirHistory);
   ADD_COMMAND("cmdEditFile",                   cmdEditFile);
   ADD_COMMAND("cmdEditTypedFileName",          cmdEditTypedFileName);
   ADD_COMMAND("cmdEscape",                     cmdEscape);
   ADD_COMMAND("cmdFeedChildProcessStdIn",      cmdFeedChildProcessStdIn);
   ADD_COMMAND("cmdFindDuplicateFiles",         cmdFindDuplicateFiles);
   ADD_COMMAND("cmdFindFile",                   cmdFindFile);
   ADD_COMMAND("cmdGotoLineNum",                cmdGotoLineNum);
   ADD_COMMAND("cmdHelpAbout",                  cmdHelpAbout);
   ADD_COMMAND("cmdHelpContents",               cmdHelpContents);
   ADD_COMMAND("cmdHelpCmdLineArgs",            cmdHelpCmdLineArgs);
   ADD_COMMAND("cmdHelpIndex",                  cmdHelpIndex);
   ADD_COMMAND("cmdHelpKeyboard",               cmdHelpKeyboard);
   ADD_COMMAND("cmdHelpShowTip",                cmdHelpShowTip);
   ADD_COMMAND("cmdInsertItemNameInCmdLine",    cmdInsertItemNameInCmdLine);
   ADD_COMMAND("cmdInsertItemPathInCmdLine",    cmdInsertItemPathInCmdLine);
   ADD_COMMAND("cmdInvertSelection",            cmdInvertSelection);
   ADD_COMMAND("cmdLeftChooseDrive",            cmdLeftChooseDrive);
   ADD_COMMAND("cmdLeftChooseFileAttrib",       cmdLeftChooseFileAttrib);
   ADD_COMMAND("cmdLeftChooseFileTime",         cmdLeftChooseFileTime);
   ADD_COMMAND("cmdLeftCycleIconTypes",         cmdLeftCycleIconTypes);
   ADD_COMMAND("cmdLeftGotoRecentDir",          cmdLeftGotoRecentDir);
   ADD_COMMAND("cmdLeftModeBrief",              cmdLeftModeBrief);
   ADD_COMMAND("cmdLeftModeFull",               cmdLeftModeFull);
   ADD_COMMAND("cmdLeftModeInfo",               cmdLeftModeInfo);
   ADD_COMMAND("cmdLeftModeTree",               cmdLeftModeTree);
   ADD_COMMAND("cmdLeftModeWide",               cmdLeftModeWide);
   ADD_COMMAND("cmdLeftOptionsColor",           cmdLeftOptionsColor);
   ADD_COMMAND("cmdLeftOptionsFileFilter",      cmdLeftOptionsFileFilter);
   ADD_COMMAND("cmdLeftOptionsFont",            cmdLeftOptionsFont);
   ADD_COMMAND("cmdLeftOptionsSort",            cmdLeftOptionsSort);
   ADD_COMMAND("cmdLeftOptionsStartup",         cmdLeftOptionsStartup);
   ADD_COMMAND("cmdLeftOptionsView",            cmdLeftOptionsView);
   ADD_COMMAND("cmdLeftReread",                 cmdLeftReread);
   ADD_COMMAND("cmdLeftShowFileInfo",           cmdLeftShowFileInfo);
   ADD_COMMAND("cmdLeftSortByDate",             cmdLeftSortByDate);
   ADD_COMMAND("cmdLeftSortByExtension",        cmdLeftSortByExtension);
   ADD_COMMAND("cmdLeftSortByName",             cmdLeftSortByName);
   ADD_COMMAND("cmdLeftSortBySize",             cmdLeftSortBySize);
   ADD_COMMAND("cmdLeftSortByTime",             cmdLeftSortByTime);
   ADD_COMMAND("cmdLeftSortReverse",            cmdLeftSortReverse);
   ADD_COMMAND("cmdLeftSortUnsorted",           cmdLeftSortUnsorted);
   ADD_COMMAND("cmdLeftWalkDirectoryUp",        cmdLeftWalkDirectoryUp);
   ADD_COMMAND("cmdLeftWalkRootDirectory",      cmdLeftWalkRootDirectory);
   ADD_COMMAND("cmdMakeDir",                    cmdMakeDir);
   ADD_COMMAND("cmdMoveFile",                   cmdMoveFile);
   ADD_COMMAND("cmdNavigateBetwConAndCmdLine",  cmdNavigateBetwConAndCmdLine);
   ADD_COMMAND("cmdNavigateDown",               cmdNavigateDown);
   ADD_COMMAND("cmdNavigateEnd",                cmdNavigateEnd);
   ADD_COMMAND("cmdNavigateEnter",              cmdNavigateEnter);
   ADD_COMMAND("cmdNavigateEnterSysObj",        cmdNavigateEnterSysObj);
   ADD_COMMAND("cmdNavigateHome",               cmdNavigateHome);
   ADD_COMMAND("cmdNavigateLeft",               cmdNavigateLeft);
   ADD_COMMAND("cmdNavigateNextMatch",          cmdNavigateNextMatch);
   ADD_COMMAND("cmdNavigatePageDown",           cmdNavigatePageDown);
   ADD_COMMAND("cmdNavigatePageUp",             cmdNavigatePageUp);
   ADD_COMMAND("cmdNavigatePrevMatch",          cmdNavigatePrevMatch);
   ADD_COMMAND("cmdNavigateRight",              cmdNavigateRight);
   ADD_COMMAND("cmdNavigateUp",                 cmdNavigateUp);
   ADD_COMMAND("cmdOpenSysFolder",              cmdOpenSysFolder);
   ADD_COMMAND("cmdOppositePanelInfo",          cmdOppositePanelInfo);
   ADD_COMMAND("cmdOptionsAliases",             cmdOptionsAliases);
   ADD_COMMAND("cmdOptionsColor",               cmdOptionsColor);
   ADD_COMMAND("cmdOptionsConfirmations",       cmdOptionsConfirmations);
   ADD_COMMAND("cmdOptionsDialog",              cmdOptionsDialog);
   ADD_COMMAND("cmdOptionsEnvironmentVars",     cmdOptionsEnvironmentVars);
   ADD_COMMAND("cmdOptionsFont",                cmdOptionsFont);
   ADD_COMMAND("cmdOptionsInternalCommands",    cmdOptionsInternalCommands);
   ADD_COMMAND("cmdOptionsSSA",                 cmdOptionsSSA);
   ADD_COMMAND("cmdOptionsSaveOnExit",          cmdOptionsSaveOnExit);
   ADD_COMMAND("cmdOptionsFileCopyMv",          cmdOptionsFileCopyMv);
   ADD_COMMAND("cmdOptionsFileDel",             cmdOptionsFileDel);
   ADD_COMMAND("cmdOptionsDirCache",            cmdOptionsDirCache);
   ADD_COMMAND("cmdOptionsStyle",               cmdOptionsStyle);
   ADD_COMMAND("cmdOptionsVarious",             cmdOptionsVarious);
   ADD_COMMAND("cmdOptionsView",                cmdOptionsView);
   ADD_COMMAND("cmdOptionsConsoleVarious",      cmdOptionsConsoleVarious);
   ADD_COMMAND("cmdOptionsVfsVarious",          cmdOptionsVfsVarious);
   ADD_COMMAND("cmdOptionsVfsZip",              cmdOptionsVfsZip);
   ADD_COMMAND("cmdProcessFilesByCommand",      cmdProcessFilesByCommand);
   ADD_COMMAND("cmdProgOptionsEditor",          cmdProgOptionsEditor);
   ADD_COMMAND("cmdProgOptionsViewer",          cmdProgOptionsViewer);
   ADD_COMMAND("cmdProgOptionsFileDiff",        cmdProgOptionsFileDiff);
   ADD_COMMAND("cmdProgOptionsFileDupFinder",   cmdProgOptionsFileDupFinder);
   ADD_COMMAND("cmdProgOptionsFileFinder",      cmdProgOptionsFileFinder);
   ADD_COMMAND("cmdProgOptionsEABrowser",       cmdProgOptionsEABrowser);
   ADD_COMMAND("cmdProgOptionsUndeleter",       cmdProgOptionsUndeleter);
   ADD_COMMAND("cmdPullDownMenu",               cmdPullDownMenu);
   ADD_COMMAND("cmdRenameFile",                 cmdRenameFile);
   ADD_COMMAND("cmdRightChooseDrive",           cmdRightChooseDrive);
   ADD_COMMAND("cmdRightChooseFileAttrib",      cmdRightChooseFileAttrib);
   ADD_COMMAND("cmdRightChooseFileTime",        cmdRightChooseFileTime);
   ADD_COMMAND("cmdRightCycleIconTypes",        cmdRightCycleIconTypes);
   ADD_COMMAND("cmdRightGotoRecentDir",         cmdRightGotoRecentDir);
   ADD_COMMAND("cmdRightModeBrief",             cmdRightModeBrief);
   ADD_COMMAND("cmdRightModeFull",              cmdRightModeFull);
   ADD_COMMAND("cmdRightModeInfo",              cmdRightModeInfo);
   ADD_COMMAND("cmdRightModeTree",              cmdRightModeTree);
   ADD_COMMAND("cmdRightModeWide",              cmdRightModeWide);
   ADD_COMMAND("cmdRightOptionsColor",          cmdRightOptionsColor);
   ADD_COMMAND("cmdRightOptionsFileFilter",     cmdRightOptionsFileFilter);
   ADD_COMMAND("cmdRightOptionsFont",           cmdRightOptionsFont);
   ADD_COMMAND("cmdRightOptionsSort",           cmdRightOptionsSort);
   ADD_COMMAND("cmdRightOptionsStartup",        cmdRightOptionsStartup);
   ADD_COMMAND("cmdRightOptionsView",           cmdRightOptionsView);
   ADD_COMMAND("cmdRightReread",                cmdRightReread);
   ADD_COMMAND("cmdRightShowFileInfo",          cmdRightShowFileInfo);
   ADD_COMMAND("cmdRightSortByDate",            cmdRightSortByDate);
   ADD_COMMAND("cmdRightSortByExtension",       cmdRightSortByExtension);
   ADD_COMMAND("cmdRightSortByName",            cmdRightSortByName);
   ADD_COMMAND("cmdRightSortBySize",            cmdRightSortBySize);
   ADD_COMMAND("cmdRightSortByTime",            cmdRightSortByTime);
   ADD_COMMAND("cmdRightSortReverse",           cmdRightSortReverse);
   ADD_COMMAND("cmdRightSortUnsorted",          cmdRightSortUnsorted);
   ADD_COMMAND("cmdRightWalkDirectoryUp",       cmdRightWalkDirectoryUp);
   ADD_COMMAND("cmdRightWalkRootDirectory",     cmdRightWalkRootDirectory);
   ADD_COMMAND("cmdRunCommandInNewSession",     cmdRunCommandInNewSession);
   ADD_COMMAND("cmdSaveOptions",                cmdSaveOptions);
   ADD_COMMAND("cmdScrollCmdHistDown",          cmdScrollCmdHistDown);
   ADD_COMMAND("cmdScrollCmdHistUp",            cmdScrollCmdHistUp);
   ADD_COMMAND("cmdSearch",                     cmdSearch);
   ADD_COMMAND("cmdSearchNext",                 cmdSearchNext);
   ADD_COMMAND("cmdSelectAll",                  cmdSelectAll);
   ADD_COMMAND("cmdSelectGroup",                cmdSelectGroup);
   ADD_COMMAND("cmdSelfTestAll",                cmdSelfTestAll);
   ADD_COMMAND("cmdSelfTestLowLevelAPIs",       cmdSelfTestLowLevelAPIs);
   ADD_COMMAND("cmdSelfTestGarbageCollector",   cmdSelfTestGarbageCollector);
   ADD_COMMAND("cmdSelfTestObjectSynch",        cmdSelfTestObjectSynch);
   ADD_COMMAND("cmdSelfTestContainerClasses",   cmdSelfTestContainerClasses);
   ADD_COMMAND("cmdSelfTestLowLevelFileIO",     cmdSelfTestLowLevelFileIO);
   ADD_COMMAND("cmdSelfTestTextViewer",         cmdSelfTestTextViewer);
   ADD_COMMAND("cmdSelfTestWorkerThread",       cmdSelfTestWorkerThread);
   ADD_COMMAND("cmdSelfTestCopySingleFile",     cmdSelfTestCopySingleFile);
   ADD_COMMAND("cmdSelfTestMoveSingleFile",     cmdSelfTestMoveSingleFile);
   ADD_COMMAND("cmdSelfTestChangeFilenameCase", cmdSelfTestChangeFilenameCase);
   ADD_COMMAND("cmdSelfTestChangeFileAttributes", cmdSelfTestChangeFileAttributes);
   ADD_COMMAND("cmdSelfTestChangeFileTime",     cmdSelfTestChangeFileTime );
   ADD_COMMAND("cmdShowContextMenu",            cmdShowContextMenu);
   ADD_COMMAND("cmdShowFileInfo",               cmdShowFileInfo);
   ADD_COMMAND("cmdStoreDir0",                  cmdStoreDir0);
   ADD_COMMAND("cmdStoreDir1",                  cmdStoreDir1);
   ADD_COMMAND("cmdStoreDir2",                  cmdStoreDir2);
   ADD_COMMAND("cmdStoreDir3",                  cmdStoreDir3);
   ADD_COMMAND("cmdStoreDir4",                  cmdStoreDir4);
   ADD_COMMAND("cmdStoreDir5",                  cmdStoreDir5);
   ADD_COMMAND("cmdStoreDir6",                  cmdStoreDir6);
   ADD_COMMAND("cmdStoreDir7",                  cmdStoreDir7);
   ADD_COMMAND("cmdStoreDir8",                  cmdStoreDir8);
   ADD_COMMAND("cmdStoreDir9",                  cmdStoreDir9);
   ADD_COMMAND("cmdStoreDirIntoFirstFreeIndex", cmdStoreDirIntoFirstFreeIndex);
   ADD_COMMAND("cmdStoredDirs",                 cmdStoredDirs);
   ADD_COMMAND("cmdSwapPanels",                 cmdSwapPanels);
   ADD_COMMAND("cmdTest",                       cmdTest);
   ADD_COMMAND("cmdToggleCommandLine",          cmdToggleCommandLine);
   ADD_COMMAND("cmdToggleFullScreen",           cmdToggleFullScreen);
   ADD_COMMAND("cmdToggleHiddenFiles",          cmdToggleHiddenFiles);
   ADD_COMMAND("cmdToggleItemMark",             cmdToggleItemMark);
   ADD_COMMAND("cmdToggleItemMarkAndMoveDown",  cmdToggleItemMarkAndMoveDown);
   ADD_COMMAND("cmdToggleKeybar",               cmdToggleKeybar);
   ADD_COMMAND("cmdTogglePanels",               cmdTogglePanels);
   ADD_COMMAND("cmdToggleSaveOptionsOnExit",    cmdToggleSaveOptionsOnExit);
   ADD_COMMAND("cmdToggleStatusbar",            cmdToggleStatusbar);
   ADD_COMMAND("cmdToggleToolbar",              cmdToggleToolbar);
   ADD_COMMAND("cmdUndeleteFile",               cmdUndeleteFile);
   ADD_COMMAND("cmdUnselectGroup",              cmdUnselectGroup);
   ADD_COMMAND("cmdUnselectInConMon",           cmdUnselectInConMon);
   ADD_COMMAND("cmdUserCommand",                cmdUserCommand);
   ADD_COMMAND("cmdUserMenu",                   cmdUserMenu);
   ADD_COMMAND("cmdViewFile",                   cmdViewFile);
   ADD_COMMAND("cmdViewTypedFileName",          cmdViewTypedFileName);
   ADD_COMMAND("cmdWalkCurDirIntoOppositePanel",cmdWalkCurDirIntoOppositePanel);
   ADD_COMMAND("cmdWalkDirectoryDown",          cmdWalkDirectoryDown);
   ADD_COMMAND("cmdWalkDirectoryUp",            cmdWalkDirectoryUp);
   ADD_COMMAND("cmdWalkOppositeDirectory",      cmdWalkOppositeDirectory);
   ADD_COMMAND("cmdWalkRootDirectory",          cmdWalkRootDirectory);
   ADD_COMMAND("cmdWalkLeftToRightDir",         cmdWalkLeftToRightDir);
   ADD_COMMAND("cmdWalkRightToLeftDir",         cmdWalkRightToLeftDir);
END_COMMAND_TABLE;

void LCmdMainWindow::cmdLeftModeBrief ( GAbstractCommand* /*cmd*/ )
{
   lcmd.fp1->setViewMode(LCmdFilePanelViewOptions::VIEWMODE_BRIEF);
}

void LCmdMainWindow::cmdLeftModeFull ( GAbstractCommand* /*cmd*/ )
{
   lcmd.fp1->setViewMode(LCmdFilePanelViewOptions::VIEWMODE_FULL);
}

void LCmdMainWindow::cmdLeftModeWide ( GAbstractCommand* /*cmd*/ )
{
   lcmd.fp1->setViewMode(LCmdFilePanelViewOptions::VIEWMODE_WIDE);
}

void LCmdMainWindow::cmdLeftModeTree ( GAbstractCommand* /*cmd*/ )
{
#if __ENABLE_TREE_VIEW
   lcmd.fp1->setViewMode(LCmdFilePanelViewOptions::VIEWMODE_TREE);
#endif
}

void LCmdMainWindow::cmdLeftModeInfo ( GAbstractCommand* /*cmd*/ )
{
   lcmd.fp1->setViewMode(LCmdFilePanelViewOptions::VIEWMODE_INFO);
}

void LCmdMainWindow::cmdLeftCycleIconTypes ( GAbstractCommand* /*cmd*/ )
{
   lcmd.fp1->cmdToggleShowIcons(-1);
}

void LCmdMainWindow::cmdLeftSortByTime ( GAbstractCommand *cmd )
{
   lcmd.fp1->cmdSortByTime(cmd);
}

void LCmdMainWindow::cmdLeftSortByName ( GAbstractCommand *cmd )
{
   lcmd.fp1->cmdSortByName(cmd);
}

void LCmdMainWindow::cmdLeftSortByExtension ( GAbstractCommand *cmd )
{
   lcmd.fp1->cmdSortByExtension(cmd);
}

void LCmdMainWindow::cmdLeftSortByDate ( GAbstractCommand* cmd )
{
   lcmd.fp1->cmdSortByDate(cmd);
}

void LCmdMainWindow::cmdLeftSortBySize ( GAbstractCommand* cmd )
{
   lcmd.fp1->cmdSortBySize(cmd);
}

void LCmdMainWindow::cmdLeftSortUnsorted ( GAbstractCommand* cmd )
{
   lcmd.fp1->cmdSortByUnsorted(cmd);
}

void LCmdMainWindow::cmdLeftSortReverse ( GAbstractCommand* /*cmd*/ )
{
   lcmd.fp1->reverseSortOrder();
}

void LCmdMainWindow::cmdLeftReread ( GAbstractCommand* /*cmd*/ )
{
   lcmd.fp1->reRead(true);
}

void LCmdMainWindow::cmdLeftChooseDrive ( GAbstractCommand* /*cmd*/ )
{
   lcmd.fp1->cmdChangeDrive();
}

void LCmdMainWindow::viewFile ( const LCmdOptions::ProgOption& viewer )
{
   if (viewer.useExternal || !viewer.internalAvail)
   {
      viewer.launchIt(true, true);
   }
   else
   {
      GString path = viewer.params;
      if (path != "")
      {
         LCmdInternalViewer* v = new LCmdInternalViewer(path);
         v->setActive();
      }
   }
}

void LCmdMainWindow::cmdViewFile ( GAbstractCommand* /*cmd*/ )
{
   // Prepare the file for physical action.
   LCmdVfsFilePreparator fprep(*lcmd.curPanel);
   GVfs::File* f = fprep.prepareSelectedFile(*this);
   if (f != null) // If user did not cancel.
   {
      // OK, view the single file that is prepared by the virtual file system.
      LCmdOptions::ProgOption viewer = lcmd.options.viewer;
      if (!viewer.useExternal && viewer.internalAvail)
         viewer.params = f->getPhysicalPath();
      viewFile(viewer);
   }
}

void LCmdMainWindow::cmdViewTypedFileName ( GAbstractCommand* /*cmd*/ )
{
   // First, check if the current virtual file system supports this command.
   LCmdFilePanel& fp = *lcmd.curPanel;
   if (!fp.vfs.isRootOnly()) // Supported only on physical file system!
   {
      GString msg("%VfsErrCommandNotSupported");
      showMessageBox(msg, GMessageBox::TYPE_ERROR);
      return;
   }

   // Let the user specify the name of which file to view.
   // const GString cancelID = "<!>";
   GString def = fp.getCurItemPath();
   GString titleStr = "%cmdViewTypedFileName";
   GString fileName = userChooseFile(titleStr, def);
   if (fileName == "")
      return;

   // OK, view the single file which path is specified by user.
   LCmdOptions::ProgOption viewer = lcmd.options.viewer;
   viewer.workDir = fp.getCurrentSysDirectory(false);
   viewer.params = fileName;
   viewFile(viewer);
}

void LCmdMainWindow::cmdEditFile ( GAbstractCommand* /*cmd*/ )
{
   lcmd.options.editor.launchIt(true, true);
}

void LCmdMainWindow::cmdEditTypedFileName ( GAbstractCommand* /*cmd*/ )
{
   // Accept this command of the standard file system only.
   LCmdFilePanel& fp = *lcmd.curPanel;
   GVfs* fs = &fp.vfs.peek();
   if (dynamic_cast<GVfsLocal*>(fs) == null)
   {
      GString msg("%VfsErrCommandNotSupported");
      showMessageBox(msg, GMessageBox::TYPE_ERROR);
      return;
   }

   // ---
   LCmdOptions::ProgOption editor = lcmd.options.editor;
   GString def = fp.getCurItemName();
   editor.workDir = fp.getCurrentSysDirectory(false);
   GString titleStr = "%cmdEditTypedFileName";
   editor.params = userChooseFile(titleStr, def);
   if (editor.params == "")
      return;
   editor.launchIt(true, true);
}

void LCmdMainWindow::cmdCloneProgramInstance ( GAbstractCommand* /*cmd*/ )
{
   GString tempDir = lcmd.getEnvironmentVar("TEMP", "");
   if (tempDir == "")
      tempDir = lcmd.getEnvironmentVar("TMP", "");
   GVfs& vfsLocal = lcmd.curPanel->vfs.root();
   GString inipath = vfsLocal.getUniqueFileName(tempDir, "lcmd", ".tmp");
   if (!lcmd.writeProfile(true, false, &inipath))
   {
      GStringl msg("%Txt_Err_CreateFileX", GVArgs(inipath));
      showMessageBox(msg, GMessageBox::TYPE_ERROR);
      return;
   }
   GString rcpath = lcmd.getResourcePath();
   GString rcpathloc = lcmd.getResourcePathLocale();
   GString cmdstr = lcmd.getFullRootPath();
   if (cmdstr.contains(' ')) // Wrap in a pair of quotes if needed.
      cmdstr.insert('"').append('"', 1);
   cmdstr += " -iniFile \"" + inipath + "\"";
   cmdstr += " -delIni";
   if (vfsLocal.existFile(rcpath))
      cmdstr += " -rcFile \"" + rcpath + "\"";
   if (vfsLocal.existFile(rcpathloc))
      cmdstr += " -rcFileLocale \"" + rcpathloc + "\"";
   int flags = LCmdCmdLineEntry::ECF_FORCE_NEW_SESSION |
               LCmdCmdLineEntry::ECF_CLOSEON_EXIT;
   subClient->cmdcont.cmdline.entry.executeCommand(cmdstr, "!h", flags);
}

void LCmdMainWindow::cmdCopyFile ( GAbstractCommand* /*cmd*/ )
{
   lcmd.curPanel->copyOrMovePanelFiles(false);
}

void LCmdMainWindow::cmdCopyFileToSameDir ( GAbstractCommand* /*cmd*/ )
{
   lcmd.curPanel->copyOrMovePanelFiles(false, true);
}

void LCmdMainWindow::cmdMoveFile ( GAbstractCommand* /*cmd*/ )
{
   lcmd.curPanel->copyOrMovePanelFiles(true);
}

void LCmdMainWindow::cmdRenameFile ( GAbstractCommand* /*cmd*/ )
{
   lcmd.curPanel->copyOrMovePanelFiles(true, true);
}

void LCmdMainWindow::cmdMakeDir ( GAbstractCommand* /*cmd*/ )
{
   lcmd.curPanel->makeDir.cmdMakeDir(*lcmd.curPanel);
}

void LCmdMainWindow::cmdDeleteFile ( GAbstractCommand* /*cmd*/ )
{
   LCmdDelete::CmdDelete(*lcmd.curPanel);
}

void LCmdMainWindow::cmdUndeleteFile ( GAbstractCommand* /*cmd*/ )
{
   if (lcmd.options.undel.useExternal)
      lcmd.options.undel.launchIt(true, true);
   else
      GSystem::OpenRecycleBinObject();
}

void LCmdMainWindow::cmdCompareFiles ( GAbstractCommand* /*cmd*/ )
{
   lcmd.options.fdiff.launchIt(true, true);
}

void LCmdMainWindow::cmdFindFile ( GAbstractCommand* /*cmd*/ )
{
   lcmd.options.ffind.launchIt(true, true);
}

void LCmdMainWindow::cmdFindDuplicateFiles ( GAbstractCommand* /*cmd*/ )
{
   lcmd.options.cclean.launchIt(true, true);
}

void LCmdMainWindow::cmdEditEAs ( GAbstractCommand* /*cmd*/ )
{
   if (lcmd.options.eas.useExternal)
      lcmd.options.eas.launchIt(true, true);
   else
      LCmdDlgEAEditor::EditEAs(this);
}

void LCmdMainWindow::cmdShowFileInfo ( GAbstractCommand* /*cmd*/ )
{
   lcmd.curPanel->cmdShowFileInfo();
}

void LCmdMainWindow::cmdLeftShowFileInfo ( GAbstractCommand* /*cmd*/ )
{
   lcmd.fp1->cmdShowFileInfo();
}

void LCmdMainWindow::cmdRightShowFileInfo ( GAbstractCommand* /*cmd*/ )
{
   lcmd.fp2->cmdShowFileInfo();
}

void LCmdMainWindow::cmdChooseFileAttrib ( GAbstractCommand* /*cmd*/ )
{
   LCmdDlgFileAttributes dlg(*lcmd.curPanel);
   dlg.execute();
}

void LCmdMainWindow::cmdLeftChooseFileAttrib ( GAbstractCommand* /*cmd*/ )
{
   LCmdDlgFileAttributes dlg(*lcmd.fp1);
   dlg.execute();
}

void LCmdMainWindow::cmdRightChooseFileAttrib ( GAbstractCommand* /*cmd*/ )
{
   LCmdDlgFileAttributes dlg(*lcmd.fp2);
   dlg.execute();
}

void LCmdMainWindow::cmdChooseFileNameCase ( GAbstractCommand* /*cmd*/ )
{
   LCmdDlgFNameCase dlg(*lcmd.curPanel);
   dlg.execute();
}

void LCmdMainWindow::cmdCompareDirs ( GAbstractCommand* /*cmd*/ )
{
   LCmdDlgCompareDirs dlg(*lcmd.curPanel);
   dlg.execute();
}

void LCmdMainWindow::cmdChooseFileTime ( GAbstractCommand* /*cmd*/ )
{
   LCmdDlgFileTime dlg(*lcmd.curPanel);
   dlg.execute();
}

void LCmdMainWindow::cmdLeftChooseFileTime ( GAbstractCommand* /*cmd*/ )
{
   LCmdDlgFileTime dlg(*lcmd.fp1);
   dlg.execute();
}

void LCmdMainWindow::cmdRightChooseFileTime ( GAbstractCommand* /*cmd*/ )
{
   LCmdDlgFileTime dlg(*lcmd.fp2);
   dlg.execute();
}

void LCmdMainWindow::cmdOpenSysFolder ( GAbstractCommand* /*cmd*/ )
{
   lcmd.curPanel->cmdOpenSysFolderCurDir();
}

void LCmdMainWindow::cmdCreateSysObject ( GAbstractCommand* /*cmd*/ )
{
   lcmd.curPanel->cmdCreateSysObject();
}

void LCmdMainWindow::cmdDebugOptions ( GAbstractCommand* /*cmd*/ )
{
   LCmdDlgDebugOptions dlg;
   dlg.execute(this);
}

void LCmdMainWindow::cmdDebugShowLogFile ( GAbstractCommand* /*cmd*/ )
{
   GString path = GLog::GetLogFilePath();
   if (path == "")
   {
      showMessageBox("%Txt_Err_NoLogFile"); // No log-file has been created yet.
   }
   else
   {
      LCmdOptions::ProgOption viewer = lcmd.options.viewer;
      if (!viewer.useExternal && viewer.internalAvail)
         viewer.params = path;

      // Disable logging while we are loading the log-file.
      GLog::FilterId prevLogLevel = GLog::GetFilterLevel();
      GLog::SetFilterLevel(GLog::LOGGING_IS_OFF);
      try {
         viewFile(viewer);
         GLog::SetFilterLevel(prevLogLevel);
      } catch(...) {
         GLog::SetFilterLevel(prevLogLevel);
         throw;
      }
   }
}

void LCmdMainWindow::doSelfTest ( LCmdSelfTest::TestID tid )
{
   LCmdSelfTest st(tid);
   GWorkerThread wt("DlgSelfTestMon", 0, &st, false, "SelfTestWork");
   wt.workModal(*this, 0);
}

void LCmdMainWindow::cmdSelfTestAll ( GAbstractCommand* /*cmd*/ )
{
   doSelfTest(LCmdSelfTest::TID_All);
}

void LCmdMainWindow::cmdSelfTestLowLevelAPIs ( GAbstractCommand* /*cmd*/ )
{
   doSelfTest(LCmdSelfTest::TID_LowLevelAPIs);
}

void LCmdMainWindow::cmdSelfTestGarbageCollector ( GAbstractCommand* /*cmd*/ )
{
   doSelfTest(LCmdSelfTest::TID_GarbageCollector);
}

void LCmdMainWindow::cmdSelfTestObjectSynch ( GAbstractCommand* /*cmd*/ )
{
   doSelfTest(LCmdSelfTest::TID_ObjectSynch);
}

void LCmdMainWindow::cmdSelfTestContainerClasses ( GAbstractCommand* /*cmd*/ )
{
   doSelfTest(LCmdSelfTest::TID_ContainerClasses);
}

void LCmdMainWindow::cmdSelfTestLowLevelFileIO ( GAbstractCommand* /*cmd*/ )
{
   doSelfTest(LCmdSelfTest::TID_LowLevelFileIO);
}

void LCmdMainWindow::cmdSelfTestTextViewer ( GAbstractCommand* /*cmd*/ )
{
   doSelfTest(LCmdSelfTest::TID_TextViewer);
}

void LCmdMainWindow::cmdSelfTestWorkerThread ( GAbstractCommand* /*cmd*/ )
{
   doSelfTest(LCmdSelfTest::TID_WorkerThread);
}

void LCmdMainWindow::cmdSelfTestCopySingleFile ( GAbstractCommand* /*cmd*/ )
{
   doSelfTest(LCmdSelfTest::TID_CopySingleFile);
}

void LCmdMainWindow::cmdSelfTestMoveSingleFile ( GAbstractCommand* /*cmd*/ )
{
   doSelfTest(LCmdSelfTest::TID_MoveSingleFile);
}

void LCmdMainWindow::cmdSelfTestChangeFilenameCase ( GAbstractCommand* /*cmd*/ )
{
   doSelfTest(LCmdSelfTest::TID_ChangeFilenameCase);
}

void LCmdMainWindow::cmdSelfTestChangeFileAttributes ( GAbstractCommand* /*cmd*/ )
{
   doSelfTest(LCmdSelfTest::TID_ChangeFileAttributes);
}

void LCmdMainWindow::cmdSelfTestChangeFileTime ( GAbstractCommand* /*cmd*/ )
{
   doSelfTest(LCmdSelfTest::TID_ChangeFileTime);
}

void LCmdMainWindow::cmdSelectGroup ( GAbstractCommand* /*cmd*/ )
{
   lcmd.curPanel->cmdSelectGroup();
}

void LCmdMainWindow::cmdUnselectGroup ( GAbstractCommand* /*cmd*/ )
{
   lcmd.curPanel->cmdUnselectGroup();
}

void LCmdMainWindow::cmdUnselectInConMon ( GAbstractCommand* cmd )
{
   subClient->cmdcont.conmon.cmdUnselectAll(cmd);
}

void LCmdMainWindow::cmdInvertSelection ( GAbstractCommand* /*cmd*/ )
{
   lcmd.curPanel->cmdInvertSelection();
}

void LCmdMainWindow::cmdCloseAndExit ( GAbstractCommand* /*cmd*/ )
{
   if (lcmd.exitWorker == null)
      lcmd.exitWorker = new LCmdExitWorker();
   lcmd.exitWorker->performExit();
}

void LCmdMainWindow::cmdStoredDirs ( GAbstractCommand* /*cmd*/ )
{
   lcmd.options.quickDirList.pickFromDialog(*lcmd.curPanel);
}

void LCmdMainWindow::cmdActivateStoredDir0 ( GAbstractCommand* /*cmd*/ )
{
   lcmd.options.quickDirList.retrieve(*lcmd.curPanel, 0);
}

void LCmdMainWindow::cmdActivateStoredDir1 ( GAbstractCommand* /*cmd*/ )
{
   lcmd.options.quickDirList.retrieve(*lcmd.curPanel, 1);
}

void LCmdMainWindow::cmdActivateStoredDir2 ( GAbstractCommand* /*cmd*/ )
{
   lcmd.options.quickDirList.retrieve(*lcmd.curPanel, 2);
}

void LCmdMainWindow::cmdActivateStoredDir3 ( GAbstractCommand* /*cmd*/ )
{
   lcmd.options.quickDirList.retrieve(*lcmd.curPanel, 3);
}

void LCmdMainWindow::cmdActivateStoredDir4 ( GAbstractCommand* /*cmd*/ )
{
   lcmd.options.quickDirList.retrieve(*lcmd.curPanel, 4);
}

void LCmdMainWindow::cmdActivateStoredDir5 ( GAbstractCommand* /*cmd*/ )
{
   lcmd.options.quickDirList.retrieve(*lcmd.curPanel, 5);
}

void LCmdMainWindow::cmdActivateStoredDir6 ( GAbstractCommand* /*cmd*/ )
{
   lcmd.options.quickDirList.retrieve(*lcmd.curPanel, 6);
}

void LCmdMainWindow::cmdActivateStoredDir7 ( GAbstractCommand* /*cmd*/ )
{
   lcmd.options.quickDirList.retrieve(*lcmd.curPanel, 7);
}

void LCmdMainWindow::cmdActivateStoredDir8 ( GAbstractCommand* /*cmd*/ )
{
   lcmd.options.quickDirList.retrieve(*lcmd.curPanel, 8);
}

void LCmdMainWindow::cmdActivateStoredDir9 ( GAbstractCommand* /*cmd*/ )
{
   lcmd.options.quickDirList.retrieve(*lcmd.curPanel, 9);
}

void LCmdMainWindow::cmdStoreDir0 ( GAbstractCommand* /*cmd*/ )
{
   lcmd.options.quickDirList.add(*lcmd.curPanel, 0);
}

void LCmdMainWindow::cmdStoreDir1 ( GAbstractCommand* /*cmd*/ )
{
   lcmd.options.quickDirList.add(*lcmd.curPanel, 1);
}

void LCmdMainWindow::cmdStoreDir2 ( GAbstractCommand* /*cmd*/ )
{
   lcmd.options.quickDirList.add(*lcmd.curPanel, 2);
}

void LCmdMainWindow::cmdStoreDir3 ( GAbstractCommand* /*cmd*/ )
{
   lcmd.options.quickDirList.add(*lcmd.curPanel, 3);
}

void LCmdMainWindow::cmdStoreDir4 ( GAbstractCommand* /*cmd*/ )
{
   lcmd.options.quickDirList.add(*lcmd.curPanel, 4);
}

void LCmdMainWindow::cmdStoreDir5 ( GAbstractCommand* /*cmd*/ )
{
   lcmd.options.quickDirList.add(*lcmd.curPanel, 5);
}

void LCmdMainWindow::cmdStoreDir6 ( GAbstractCommand* /*cmd*/ )
{
   lcmd.options.quickDirList.add(*lcmd.curPanel, 6);
}

void LCmdMainWindow::cmdStoreDir7 ( GAbstractCommand* /*cmd*/ )
{
   lcmd.options.quickDirList.add(*lcmd.curPanel, 7);
}

void LCmdMainWindow::cmdStoreDir8 ( GAbstractCommand* /*cmd*/ )
{
   lcmd.options.quickDirList.add(*lcmd.curPanel, 8);
}

void LCmdMainWindow::cmdStoreDir9 ( GAbstractCommand* /*cmd*/ )
{
   lcmd.options.quickDirList.add(*lcmd.curPanel, 9);
}

void LCmdMainWindow::cmdStoreDirIntoFirstFreeIndex ( GAbstractCommand* /*cmd*/ )
{
   lcmd.options.quickDirList.add(*lcmd.curPanel, -1);
}

void LCmdMainWindow::cmdCommandHistory ( GAbstractCommand* /*cmd*/ )
{
   subClient->cmdcont.cmdline.entry.cmdHist.pickFromDialog();
}

void LCmdMainWindow::cmdDirHistory ( GAbstractCommand* /*cmd*/ )
{
   subClient->cmdcont.cmdline.entry.dirHist.pickFromDialog();
}

void LCmdMainWindow::cmdSwapPanels ( GAbstractCommand* /*cmd*/ )
{
   LCmdFilePanel* tmp1 = lcmd.fp1;
   lcmd.fp1 = lcmd.fp2;
   lcmd.fp2 = tmp1;

   lcmd.options.panel1RelWidth = 1000 - lcmd.options.panel1RelWidth;

   LCmdPanelsParent& pp = subClient->panelsParent;
   GString tmp2(pp.frame1.getLayoutConstraints());
   pp.frame1.setLayoutConstraints(pp.frame2.getLayoutConstraints(), false);
   pp.frame2.setLayoutConstraints(tmp2, true);

   ensureFocusOK();
}

void LCmdMainWindow::cmdTogglePanels ( GAbstractCommand* /*cmd*/ )
{
   subClient->setFilePanelsVisible(-1);
}

void LCmdMainWindow::cmdCalcDirSize ( GAbstractCommand* /*cmd*/ )
{
   LCmdDlgCalcDirSize::CmdCalcSubDirSize(*lcmd.curPanel);
}

void LCmdMainWindow::cmdCalcRootDirSize ( GAbstractCommand* /*cmd*/ )
{
   LCmdDlgCalcDirSize::CmdCalcRootDirSize(*lcmd.curPanel);
}

void LCmdMainWindow::cmdUserMenu ( GAbstractCommand* /*cmd*/ )
{
   int ypos = 1;
   GKeyBar* kbar = getKeybar();
   if (kbar != null && kbar->isVisible())
      ypos += kbar->getWindowSize().height;
   GStatusbar* sbar = getStatusbar();
   if (sbar != null && sbar->isVisible())
      ypos += sbar->getWindowSize().height;
   setVisiblePopupMenu(true, 1, ypos);
}

void LCmdMainWindow::cmdProcessFilesByCommand ( GAbstractCommand* /*cmd*/ )
{
   GStringl msg("%Txt_Err_NotImplemented");
   showMessageBox(msg);
}

void LCmdMainWindow::cmdToggleToolbar ( GAbstractCommand* /*cmd*/ )
{
   bool show = !isToolbarVisible();
   setToolbarVisible(show);
}

void LCmdMainWindow::cmdToggleCommandLine ( GAbstractCommand* /*cmd*/ )
{
   subClient->setCommandLineVisible(-1);
}

void LCmdMainWindow::cmdToggleKeybar ( GAbstractCommand* /*cmd*/ )
{
   bool show = !isKeybarVisible();
   setKeybarVisible(show);
}

void LCmdMainWindow::cmdToggleStatusbar ( GAbstractCommand* /*cmd*/ )
{
   bool show = !isStatusbarVisible();
   setStatusbarVisible(show);
   if (show) // Update immediately if it is made visible.
      onTimer("INFOUPDATE", null);
}

void LCmdMainWindow::cmdToggleFullScreen ( GAbstractCommand* /*cmd*/ )
{
   subClient->setFullScreenMode(-1);
}

void LCmdMainWindow::cmdToggleHiddenFiles ( GAbstractCommand* /*cmd*/ )
{
   bool oldFlag = lcmd.curPanel->isShowHiddenFiles();
   lcmd.curPanel->setShowHiddenFiles(!oldFlag);
}

void LCmdMainWindow::cmdToggleSaveOptionsOnExit ( GAbstractCommand* /*cmd*/ )
{
   lcmd.options.saveOnExit.everything = !lcmd.options.saveOnExit.everything;

   // True = Save only current value of lcmd.options.saveOnExit.*
   lcmd.writeProfile(false, true);
}

void LCmdMainWindow::cmdLeftOptionsStartup ( GAbstractCommand* /*cmd*/ )
{
   LCmdDlgOptions::CmdJumpToOptionsPage("LStartup");
}

void LCmdMainWindow::cmdRightOptionsStartup ( GAbstractCommand* /*cmd*/ )
{
   LCmdDlgOptions::CmdJumpToOptionsPage("RStartup");
}

void LCmdMainWindow::cmdLeftOptionsFileFilter ( GAbstractCommand* /*cmd*/ )
{
   LCmdDlgOptions::CmdJumpToOptionsPage("LFilter");
}

void LCmdMainWindow::cmdRightOptionsFileFilter ( GAbstractCommand* /*cmd*/ )
{
   LCmdDlgOptions::CmdJumpToOptionsPage("RFilter");
}

void LCmdMainWindow::cmdLeftOptionsSort ( GAbstractCommand* /*cmd*/ )
{
   LCmdDlgOptions::CmdJumpToOptionsPage("LSort");
}

void LCmdMainWindow::cmdRightOptionsSort ( GAbstractCommand* /*cmd*/ )
{
   LCmdDlgOptions::CmdJumpToOptionsPage("RSort");
}

void LCmdMainWindow::cmdLeftOptionsView ( GAbstractCommand* /*cmd*/ )
{
   LCmdDlgOptions::CmdJumpToOptionsPage("LView");
}

void LCmdMainWindow::cmdRightOptionsView ( GAbstractCommand* /*cmd*/ )
{
   LCmdDlgOptions::CmdJumpToOptionsPage("RView");
}

void LCmdMainWindow::cmdOptionsDialog ( GAbstractCommand* /*cmd*/ )
{
   LCmdDlgOptions::CmdJumpToOptionsPage(""); // Start with the recent active page.
}

void LCmdMainWindow::cmdOptionsView ( GAbstractCommand* /*cmd*/ )
{
   LCmdDlgOptions::CmdJumpToOptionsPage("ViewCmn");
}

void LCmdMainWindow::cmdOptionsStyle ( GAbstractCommand* /*cmd*/ )
{
   LCmdDlgOptions::CmdJumpToOptionsPage("ViewStyle");
}

void LCmdMainWindow::cmdOptionsConfirmations ( GAbstractCommand* /*cmd*/ )
{
   LCmdDlgOptions::CmdJumpToOptionsPage("Confirm");
}

void LCmdMainWindow::cmdOptionsSaveOnExit ( GAbstractCommand* /*cmd*/ )
{
   LCmdDlgOptions::CmdJumpToOptionsPage("SaveOpt");
}

void LCmdMainWindow::cmdOptionsFileCopyMv ( GAbstractCommand* /*cmd*/ )
{
   LCmdDlgOptions::CmdJumpToOptionsPage("FileCopy");
}

void LCmdMainWindow::cmdOptionsVarious ( GAbstractCommand* /*cmd*/ )
{
   LCmdDlgOptions::CmdJumpToOptionsPage("Various");
}

void LCmdMainWindow::cmdOptionsConsoleVarious ( GAbstractCommand* /*cmd*/ )
{
   LCmdDlgOptions::CmdJumpToOptionsPage("ConVarious");
}

void LCmdMainWindow::cmdOptionsVfsVarious ( GAbstractCommand* /*cmd*/ )
{
   LCmdDlgOptions::CmdJumpToOptionsPage("VfsVarious");
}

void LCmdMainWindow::cmdOptionsVfsZip ( GAbstractCommand* /*cmd*/ )
{
   LCmdDlgOptions::CmdJumpToOptionsPage("VfsZip");
}

void LCmdMainWindow::cmdOptionsFileDel ( GAbstractCommand* /*cmd*/ )
{
   LCmdDlgOptions::CmdJumpToOptionsPage("FileDel");
}

void LCmdMainWindow::cmdOptionsDirCache ( GAbstractCommand* /*cmd*/ )
{
   LCmdDlgOptions::CmdJumpToOptionsPage("DirCache");
}

void LCmdMainWindow::cmdLeftOptionsFont ( GAbstractCommand* /*cmd*/ )
{
   LCmdDlgOptions::CmdJumpToOptionsPage("LFont");
}

void LCmdMainWindow::cmdRightOptionsFont ( GAbstractCommand* /*cmd*/ )
{
   LCmdDlgOptions::CmdJumpToOptionsPage("RFont");
}

void LCmdMainWindow::cmdOptionsFont ( GAbstractCommand* /*cmd*/ )
{
   LCmdDlgOptions::CmdJumpToOptionsPage("FontCmn");
}

void LCmdMainWindow::cmdLeftOptionsColor ( GAbstractCommand* /*cmd*/ )
{
   LCmdDlgOptions::CmdJumpToOptionsPage("LColor");
}

void LCmdMainWindow::cmdRightOptionsColor ( GAbstractCommand* /*cmd*/ )
{
   LCmdDlgOptions::CmdJumpToOptionsPage("RColor");
}

void LCmdMainWindow::cmdOptionsColor ( GAbstractCommand* /*cmd*/ )
{
   LCmdDlgOptions::CmdJumpToOptionsPage("ColorCmn");
}

void LCmdMainWindow::cmdProgOptionsEditor ( GAbstractCommand* /*cmd*/ )
{
   LCmdDlgOptions::CmdJumpToOptionsPage("ProgEdit");
}

void LCmdMainWindow::cmdProgOptionsViewer ( GAbstractCommand* /*cmd*/ )
{
   LCmdDlgOptions::CmdJumpToOptionsPage("ProgView");
}

void LCmdMainWindow::cmdProgOptionsFileFinder ( GAbstractCommand* /*cmd*/ )
{
   LCmdDlgOptions::CmdJumpToOptionsPage("ProgFFind");
}

void LCmdMainWindow::cmdProgOptionsFileDiff ( GAbstractCommand* /*cmd*/ )
{
   LCmdDlgOptions::CmdJumpToOptionsPage("ProgFDiff");
}

void LCmdMainWindow::cmdProgOptionsFileDupFinder ( GAbstractCommand* /*cmd*/ )
{
   LCmdDlgOptions::CmdJumpToOptionsPage("ProgCClean");
}

void LCmdMainWindow::cmdProgOptionsUndeleter ( GAbstractCommand* /*cmd*/ )
{
   LCmdDlgOptions::CmdJumpToOptionsPage("ProgUndel");
}

void LCmdMainWindow::cmdProgOptionsEABrowser ( GAbstractCommand* /*cmd*/ )
{
   LCmdDlgOptions::CmdJumpToOptionsPage("ProgEAEdit");
}

void LCmdMainWindow::cmdOptionsInternalCommands ( GAbstractCommand* /*cmd*/ )
{
   LCmdDlgOptions::CmdJumpToOptionsPage("Cmd");
}

void LCmdMainWindow::cmdOptionsEnvironmentVars ( GAbstractCommand* /*cmd*/ )
{
   LCmdDlgOptions::CmdJumpToOptionsPage("Env");
}

void LCmdMainWindow::cmdOptionsAliases ( GAbstractCommand* /*cmd*/ )
{
   LCmdDlgOptions::CmdJumpToOptionsPage("Aliases");
}

void LCmdMainWindow::cmdOptionsSSA ( GAbstractCommand* /*cmd*/ )
{
   LCmdDlgOptions::CmdJumpToOptionsPage("SSA");
}

void LCmdMainWindow::cmdSaveOptions ( GAbstractCommand* /*cmd*/ )
{
   lcmd.writeProfile(true, false);
}

void LCmdMainWindow::cmdRightModeBrief ( GAbstractCommand* /*cmd*/ )
{
   lcmd.fp2->setViewMode(LCmdFilePanelViewOptions::VIEWMODE_BRIEF);
}

void LCmdMainWindow::cmdRightModeFull ( GAbstractCommand* /*cmd*/ )
{
   lcmd.fp2->setViewMode(LCmdFilePanelViewOptions::VIEWMODE_FULL);
}

void LCmdMainWindow::cmdRightModeWide ( GAbstractCommand* /*cmd*/ )
{
   lcmd.fp2->setViewMode(LCmdFilePanelViewOptions::VIEWMODE_WIDE);
}

void LCmdMainWindow::cmdRightModeTree ( GAbstractCommand* /*cmd*/ )
{
#if __ENABLE_TREE_VIEW
   lcmd.fp2->setViewMode(LCmdFilePanelViewOptions::VIEWMODE_TREE);
#endif
}

void LCmdMainWindow::cmdRightModeInfo ( GAbstractCommand* /*cmd*/ )
{
   lcmd.fp2->setViewMode(LCmdFilePanelViewOptions::VIEWMODE_INFO);
}

void LCmdMainWindow::cmdRightCycleIconTypes ( GAbstractCommand* /*cmd*/ )
{
   lcmd.fp2->cmdToggleShowIcons(-1);
}

void LCmdMainWindow::cmdRightSortByTime ( GAbstractCommand *cmd )
{
   lcmd.fp2->cmdSortByTime(cmd);
}

void LCmdMainWindow::cmdRightSortByName ( GAbstractCommand *cmd )
{
   lcmd.fp2->cmdSortByName(cmd);
}

void LCmdMainWindow::cmdRightSortByExtension ( GAbstractCommand *cmd )
{
   lcmd.fp2->cmdSortByExtension(cmd);
}

void LCmdMainWindow::cmdRightSortByDate ( GAbstractCommand *cmd )
{
   lcmd.fp2->cmdSortByDate(cmd);
}

void LCmdMainWindow::cmdRightSortBySize ( GAbstractCommand *cmd )
{
   lcmd.fp2->cmdSortBySize(cmd);
}

void LCmdMainWindow::cmdRightSortUnsorted ( GAbstractCommand *cmd )
{
   lcmd.fp2->cmdSortByUnsorted(cmd);
}

void LCmdMainWindow::cmdRightSortReverse ( GAbstractCommand* /*cmd*/ )
{
   lcmd.fp2->reverseSortOrder();
}

void LCmdMainWindow::cmdRightReread ( GAbstractCommand* /*cmd*/ )
{
   lcmd.fp2->reRead(true);
}

void LCmdMainWindow::cmdRightChooseDrive ( GAbstractCommand* /*cmd*/ )
{
   lcmd.fp2->cmdChangeDrive();
}

void LCmdMainWindow::cmdHelpContents ( GAbstractCommand* /*cmd*/ )
{
   GStringl msg("%Txt_Err_NotImplemented");
   showMessageBox(msg);
}

void LCmdMainWindow::cmdHelpIndex ( GAbstractCommand* /*cmd*/ )
{
   GStringl msg("%Txt_Err_NotImplemented");
   showMessageBox(msg);
}

void LCmdMainWindow::cmdHelpKeyboard ( GAbstractCommand* /*cmd*/ )
{
   GStringl msg("%Txt_Err_NotImplemented");
   showMessageBox(msg);
}

void LCmdMainWindow::cmdHelpCmdLineArgs ( GAbstractCommand* /*cmd*/ )
{
   GStringl helpText("%TxtSupportedCommandLineArguments");
   GStringl helpTextDetails("%TxtSupportedCommandLineArguments_Details");
   showMessageBox(helpText, GMessageBox::TYPE_INFO, "Do", GString::Empty, false, helpTextDetails);
}

void LCmdMainWindow::cmdHelpAbout ( GAbstractCommand* /*cmd*/ )
{
   lcmd.aboutBox.execute(this);
}

void LCmdMainWindow::cmdHelpShowTip ( GAbstractCommand* /*cmd*/ )
{
   LCmdDlgStartupTip::Execute(*this);
}

void LCmdMainWindow::cmdUserCommand ( GAbstractCommand* cmd )
{
   if (cmd != null)
   {
      GString cmdstr = cmd->getUserData1();
      GString workDir = "";
      int flags = LCmdCmdLineEntry::ECF_FORCE_NEW_SESSION |
                  LCmdCmdLineEntry::ECF_DO_ECHO |
                  LCmdCmdLineEntry::ECF_CLOSEON_EXIT | 
                  LCmdCmdLineEntry::ECF_DONT_ADD_TO_CMD_HIST;
      LCmdCmdLineEntry& entry = subClient->cmdcont.cmdline.entry;
      entry.executeCommand(cmdstr, workDir, flags);
   }
}

void LCmdMainWindow::cmdCurPanelModeBrief ( GAbstractCommand* /*cmd*/ )
{
   lcmd.curPanel->setViewMode(LCmdFilePanelViewOptions::VIEWMODE_BRIEF);
}

void LCmdMainWindow::cmdCurPanelModeFull ( GAbstractCommand* /*cmd*/ )
{
   lcmd.curPanel->setViewMode(LCmdFilePanelViewOptions::VIEWMODE_FULL);
}

void LCmdMainWindow::cmdCurPanelModeWide ( GAbstractCommand* /*cmd*/ )
{
   lcmd.curPanel->setViewMode(LCmdFilePanelViewOptions::VIEWMODE_WIDE);
}

void LCmdMainWindow::cmdCurPanelModeTree ( GAbstractCommand* /*cmd*/ )
{
#if __ENABLE_TREE_VIEW
   lcmd.curPanel->setViewMode(LCmdFilePanelViewOptions::VIEWMODE_TREE);
#endif
}

void LCmdMainWindow::cmdCurPanelSortByTime ( GAbstractCommand *cmd )
{
   lcmd.curPanel->cmdSortByTime(cmd);
}

void LCmdMainWindow::cmdCurPanelSortByName ( GAbstractCommand *cmd )
{
   lcmd.curPanel->cmdSortByName(cmd);
}

void LCmdMainWindow::cmdCurPanelSortByExtension ( GAbstractCommand *cmd )
{
   lcmd.curPanel->cmdSortByExtension(cmd);
}

void LCmdMainWindow::cmdCurPanelSortByDate ( GAbstractCommand *cmd )
{
   lcmd.curPanel->cmdSortByDate(cmd);
}

void LCmdMainWindow::cmdCurPanelSortBySize ( GAbstractCommand *cmd )
{
   lcmd.curPanel->cmdSortBySize(cmd);
}

void LCmdMainWindow::cmdCurPanelSortUnsorted ( GAbstractCommand *cmd )
{
   lcmd.curPanel->cmdSortByUnsorted(cmd);
}

void LCmdMainWindow::cmdCurPanelSortReverse ( GAbstractCommand* /*cmd*/ )
{
   lcmd.curPanel->reverseSortOrder();
}

void LCmdMainWindow::cmdCurPanelOptionsSort ( GAbstractCommand* /*cmd*/ )
{
   LCmdDlgOptions::CmdJumpToOptionsPage(lcmd.curPanel == lcmd.fp1 ? "LSort" : "RSort");
}

void LCmdMainWindow::cmdCurPanelCycleIconTypes ( GAbstractCommand* /*cmd*/ )
{
   lcmd.curPanel->cmdToggleShowIcons(-1);
}

void LCmdMainWindow::cmdCurPanelReread ( GAbstractCommand* /*cmd*/ )
{
   setStatusbarText(GString::Empty);
   lcmd.curPanel->reRead(true);
}

void LCmdMainWindow::cmdSearch ( GAbstractCommand* cmd )
{
   if (!lcmd.options.panelsFullScreen)
      subClient->cmdcont.conmon.cmdSearch(cmd);
}

void LCmdMainWindow::cmdSearchNext ( GAbstractCommand* cmd )
{
   if (!lcmd.options.panelsFullScreen)
      subClient->cmdcont.conmon.cmdSearchNext(cmd);
}

void LCmdMainWindow::cmdGotoLineNum ( GAbstractCommand* cmd )
{
   if (!lcmd.options.panelsFullScreen)
      subClient->cmdcont.conmon.cmdGoto(cmd);
}

void LCmdMainWindow::cmdCopy ( GAbstractCommand* cmd )
{
   if (!lcmd.options.panelsFullScreen)
      subClient->cmdcont.conmon.cmdCopy(cmd);
}

void LCmdMainWindow::cmdSelectAll ( GAbstractCommand* cmd )
{
   if (!lcmd.options.panelsFullScreen)
      subClient->cmdcont.conmon.cmdSelectAll(cmd);
}

void LCmdMainWindow::cmdCurPanelGotoRecentDir ( GAbstractCommand* /*cmd*/ )
{
   lcmd.curPanel->activatePreviousDir();
}

void LCmdMainWindow::cmdLeftGotoRecentDir ( GAbstractCommand* /*cmd*/ )
{
   lcmd.fp1->activatePreviousDir();
}

void LCmdMainWindow::cmdRightGotoRecentDir ( GAbstractCommand* /*cmd*/ )
{
   lcmd.fp2->activatePreviousDir();
}

void LCmdMainWindow::cmdShowContextMenu ( GAbstractCommand* /*cmd*/ )
{
   // ---
   GKeyBar* keybar = getKeybar();
   if (keybar != null)
   {
      keybar->setShiftFlag(false);
      keybar->setAltFlag(false);
      keybar->setCtrlFlag(false);
   }

   // ---
   LCmdFilePanel& curPanel = LCmdFilePanel::GetCurrentPanel();
   int selIndex = curPanel.getCurrentSelectedIndex();

   // ---
   GRectangle r;
   LCmdCmdContainer& cmdcont = subClient->cmdcont; // Make this a fast one
   if (getFocusedWindow() == &cmdcont.conmon)
   {
      int xpos = cmdcont.conmon.getCaretXPos() + 5;
      int ypos = cmdcont.conmon.getCaretYPos();
      cmdcont.conmon.setVisiblePopupMenu(true, xpos, ypos);
   }
   else
   if (lcmd.options.panelsAreVisible &&
       curPanel.calcItemRect(selIndex, r))
   {
      int xpos = r.x + (r.width / 2);
      int ypos = r.y + r.height + 1;
      curPanel.setVisiblePopupMenu(true, xpos, ypos);
   }
   else
   if (cmdcont.cmdline.entry.containsFocusedWindow())
   {
      cmdcont.cmdline.entry.setVisiblePopupMenu(true, 10, 10);
   }
}

void LCmdMainWindow::cmdNavigateDown ( GAbstractCommand* /*cmd*/ )
{
   if (lcmd.options.panelsAreVisible)
      lcmd.curPanel->selectItem(LCmdFilePanel::NavigateDown);
   else
      subClient->cmdcont.cmdline.entry.cmdHist.scrollForeward();
}

void LCmdMainWindow::cmdNavigateUp ( GAbstractCommand* /*cmd*/ )
{
   if (lcmd.options.panelsAreVisible)
      lcmd.curPanel->selectItem(LCmdFilePanel::NavigateUp);
   else
      subClient->cmdcont.cmdline.entry.cmdHist.scrollBackward();
}

void LCmdMainWindow::cmdNavigatePageDown ( GAbstractCommand* /*cmd*/ )
{
   if (lcmd.options.panelsAreVisible)
      lcmd.curPanel->selectItem(LCmdFilePanel::NavigatePageDown);
   else
      subClient->cmdcont.conmon.cmdNavigatePageDown();
}

void LCmdMainWindow::cmdNavigatePageUp ( GAbstractCommand* /*cmd*/ )
{
   if (lcmd.options.panelsAreVisible)
      lcmd.curPanel->selectItem(LCmdFilePanel::NavigatePageUp);
   else
      subClient->cmdcont.conmon.cmdNavigatePageUp();
}

void LCmdMainWindow::cmdNavigateHome ( GAbstractCommand* /*cmd*/ )
{
   if (lcmd.options.panelsAreVisible)
      lcmd.curPanel->selectItem(LCmdFilePanel::NavigateHome);
}

void LCmdMainWindow::cmdNavigateEnd ( GAbstractCommand* /*cmd*/ )
{
   if (lcmd.options.panelsAreVisible)
      lcmd.curPanel->selectItem(LCmdFilePanel::NavigateEnd);
}

void LCmdMainWindow::cmdActivateOppositePanel ( GAbstractCommand* /*cmd*/ )
{
   // Activate the opposite file panel
   lcmd.curPanel = &lcmd.curPanel->getOppositePanel();
   lcmd.fp1->drawItem(lcmd.fp1->getCurrentSelectedIndex());
   lcmd.fp2->drawItem(lcmd.fp2->getCurrentSelectedIndex());
   lcmd.fp1->headerWin.invalidateClientAreaForText();
   lcmd.fp2->headerWin.invalidateClientAreaForText();

   // Activate the drive and dir of the newly selected panel
   lcmd.curPanel->activatePanelDriveAndDir(true);

   ensureFocusOK();
}

void LCmdMainWindow::cmdNavigateEnter ( GAbstractCommand* /*cmd*/ )
{
   lcmd.curPanel->doEnter();
}

void LCmdMainWindow::cmdNavigateEnterSysObj ( GAbstractCommand* /*cmd*/ )
{
   lcmd.curPanel->startSelectedProgram();
}

void LCmdMainWindow::cmdNavigateRight ( GAbstractCommand* /*cmd*/ )
{
   if (lcmd.options.panelsAreVisible)
      lcmd.curPanel->selectItem(LCmdFilePanel::NavigateRight);
}

void LCmdMainWindow::cmdNavigateLeft ( GAbstractCommand* /*cmd*/ )
{
   if (lcmd.options.panelsAreVisible)
      lcmd.curPanel->selectItem(LCmdFilePanel::NavigateLeft);
}

void LCmdMainWindow::cmdNavigateNextMatch ( GAbstractCommand* /*cmd*/ )
{
   LCmdFilePanel* panel = lcmd.curPanel;
   if (panel->view.viewMode == LCmdFilePanelViewOptions::VIEWMODE_INFO)
      return;

   if (lcmd.options.panelsAreVisible && 
       panel->headerWin.isPerformingDynamicSearch())
   {
      GString text = panel->headerWin.dynaFileSearch->getText();
      int curSel = panel->getCurrentSelectedIndex();
      int idx = panel->findNextMatchingItem(text, curSel);
      if (idx >= 0 && idx != curSel)
         panel->selectItem(idx);
   }
}

void LCmdMainWindow::cmdNavigatePrevMatch ( GAbstractCommand* /*cmd*/ )
{
   LCmdFilePanel* panel = lcmd.curPanel;
   if (panel->view.viewMode == LCmdFilePanelViewOptions::VIEWMODE_INFO)
      return;

   if (lcmd.options.panelsAreVisible &&
       panel->headerWin.isPerformingDynamicSearch())
   {
      GString text = panel->headerWin.dynaFileSearch->getText();
      int curSel = panel->getCurrentSelectedIndex();
      int idx = panel->findPrevMatchingItem(text, curSel);
      if (idx >= 0 && idx != curSel)
         panel->selectItem(idx);
   }
}

void LCmdMainWindow::cmdNavigateBetwConAndCmdLine ( GAbstractCommand* /*cmd*/ )
{
   if (!lcmd.options.panelsAreVisible || !lcmd.options.panelsFullScreen)
   {
      LCmdCmdContainer& cmdcont = subClient->cmdcont; // Make this a fast one

      if (cmdcont.conmon.containsFocusedWindow())
         cmdcont.cmdline.entry.grabFocus();
      else
         cmdcont.conmon.grabFocus();
   }
}

void LCmdMainWindow::cmdToggleItemMark ( GAbstractCommand* /*cmd*/ )
{
   if (lcmd.curPanel->view.viewMode == LCmdFilePanelViewOptions::VIEWMODE_INFO)
      return;
   if (!lcmd.options.panelsAreVisible)
      return;
   LCmdFilePanel& curPanel = LCmdFilePanel::GetCurrentPanel();
   int curSel = curPanel.getCurrentSelectedIndex();
   if (curPanel.toggleTag(curSel, true))
      curPanel.selectItem(curSel);
}

void LCmdMainWindow::cmdToggleItemMarkAndMoveDown ( GAbstractCommand* /*cmd*/ )
{
   if (!lcmd.options.panelsAreVisible)
      return;

   LCmdFilePanel& curPanel = LCmdFilePanel::GetCurrentPanel();
   if (curPanel.view.viewMode == LCmdFilePanelViewOptions::VIEWMODE_INFO)
      return;

   int curSel = curPanel.getCurrentSelectedIndex();
   curPanel.toggleTag(curSel, true);

   // Force the "next" item to be selected, even if the recent item was
   // actually not tagged. This is for instance when the recent item was
   // the ".." directory. If the user press INSERT when this item is selected
   // then the selection should move down even if the ".." item it self was
   // not tagged. In addition, this will cause the item to be repainted
   // to affect the eventually new marking state of it.
   int idx = ((curSel == curPanel.items.getCount() - 1) ? curSel : curSel + 1);
   curPanel.selectItem(idx);
}

void LCmdMainWindow::cmdScrollCmdHistDown ( GAbstractCommand* /*cmd*/ )
{
   if (subClient->cmdcont.cmdline.entry.containsFocusedWindow())
      subClient->cmdcont.cmdline.entry.cmdHist.scrollForeward();
}

void LCmdMainWindow::cmdScrollCmdHistUp ( GAbstractCommand* /*cmd*/ )
{
   if (subClient->cmdcont.cmdline.entry.containsFocusedWindow())
      subClient->cmdcont.cmdline.entry.cmdHist.scrollBackward();
}

void LCmdMainWindow::cmdWalkDirectoryDown ( GAbstractCommand* /*cmd*/ )
{
   lcmd.curPanel->walkDownDir();
}

void LCmdMainWindow::cmdWalkDirectoryUp ( GAbstractCommand* /*cmd*/ )
{
   lcmd.curPanel->walkUpDir();
}

void LCmdMainWindow::cmdLeftWalkDirectoryUp ( GAbstractCommand* /*cmd*/ )
{
   lcmd.fp1->walkUpDir();
}

void LCmdMainWindow::cmdRightWalkDirectoryUp ( GAbstractCommand* /*cmd*/ )
{
   lcmd.fp2->walkUpDir();
}

void LCmdMainWindow::cmdWalkRootDirectory ( GAbstractCommand* /*cmd*/ )
{
   lcmd.curPanel->walkDir(GFile::SlashStr);
}

void LCmdMainWindow::cmdWalkLeftToRightDir ( GAbstractCommand* /*cmd*/ )
{
   GString dir = lcmd.fp1->getCurrentVfsDirectory(false);
   lcmd.fp2->walkDir(dir);
}

void LCmdMainWindow::cmdWalkRightToLeftDir ( GAbstractCommand* /*cmd*/ )
{
   GString dir = lcmd.fp2->getCurrentVfsDirectory(false);
   lcmd.fp1->walkDir(dir);
}

void LCmdMainWindow::cmdLeftWalkRootDirectory ( GAbstractCommand* /*cmd*/ )
{
   lcmd.fp1->walkDir(GFile::SlashStr);
}

void LCmdMainWindow::cmdRightWalkRootDirectory ( GAbstractCommand* /*cmd*/ )
{
   lcmd.fp2->walkDir(GFile::SlashStr);
}

void LCmdMainWindow::cmdWalkOppositeDirectory ( GAbstractCommand* /*cmd*/ )
{
   LCmdFilePanel& fp = lcmd.curPanel->getOppositePanel();
   GString dir = fp.getCurrentVfsDirectory(false);
   lcmd.curPanel->walkDir(dir);
}

void LCmdMainWindow::cmdWalkCurDirIntoOppositePanel ( GAbstractCommand* /*cmd*/ )
{
   LCmdFilePanel& fp = lcmd.curPanel->getOppositePanel();
   GString dir = lcmd.curPanel->getCurrentVfsDirectory(false);
   fp.walkDir(dir);
}

void LCmdMainWindow::cmdAutoFilenameForeward ( GAbstractCommand* /*cmd*/ )
{
   if (subClient->cmdcont.cmdline.entry.containsFocusedWindow())
      subClient->cmdcont.cmdline.entry.autoCompleteFileName(true); // true = Foreward
}

void LCmdMainWindow::cmdAutoFilenameBackward ( GAbstractCommand* /*cmd*/ )
{
   if (subClient->cmdcont.cmdline.entry.containsFocusedWindow())
      subClient->cmdcont.cmdline.entry.autoCompleteFileName(false); // false = Backward
}

void LCmdMainWindow::cmdEscape ( GAbstractCommand* /*cmd*/ )
{
   if (lcmd.curPanel->headerWin.isPerformingDynamicSearch())
      lcmd.curPanel->headerWin.dismissDynamicSearch();
   else
   if (!lcmd.cmdEntry.isEmpty())
   {
      lcmd.cmdEntry.cmdHist.setCursor(0);
      lcmd.cmdEntry.setText(GString::Empty);
   }
   else
      subClient->setFilePanelsVisible(-1);
}

void LCmdMainWindow::cmdFeedChildProcessStdIn ( GAbstractCommand* /*cmd*/ )
{
   LCmdCmdLineEntry& entry = lcmd.cmdEntry;
   GString txt = entry.getText();
   txt += GString::EOL; // "\r\n"
   if (entry.feedChildProcessStdIn(null, txt))
   {
      lcmd.cmdLine.setText(GString::Empty);
      setStatusbarText(GString::Empty);
   }
}

void LCmdMainWindow::cmdRunCommandInNewSession ( GAbstractCommand* /*cmd*/ )
{
   LCmdCmdLine& cmdline = subClient->cmdcont.cmdline; // Make this a fast one.
   if (cmdline.isVisible() || !lcmd.options.panelsAreVisible)
   {
      GString cmd = cmdline.entry.getText();
      if (!cmd.isEmpty())
      {
         int flags = LCmdCmdLineEntry::ECF_FORCE_NEW_SESSION |
                     LCmdCmdLineEntry::ECF_DO_ECHO;
         if (lcmd.options.conVarious.defaultCloseConOnExit)
            flags |= LCmdCmdLineEntry::ECF_CLOSEON_EXIT;
         LCmdCmdLineEntry& entry = cmdline.entry;
         entry.executeCommand(cmd, GString::Empty, flags);
         entry.setText("");
      }
   }
}

void LCmdMainWindow::cmdInsertItemPathInCmdLine ( GAbstractCommand* /*cmd*/ )
{
   GString fullPath = lcmd.curPanel->getCurItemPath();
   if (fullPath != "")
   {
      GString buff = GToken(fullPath).toString(true) + GString::Blank;
      subClient->cmdcont.cmdline.entry.replaceSelection(buff);
   }
}

void LCmdMainWindow::cmdInsertItemNameInCmdLine ( GAbstractCommand* /*cmd*/ )
{
   GString itemName = lcmd.curPanel->getCurItemName();
   if (itemName != "")
   {
      GString buff = GToken(itemName).toString(true) + GString::Blank;
      subClient->cmdcont.cmdline.entry.replaceSelection(buff);
   }
}

void LCmdMainWindow::cmdBreak ( GAbstractCommand* /*cmd*/ )
{
   GSystem::BreakType defaultBt = GSystem::BT_CtrlC;
   subClient->cmdcont.cmdline.entry.breakChildProg(defaultBt);
}

void LCmdMainWindow::cmdCopyCurFilePath ( GAbstractCommand* /*cmd*/ )
{
   if (lcmd.options.panelsAreVisible)
   {
      GString filePath = lcmd.curPanel->getCurItemPath();
      if (filePath != "")
         GSystem::CopyTextToClipboard(filePath);
   }
}

void LCmdMainWindow::cmdCopyCurFileName ( GAbstractCommand* /*cmd*/ )
{
   if (lcmd.options.panelsAreVisible)
   {
      GString fileName = lcmd.curPanel->getCurItemName();
      if (fileName != "")
         GSystem::CopyTextToClipboard(fileName);
   }
}

void LCmdMainWindow::cmdOppositePanelInfo ( GAbstractCommand* /*cmd*/ )
{
   LCmdFilePanel* fp;
   if (lcmd.curPanel->view.viewMode == LCmdFilePanelViewOptions::VIEWMODE_INFO)
      fp = lcmd.curPanel;
   else
      fp = &lcmd.curPanel->getOppositePanel();
   if (fp->view.viewMode == LCmdFilePanelViewOptions::VIEWMODE_INFO)
      fp->setViewMode(fp->previousViewMode);
   else
      fp->setViewMode(LCmdFilePanelViewOptions::VIEWMODE_INFO); // Toggle back to the recent mode of the INFO-panel
}

void LCmdMainWindow::cmdPullDownMenu ( GAbstractCommand* /*cmd*/ )
{
   if (lcmd.curPanel == lcmd.fp1)
      startMenubarMode("LEFTPANELMENU");
   else
      startMenubarMode("RIGHTPANELMENU");
}

void LCmdMainWindow::cmdExtendLeftPanelWidth ( GAbstractCommand* /*cmd*/ )
{
   if (lcmd.options.panel1RelWidth < 1000)
   {
      lcmd.options.panel1RelWidth += 50; // 5%
      if (lcmd.options.panel1RelWidth > 1000)
         lcmd.options.panel1RelWidth = 1000;
      subClient->panelsParent.layout();
   }
}

void LCmdMainWindow::cmdExtendRightPanelWidth ( GAbstractCommand* /*cmd*/ )
{
   if (lcmd.options.panel1RelWidth > 0)
   {
      lcmd.options.panel1RelWidth -= 50; // 5%
      if (lcmd.options.panel1RelWidth < 0)
         lcmd.options.panel1RelWidth = 0;
      subClient->panelsParent.layout();
   }
}

void LCmdMainWindow::cmdExtendPanelsHeight ( GAbstractCommand* /*cmd*/ )
{
   if (!lcmd.options.panelsAreVisible)
   {
      subClient->setFilePanelsVisible(true);
   }
   else
   if (lcmd.options.panelsFullScreen)
   {
      subClient->setFullScreenMode(false);
   }
   else
   if (lcmd.options.consoleRelHeight >= 200)
   {
      lcmd.options.consoleRelHeight -= 50; // 5%
      subClient->recalcAllPartsOfLayout();
   }
   else
   {
      subClient->setFullScreenMode(true);
   }
}

void LCmdMainWindow::cmdExtendConMonHeight ( GAbstractCommand* /*cmd*/ )
{
   if (!lcmd.options.panelsAreVisible)
   {
      subClient->setFilePanelsVisible(true);
   }
   else
   if (lcmd.options.panelsFullScreen)
   {
      subClient->setFullScreenMode(false);
   }
   else
   if (lcmd.options.consoleRelHeight <= 800)
   {
      lcmd.options.consoleRelHeight += 50; // 5%
      subClient->recalcAllPartsOfLayout();
   }
   else
   {
      subClient->setFilePanelsVisible(false);
   }
}

void LCmdMainWindow::cmdAlignPanelsWidth ( GAbstractCommand* /*cmd*/ )
{
   if (lcmd.options.panel1RelWidth != 500)
   {
      lcmd.options.panel1RelWidth = 500;
      subClient->panelsParent.layout();
   }
}

void LCmdMainWindow::cmdAppendEndOfStream ( GAbstractCommand* /*cmd*/ )
{
   LCmdCmdLineEntry& entry = lcmd.cmdEntry;
   LCmdProcessLauncher* child = entry.selectChildProcess(null, null);
   if (child == null)
      return;
   LCmdConsoleMonitor& conmon = subClient->cmdcont.conmon;
   conmon.gotoPos(conmon.getLinesCount(), 0);
   GString txt("%c", GVArgs(char(0x1A))); // Ctrl+Z (End-of-file).
   if (entry.feedChildProcessStdIn(child, txt))
      setStatusbarText(GString::Empty);
}

void LCmdMainWindow::cmdAlignPanelsHeight ( GAbstractCommand* /*cmd*/ )
{
   if (lcmd.options.consoleRelHeight != 360)
   {
      lcmd.options.consoleRelHeight = 360;
      subClient->recalcAllPartsOfLayout();
   }
}

void LCmdMainWindow::updateStatusbarChildProcessCount ()
{
   GStatusbar* sbar = getStatusbar();
   if (sbar != null && sbar->isVisible())
   {
      int count = lcmd.cmdLine.entry.getRunningChildrenCount();
      GString str(strCountChildProcesses, GVArgs(count));
      sbar->setCellText("CountChildProcesses", str);
   }
}

void LCmdMainWindow::updateStatusbarDriveFreeCell ()
{
   GStatusbar* sbar = getStatusbar();
   if (sbar != null && sbar->isVisible())
   {
      LCmdFilePanel* panel = lcmd.curPanel;
      if (lcmd.curPanel->view.viewMode == LCmdFilePanelViewOptions::VIEWMODE_INFO)
         panel = &lcmd.curPanel->getOppositePanel();
      const GVfsLocal& fs = panel->vfs.root();
      const GDriveInfo& di = fs.getCurrentDriveInfo();
      GString driveFreeStr = GStringUtil::ToByteCountString(di.diskFree);
      GString driveName = fs.getCurrentDriveName();
      GString str(strDriveFree, GVArgs(driveName).add(driveFreeStr));
      sbar->setCellText("DriveFree", str);
   }
}

void LCmdMainWindow::updateStatusbarConsoleCurrentLine ()
{
   GStatusbar* sbar = getStatusbar();
   if (sbar != null && sbar->isVisible())
   {
      int curLine;
      if (lcmd.conmon.hasFocus())
         curLine = lcmd.conmon.getCurrentPosY() + 1;
      else
         curLine = lcmd.conmon.getIndexOfFirstVisibleLine() + 1;
      GString str(strConLineCurrent, GVArgs(curLine));
      sbar->setCellText("ConLineCurrent", str);
   }
}

void LCmdMainWindow::updateStatusbarConsoleLineCount ()
{
   GStatusbar* sbar = getStatusbar();
   if (sbar != null && sbar->isVisible())
   {
      int lineCount = lcmd.conmon.getLinesCount() + 1;
      GString str(strConLineCount, GVArgs(lineCount));
      sbar->setCellText("ConLineCount", str);
   }
}

void LCmdMainWindow::updateStatusbarDirCacheCount ()
{
   GStatusbar* sbar = getStatusbar();
   if (sbar != null && sbar->isVisible())
   {
      int count = lcmd.dirCache->dirs.getCount();
      GString str(strDirCacheCount, GVArgs(count));
      sbar->setCellText("DirCacheCount", str);
   }
}

bool LCmdMainWindow::onTimer ( const GString& timerID, GObject* userData )
{
   if (timerID != "INFOUPDATE")
      return GFrameWindow::onTimer(timerID, userData);

   // Update our shared system information container.
   GSystemInfo& si = lcmd.sysInfo; // Make this a fast one.
   si.update(); 

   // Update statusbar items.
   GStatusbar* sbar = getStatusbar();
   if (sbar != null && sbar->isVisible())
   {
      updateStatusbarChildProcessCount();
      updateStatusbarConsoleCurrentLine();
      updateStatusbarConsoleLineCount();
      sbar->setCellText("PhysMemFree", GString(strPhysMemFree, GVArgs(GStringUtil::ToByteCountString(si.getFreePhysicalRAM()))));
      sbar->setCellText("SysCountProcesses", GString(strSysCountProcesses, GVArgs(si.getProcessCount())));
      sbar->setCellText("SysCountThreads", GString(strSysCountThreads, GVArgs(si.getThreadCount())));
      updateStatusbarDriveFreeCell();
      updateStatusbarDirCacheCount();
   }

   // Update items in the INFO-Panel (if any).
   lcmd.fp1->info.updateUpon1Hz();
   lcmd.fp2->info.updateUpon1Hz();
   return true;
}

bool LCmdMainWindow::onClose ()
{
   cmdCloseAndExit(null);
   return true;
}

bool LCmdMainWindow::onKeyDown ( const GKeyMessage& key )
{
   char chr = key.getCharacter();
   if (chr != '\0')
   {
      // A character key with or without the combination of any shift-key(s).
      if (key.isAltDown() && !key.isShiftDown() && !key.isCtrlDown())
      {
         return lcmd.curPanel->startDynamicSearch(chr);
      }
      else
      if (!key.isAltDown() && !key.isShiftDown() && !key.isCtrlDown())
      {
         // Handle Standard Keyboard Characters.
         if (!lcmd.cmdCont.cmdline.isVisible() && 
             !lcmd.curPanel->headerWin.dynaFileSearch->isVisible())
         {
            return lcmd.curPanel->findItemByKey(chr);
         }
      }
   }
   else
   {
      GKey::KeyCode keyCode = key.getCode();
      if (keyCode == GKey::KEY_DELETE &&
         lcmd.options.fileDel.useDelKey && 
         !lcmd.cmdLine.isVisible())
      {
         cmdDeleteFile();
         return true;
      }
   }

   return GFrameWindow::onKeyDown(key);
}

bool LCmdMainWindow::CheckSortOpt ( LCmdFilePanel& p, LCmdFilePanelSortOptions::SORT_WHAT what )
{
   return p.sortOpt.what[0] == what ||
          p.sortOpt.what[0] == LCmdFilePanelSortOptions::PSW_TYPE &&
          p.sortOpt.what[1] == what;
}

void LCmdMainWindow::updateCommandStates ()
{
   setCommandToggleState("cmdLeftModeBrief",       lcmd.fp1->view.viewMode == LCmdFilePanelViewOptions::VIEWMODE_BRIEF);
   setCommandToggleState("cmdLeftModeWide",        lcmd.fp1->view.viewMode == LCmdFilePanelViewOptions::VIEWMODE_WIDE);
   setCommandToggleState("cmdLeftModeFull",        lcmd.fp1->view.viewMode == LCmdFilePanelViewOptions::VIEWMODE_FULL);
   setCommandToggleState("cmdLeftModeInfo",        lcmd.fp1->view.viewMode == LCmdFilePanelViewOptions::VIEWMODE_INFO);
#if __ENABLE_TREE_VIEW
   setCommandToggleState("cmdLeftModeTree",        lcmd.fp1->view.viewMode == LCmdFilePanelViewOptions::VIEWMODE_TREE);
#endif
   setCommandToggleState("cmdLeftCycleIconTypes",  lcmd.fp1->view.showItemIcon);

   setCommandToggleState("cmdRightModeBrief",      lcmd.fp2->view.viewMode == LCmdFilePanelViewOptions::VIEWMODE_BRIEF);
   setCommandToggleState("cmdRightModeWide",       lcmd.fp2->view.viewMode == LCmdFilePanelViewOptions::VIEWMODE_WIDE);
   setCommandToggleState("cmdRightModeFull",       lcmd.fp2->view.viewMode == LCmdFilePanelViewOptions::VIEWMODE_FULL);
   setCommandToggleState("cmdRightModeInfo",       lcmd.fp2->view.viewMode == LCmdFilePanelViewOptions::VIEWMODE_INFO);
#if __ENABLE_TREE_VIEW
   setCommandToggleState("cmdRightModeTree",       lcmd.fp2->view.viewMode == LCmdFilePanelViewOptions::VIEWMODE_TREE);
#endif
   setCommandToggleState("cmdRightCycleIconTypes", lcmd.fp2->view.showItemIcon);

   setCommandToggleState("cmdLeftSortByTime",       CheckSortOpt(*lcmd.fp1, LCmdFilePanelSortOptions::PSW_TIME));
   setCommandToggleState("cmdLeftSortByName",       CheckSortOpt(*lcmd.fp1, LCmdFilePanelSortOptions::PSW_NAME));
   setCommandToggleState("cmdLeftSortByExtension",  CheckSortOpt(*lcmd.fp1, LCmdFilePanelSortOptions::PSW_EXTENTION));
   setCommandToggleState("cmdLeftSortByDate",       CheckSortOpt(*lcmd.fp1, LCmdFilePanelSortOptions::PSW_DATE));
   setCommandToggleState("cmdLeftSortBySize",       CheckSortOpt(*lcmd.fp1, LCmdFilePanelSortOptions::PSW_SIZE));
   setCommandToggleState("cmdLeftSortUnsorted",     CheckSortOpt(*lcmd.fp1, LCmdFilePanelSortOptions::PSW_UNSORTED));

   setCommandToggleState("cmdRightSortByTime",      CheckSortOpt(*lcmd.fp2, LCmdFilePanelSortOptions::PSW_TIME));
   setCommandToggleState("cmdRightSortByName",      CheckSortOpt(*lcmd.fp2, LCmdFilePanelSortOptions::PSW_NAME));
   setCommandToggleState("cmdRightSortByExtension", CheckSortOpt(*lcmd.fp2, LCmdFilePanelSortOptions::PSW_EXTENTION));
   setCommandToggleState("cmdRightSortByDate",      CheckSortOpt(*lcmd.fp2, LCmdFilePanelSortOptions::PSW_DATE));
   setCommandToggleState("cmdRightSortBySize",      CheckSortOpt(*lcmd.fp2, LCmdFilePanelSortOptions::PSW_SIZE));
   setCommandToggleState("cmdRightSortUnsorted",    CheckSortOpt(*lcmd.fp2, LCmdFilePanelSortOptions::PSW_UNSORTED));

   setCommandToggleState("cmdToggleToolbar",        isToolbarVisible());
   setCommandToggleState("cmdToggleKeybar",         isKeybarVisible());
   setCommandToggleState("cmdToggleStatusbar",      isStatusbarVisible());
   setCommandToggleState("cmdToggleCommandLine",    subClient->cmdcont.cmdline.isVisible());
   setCommandToggleState("cmdToggleFullScreen",     lcmd.options.panelsFullScreen);
   setCommandToggleState("cmdToggleHiddenFiles",    lcmd.curPanel->isShowHiddenFiles());
   setCommandToggleState("cmdToggleSaveOptionsOnExit", lcmd.options.saveOnExit.everything);

   // Disable commands that aren't implemented yet.
   setCommandEnableState("cmdProcessFilesByCommand", false);
   setCommandEnableState("cmdHelpContents", false);
   setCommandEnableState("cmdHelpIndex", false);
   setCommandEnableState("cmdHelpKeyboard", false);
}

bool LCmdMainWindow::onActivate ()
{
   ensureFocusOK();
   return true;
}

bool LCmdMainWindow::onUserMessage ( GUserMessage& msg )
{
   GString id = msg.getParam1String();
   if (id == "EXIT_TEXTVIEWER")
   {
      GObject* p2 = msg.getParam2();
      LCmdInternalViewer* v = dynamic_cast<LCmdInternalViewer*>(p2);
      closeAndDestroyTextViewer(v);
      return true;
   }
   else
   {
      return GFrameWindow::onUserMessage(msg);
   }
}

void LCmdMainWindow::closeAndDestroyTextViewer ( LCmdInternalViewer* v )
{
   int idx = lcmd.openViewers.indexOf(*v);
   if (idx >= 0)
   {
      GString sectionName = v->getName();
      v->writeProfile(sectionName);
      // Destroy the text viewer object.
      // Its destructor will remove it self from LCmd's list of 
      // text viewer instances ("lcmd.openViewers").
      delete v;
   }
}

bool LCmdMainWindow::executeAbstractCommand ( GAbstractCommand *cmd )
{
   if (cmd == null)
      return false;

   const GString& id = cmd->getIDString();
   if (id != "cmdNavigateNextMatch" &&
       id != "cmdNavigatePrevMatch" &&
       id != "cmdEscape")
   {
      if (lcmd.curPanel->headerWin.isPerformingDynamicSearch())
         lcmd.curPanel->headerWin.dismissDynamicSearch();
   }

   return GFrameWindow::executeAbstractCommand(cmd);
}

void LCmdMainWindow::grabFocus ( bool force )
{
   GFrameWindow::grabFocus(force);
   ensureFocusOK();
}

void LCmdMainWindow::ensureFocusOK ()
{
   if (!lcmd.finishedStartup)
      return;
   else
   if (lcmd.cmdLine.isVisible() || !lcmd.options.panelsAreVisible)
      lcmd.cmdLine.entry.grabFocus();
   else
      lcmd.curPanel->grabFocus();
}

void LCmdMainWindow::cmdTest ( GAbstractCommand* /*cmd*/ )
{
}
