/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __LCMD_INFOPANEL
#define __LCMD_INFOPANEL

#include "glib/sys/GSystemInfo.h"
#include "glib/gui/GGraphics.h"

/**
 * @author  Leif Erik Larsen
 * @since   1998.09.22
 */
class LCmdInfoPanel : public GObject
{
   public:

      /**
       * This is the base class of all items that can be contaned
       * in the information panel.
       *
       * @author  Leif Erik Larsen
       * @since   2001.03.21
       */
      class AbstractItem : public GObject
      {
         public:

            enum UpdateType
            {
               /** Item will never need to be updated. */
               NEVER = 0,

               /** Item needs to be updated constantly. */
               UPON_1HZ,

               /** Whenever Opposite File Panel rereads. */
               UPON_REREAD,

               /** Whenever the size of a directory structure has been counted. */
               UPON_CALC_SIZE,

               /** Whenever the current selected filename item has changed. */
               UPON_SELECTION,
            };

            int ypos;
            UpdateType updt;

         protected:

            int txtColor;
            int txtBoldColor;
            class LCmdInfoPanel& info;

         public:

            AbstractItem ( class LCmdInfoPanel& info, UpdateType updt );
            virtual ~AbstractItem ();

         public:

            int getHeight ();

            int getWidthOfString ( const GString& str1 = GString::Empty,
                                   const GString& str2 = GString::Empty,
                                   const GString& str3 = GString::Empty,
                                   const GString& str4 = GString::Empty );

            void paintText ( GGraphics& g,
                             const GString& str1 = GString::Empty, const GColor& c1 = GColor::BLACK,
                             const GString& str2 = GString::Empty, const GColor& c2 = GColor::BLACK,
                             const GString& str3 = GString::Empty, const GColor& c3 = GColor::BLACK,
                             const GString& str4 = GString::Empty, const GColor& c4 = GColor::BLACK );

            void paint (); // Will just call <i>paint(GGraphics&)</i>.

            virtual bool update ();
            virtual int getWidth () = 0;
            virtual void paint ( GGraphics& g ) = 0;
            virtual GString getItemString ();
      };

      /** Item used to separate other items. */
      class Separator : public AbstractItem
      {
         friend class LCmdInfoPanel;

         GStringl str1;

         Separator ( class LCmdInfoPanel& info );
         virtual int getWidth ();
         virtual void paint ( class GGraphics& g );
      };

      /** LCmdVersion. */
      class LCmdVersion : public AbstractItem
      {
         friend class LCmdInfoPanel;

         GStringl str1;

         LCmdVersion ( class LCmdInfoPanel& info );
         virtual int getWidth ();
         virtual GString getItemString ();
         virtual void paint ( class GGraphics& g );
      };

      /** LCmdCopyright. */
      class LCmdCopyright : public AbstractItem
      {
         friend class LCmdInfoPanel;

         GStringl str1;

         LCmdCopyright ( class LCmdInfoPanel& info );
         virtual int getWidth ();
         virtual GString getItemString ();
         virtual void paint ( class GGraphics& g );
      };

      /** TotalPhysMem. */
      class TotalPhysMem : public AbstractItem
      {
         friend class LCmdInfoPanel;

         GString str1;
         GStringl str2;
         ulonglong totalPhysRam;

         TotalPhysMem ( class LCmdInfoPanel& info );
         virtual int getWidth ();
         virtual bool update ();
         virtual GString getItemString ();
         virtual void paint ( class GGraphics& g );
      };

      /** FreePhysMem. */
      class FreePhysMem : public AbstractItem
      {
         friend class LCmdInfoPanel;

         GString str1;
         GStringl str2;
         ulonglong freePhysRam;

         FreePhysMem ( class LCmdInfoPanel& info );
         virtual int getWidth ();
         virtual bool update ();
         virtual GString getItemString ();
         virtual void paint ( class GGraphics& g );
      };

      /** TotalBytesOnDrive. */
      class TotalBytesOnDrive : public AbstractItem
      {
         friend class LCmdInfoPanel;

         GString str1;
         GStringl str2;
         GString str3;

         TotalBytesOnDrive ( class LCmdInfoPanel& info );
         virtual int getWidth ();
         virtual bool update ();
         virtual GString getItemString ();
         virtual void paint ( class GGraphics& g );
      };

      /** BytesFreeOnDrive. */
      class BytesFreeOnDrive : public AbstractItem
      {
         friend class LCmdInfoPanel;

         GString str1;
         GStringl str2;
         GString str3;
         longlong diskFree;

         BytesFreeOnDrive ( class LCmdInfoPanel& info );
         virtual int getWidth ();
         virtual bool update ();
         virtual GString getItemString ();
         virtual void paint ( class GGraphics& g );
      };

      /** PanelFilesStatistic. */
      class PanelFilesStatistic : public AbstractItem
      {
         friend class LCmdInfoPanel;

         GString str1;
         GStringl str2;
         GString str3;
         GStringl str4;

         PanelFilesStatistic ( class LCmdInfoPanel& info );
         virtual int getWidth ();
         virtual bool update ();
         virtual GString getItemString ();
         virtual void paint ( class GGraphics& g );
      };

      /** CurrentDir. */
      class CurrentDir : public AbstractItem
      {
         friend class LCmdInfoPanel;

         GString str1;

         CurrentDir ( class LCmdInfoPanel& info );
         virtual int getWidth ();
         virtual bool update ();
         virtual GString getItemString ();
         virtual void paint ( class GGraphics& g );
      };

      /** CurrentFileName. */
      class CurrentFileName : public AbstractItem
      {
         friend class LCmdInfoPanel;

         GString str1;

         CurrentFileName ( class LCmdInfoPanel& info );
         virtual int getWidth ();
         virtual bool update ();
         virtual void paint ( class GGraphics& g );
      };

      /** CurrentFileAttributes. */
      class CurrentFileAttributes : public AbstractItem
      {
         friend class LCmdInfoPanel;

         GStringl str1;
         GString str2;

         CurrentFileAttributes ( class LCmdInfoPanel& info );
         virtual int getWidth ();
         virtual bool update ();
         virtual void paint ( class GGraphics& g );
      };

      /** CurrentFileAttributes. */
      class CurrentFileCreated : public AbstractItem
      {
         friend class LCmdInfoPanel;

         GStringl str1;
         GString str2;

         CurrentFileCreated ( class LCmdInfoPanel& info );
         virtual int getWidth ();
         virtual bool update ();
         virtual void paint ( class GGraphics& g );
      };

      /** CurrentFileUpdated. */
      class CurrentFileUpdated : public AbstractItem
      {
         friend class LCmdInfoPanel;

         GStringl str1;
         GString str2;

         CurrentFileUpdated ( class LCmdInfoPanel& info );
         virtual int getWidth ();
         virtual bool update ();
         virtual void paint ( class GGraphics& g );
      };

      /** CurrentFileAccessed. */
      class CurrentFileAccessed : public AbstractItem
      {
         friend class LCmdInfoPanel;

         GStringl str1;
         GString str2;

         CurrentFileAccessed ( class LCmdInfoPanel& info );
         virtual int getWidth ();
         virtual bool update ();
         virtual void paint ( class GGraphics& g );
      };

      /** CurrentFileSize. */
      class CurrentFileSize : public AbstractItem
      {
         friend class LCmdInfoPanel;

         GStringl str1;
         GString str2;

         CurrentFileSize ( class LCmdInfoPanel& info );
         virtual int getWidth ();
         virtual bool update ();
         virtual void paint ( class GGraphics& g );
      };

      /** DriveLabel. */
      class DriveLabel : public AbstractItem
      {
         friend class LCmdInfoPanel;

         GStringl str1;
         GString str2;

         DriveLabel ( class LCmdInfoPanel& info );
         virtual int getWidth ();
         virtual bool update ();
         virtual GString getItemString ();
         virtual void paint ( class GGraphics& g );
      };

      /** DriveFileSystem. */
      class DriveFileSystem : public AbstractItem
      {
         friend class LCmdInfoPanel;

         GStringl str1;
         GString str2;

         DriveFileSystem ( class LCmdInfoPanel& info );
         virtual int getWidth ();
         virtual bool update ();
         virtual GString getItemString ();
         virtual void paint ( class GGraphics& g );
      };

      /** DriveSerialNumber. */
      class DriveSerialNumber : public AbstractItem
      {
         friend class LCmdInfoPanel;

         GStringl str1;
         GString str2;

         DriveSerialNumber ( class LCmdInfoPanel& info );
         virtual int getWidth ();
         virtual bool update ();
         virtual GString getItemString ();
         virtual void paint ( class GGraphics& g );
      };

      /** Uptime. */
      class Uptime : public AbstractItem
      {
         friend class LCmdInfoPanel;

         GStringl str1;
         GString str2;
         GTimeAmount uptime;

         Uptime ( class LCmdInfoPanel& info );
         virtual int getWidth ();
         virtual bool update ();
         virtual GString getItemString ();
         virtual void paint ( class GGraphics& g );
      };

      /** NumberOfProcesses. */
      class NumberOfProcesses : public AbstractItem
      {
         friend class LCmdInfoPanel;

         GStringl str1;
         GString str2;
         int numProcesses;

         NumberOfProcesses ( class LCmdInfoPanel& info );
         virtual int getWidth ();
         virtual bool update ();
         virtual GString getItemString ();
         virtual void paint ( class GGraphics& g );
      };

      /** NumberOfThreads. */
      class NumberOfThreads : public AbstractItem
      {
         friend class LCmdInfoPanel;

         GStringl str1;
         GString str2;
         int numThreads;

         NumberOfThreads ( class LCmdInfoPanel& info );
         virtual int getWidth ();
         virtual bool update ();
         virtual GString getItemString ();
         virtual void paint ( class GGraphics& g );
      };

      /** SwapFileSize. */
      class SwapFileSize : public AbstractItem
      {
         friend class LCmdInfoPanel;

         GStringl str1;
         GString str2;
         longlong swapSize;

         SwapFileSize ( class LCmdInfoPanel& info );
         virtual int getWidth ();
         virtual bool update ();
         virtual GString getItemString ();
         virtual void paint ( class GGraphics& g );
      };

      /** FreeSwapSpace. */
      class FreeSwapSpace : public AbstractItem
      {
         friend class LCmdInfoPanel;

         GStringl str1;
         GString str2;
         longlong swapFree;

         FreeSwapSpace ( class LCmdInfoPanel& info );
         virtual int getWidth ();
         virtual bool update ();
         virtual GString getItemString ();
         virtual void paint ( class GGraphics& g );
      };

   public:

      /** The Panel of where this LCmdInfoPanel object is contained. */
      class LCmdFilePanel& ownerPanel;

      /** */
      ulonglong minFree;

      /** The fully qualified path of the system dependent swap-file. */
      GString swapPath;

      /** The dynamic array of information items. */
      GArray<AbstractItem> items;

      /** X-Pos of the left margine with respect to scrollbar. */
      int xpos0;

      /** Y-Pos of the uppermost item with respect to scrollbar. */
      int ypos0;

   public:

      explicit LCmdInfoPanel ( class LCmdFilePanel& filePanel );
      virtual ~LCmdInfoPanel ();

   public:

      void paint ( class GGraphics& g, const class GRectangle* rect = null );
      int sumHeightOfItems ();
      void updateItemRects ();

      /**
       * Will be called whenever another filename item has been selected.
       */
      void updateUponSelection ();

      /**
       * Will be constantly called at 1Hz when the Info-Mode active.
       */
      void updateUpon1Hz ();

      /**
       * Will be called whenever the Opposite File Panel has been reread.
       */
      void updateUponReread ();

      /**
       * Will be called whenever a directory structure size has been counted.
       */
      void updateUponCalcSize ();

      int widthOfWidestItem ();

   private:

      static GString QuerySwapPath ( ulonglong* minFree );
};

#endif // #ifndef __LCMD_INFOPANEL
