/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "lcmd/LCmdFilePanelObjRenderer.h"
#include "lcmd/LCmdCmdLineEntry.h"
#include "glib/gui/dragndrop/GDragInfo.h"

LCmdFilePanelObjRenderer::LCmdFilePanelObjRenderer ( LCmdFilePanel& ownerPanel)
                         :GThread("LCmdFilePanel-Obj-Renderer", 65536),
                          ownerPanel(ownerPanel),
                          hWndObj(null)
{
   start();
}

LCmdFilePanelObjRenderer::~LCmdFilePanelObjRenderer ()
{
   if (isRunning() && hWndObj != null)
   {
      WinPostMsg(hWndObj, WM_QUIT, 0, 0); // Break the object window message loop.
      while (isRunning()) // Wait for the thread to terminate.
         GThread::Sleep(50);
   }
}

DRAGTRANSFER* LCmdFilePanelObjRenderer::makeXfer ( DRAGITEM* dragItem, USHORT usOp, LCmdFileItem* destItem )
{
   DRAGTRANSFER* xfer = DrgAllocDragtransfer(1);
   if (xfer != null)
   {
      memset(xfer, 0, sizeof(*xfer));
      xfer->cb = sizeof(*xfer);
      xfer->hwndClient = ownerPanel.getHWND();
      xfer->pditem = dragItem;
      xfer->hstrSelectedRMF = ::DrgAddStrHandle("<DRM_OS2FILE,DRF_UNKNOWN>");
      int len = ::DrgQueryStrNameLen(dragItem->hstrSourceName) + 1;
      char* src = new char[len];
      ::DrgQueryStrName(dragItem->hstrSourceName, len, src);
      GString dst = ownerPanel.getCurrentSysDirectory(true);
      if (destItem != null)
      {
         // The item has been drop on an item in the file panel.
         // If the item is a directory then we should drop the object
         // into that directory. If it is a program file then we should
         // launch the program with the dropped file path as an argument.
         if (destItem->isDirectory() && !destItem->isUpDir())
         {
            GString dirName = destItem->getFileName();
            dst += (dirName + "\\" + src);
         }
         else
         if (destItem->isProgramFileName())
         {
            dst += destItem->getFileName();
         }
         else
         {
            dst += src;
         }
      }
      else
      {
         dst += src;
      }
      xfer->hstrRenderToName = ::DrgAddStrHandle(dst.cstring());
      delete src;
      xfer->ulTargetInfo = 0;
      xfer->usOperation = usOp;

   }
   return xfer;
}

void LCmdFilePanelObjRenderer::killXfer ( DRAGTRANSFER* xfer )
{
   if (xfer->hstrSelectedRMF)
      ::DrgDeleteStrHandle(xfer->hstrSelectedRMF);
   if (xfer->hstrRenderToName)
      ::DrgDeleteStrHandle(xfer->hstrRenderToName);
   ::DrgFreeDragtransfer(xfer);
}

void LCmdFilePanelObjRenderer::doDrop ( GDragInfo& dragInfo )
{
   DRAGINFO& di = dragInfo.di;
   LCmdFilePanelModeAbstract& curView = ownerPanel.getCurrentView();
   int destItemIndex = curView.calcItemIdxFromPos(di.xDrop, di.yDrop);
   LCmdFileItem* destItem = null;
   if (destItemIndex >= 0)
      destItem = &ownerPanel.items.get(destItemIndex);

   GArray<LCmdCopyFileItem>* dragFileItems = new GArray<LCmdCopyFileItem>(di.cditem);
   for (int c1=0; c1<di.cditem; c1++)
   {
      DRAGITEM* dragItem = ::DrgQueryDragitemPtr(&di, c1);
      DRAGTRANSFER* xfer = makeXfer(dragItem, di.usOperation, destItem);
      HWND hWndRender = di.hwndSource;

      /*
      if (dragItem->fsControl & DC_PREPARE)
      {
         if (!DrgSendTransferMsg(di.hwndSource, DM_RENDERPREPARE, MPFROMP(xfer), 0))
         {
            killXfer(xfer);
            continue;
         }
         hWndRender = dragItem->hwndItem;
      }

      if (DrgSendTransferMsg(hWndRender, DM_RENDER, MPFROMP(xfer), 0))
         continue;
      */

      /*
      if (xfer->fsReply & DMFL_RENDERRETRY)
         ; // No different ops supported - try target render
         */

      /* --------------------------------------------------------------------------
       * The source must also have provided the directory and file name -
       * retreive the length of these, then the strings into allocated memory
       */
      bool ok = false;
      if (dragItem->hstrContainerName && dragItem->hstrSourceName)
      {
         // Do the source directory name
         int len1 = ::DrgQueryStrNameLen(dragItem->hstrContainerName) + 1;
         char *pszContainer = new char[len1];
         ::DrgQueryStrName(dragItem->hstrContainerName, len1, pszContainer);

         // Do the source file name
         int len2 = DrgQueryStrNameLen(dragItem->hstrSourceName) + 1;
         char *pszSource = new char[len2];
         ::DrgQueryStrName(dragItem->hstrSourceName, len2, pszSource);

         // Now, combine the two and free the individual strings
         int len3 = len1 + len2 - 1;
         char *pszSourcePath = new char[len3];
         sprintf(pszSourcePath, "%s%s", pszContainer, pszSource);
         delete pszContainer;
         delete pszSource;

         // Now, get the target file name
         int len4 = DrgQueryStrNameLen(xfer->hstrRenderToName) + 1;
         char *pszTargetPath = new char[len4];
         ::DrgQueryStrName(xfer->hstrRenderToName, len4, pszTargetPath);

         // Add the source path and the target name to the working array
         GString srcPath(pszSourcePath);
         GString trgtPath(pszTargetPath);
         dragFileItems->add(new LCmdCopyFileItem(srcPath, &trgtPath));

         // Free the memory allocated for the file names
         delete pszSourcePath;
         delete pszTargetPath;
         ok = true;
      }

      if (ok)
         ::DrgSendTransferMsg(hWndRender, DM_ENDCONVERSATION, MPFROMLONG(dragItem->ulItemID), MPFROMLONG(DMFL_TARGETSUCCESSFUL));
      else
         ::DrgSendTransferMsg(hWndRender, DM_ENDCONVERSATION, MPFROMLONG(dragItem->ulItemID), MPFROMLONG(DMFL_TARGETFAIL));

      killXfer(xfer);
   }

   if (destItem != null && destItem->isProgramFileName())
   {
      // Launch one instance of the program file for each source file.
      char* cmd = new char[1024]; // Allocate a buff on the heap rather than consume much stack space
      int num = dragFileItems->getCount();
      for (int i=0; i<num; i++)
      {
         const GString& progPath = dragFileItems->get(i).getDestPath();
         const GString& progArg = dragFileItems->get(i).getSourcePath();
         const char* progPathQuote = (progPath.indexOf(' ') <= -1 ? "" : "\"");
         const char* progArgQuote = (progArg.indexOf(' ') <= -1 ? "" : "\"");
         sprintf(cmd, "%s%s%s %s%s%s", progPathQuote, progPath.cstring(), progPathQuote, progArgQuote, progArg.cstring(), progArgQuote);

         int flags = LCmdCmdLineEntry::ECF_FORCE_NEW_SESSION |
                     LCmdCmdLineEntry::ECF_CLOSEON_EXIT;
         LCmdCmdLineEntry& entry = LCmdCmdLineEntry::GetCmdLineEntry();
         entry.executeCommand(cmd, GString::Empty, flags);

         // Wait a few milliseconds before attempting to launch the next program
         // instanc. Just to give the OS a chance to not stress too much in case
         // the user has dropped a huge number of files on a program object.
         if (i < num - 1)
            DosSleep(25);
      }
      delete cmd;
      delete dragFileItems;
   }
   else
   {
      DROPOPTYPE opType;
      if (di.usOperation == DO_COPY)
         opType = DROPOPTYPE_COPY;
      else
         opType = DROPOPTYPE_MOVE;
      DropInfo dropInfo(opType, dragFileItems);
      GUserMessage umsg("UM_DROPFILES", &dropInfo);
      sendGuiUserMessage(ownerPanel, umsg);
   }

   DrgDeleteDraginfoStrHandles(&di);
   DrgFreeDraginfo(&di);
}

MRESULT APIENTRY LCmdFilePanelObjRenderer::ObjectWndProc ( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 )
{
   switch (msg)
   {
      /*
      case DM_ENDCONVERSATION:
         return dm_endconversation(hwnd, mp1, mp2);

      case DM_RENDER:
         return dm_render(hwnd, mp1, mp2);

      case DM_RENDERCOMPLETE:
         return dm_rendercomplete(hwnd, mp1, mp2);

      case LAZY_FILLLIST:
         return lazy_filllist(hwnd, mp1, mp2);

      case LAZY_DIRFROMINI:
         return lazy_dirfromini(hwnd, mp1, mp2);

      case LAZY_DODROP:
         return lazy_dodrop(hwnd, mp1, mp2);

      case LAZY_DORENDER:
         return lazy_dorender(hwnd, mp1, mp2);

      case LAZY_INIFROMDIR:
         return lazy_inifromdir(hwnd, mp1, mp2);

      case LAZY_DISCARDOBJ:
         return lazy_discardobj(hwnd, mp1, mp2);
         */

      case UM_OBJRENDERER_DODROP:
      {
         LCmdFilePanelObjRenderer* objRend = (LCmdFilePanelObjRenderer*) mp1;
         GDragInfo* di = (GDragInfo*) mp2;
         objRend->doDrop(*di);
         return 0;
      }

      default:
         return WinDefWindowProc(hwnd, msg, mp1, mp2);
   }
}

void LCmdFilePanelObjRenderer::postDropOperation ( GDragInfo& di )
{
   ::WinPostMsg(hWndObj, UM_OBJRENDERER_DODROP, MPFROMP(this), MPFROMP(&di));
}

void LCmdFilePanelObjRenderer::run ()
{
   QMSG qmsg;
   char szObjectClassName[] = "FilePanelObjRendererClass";

   for (;;)
   {
      // --------------------------------------------------------------------------
      // Register the window class for object window
      // --------------------------------------------------------------------------
      if (!::WinRegisterClass(hAB_Thread, szObjectClassName, LCmdFilePanelObjRenderer::ObjectWndProc, 0, 0))
      {
         // Failed!
         GSystem::Beep(60, 100);
         break;
      }

      // --------------------------------------------------------------------------
      // Finally, now the window can be created
      // --------------------------------------------------------------------------
      hWndObj = ::WinCreateWindow(HWND_OBJECT,
                                  szObjectClassName,
                                  NULL,
                                  0, 0, 0, 0, 0,
                                  null,
                                  HWND_BOTTOM,
                                  0, null, null);

      if (hWndObj == (HWND) null)
      {
         // No window - gotta leave.
         GSystem::Beep(60, 100);
         break;
      }

      // --------------------------------------------------------------------------
      while (::WinGetMsg(hAB_Thread, &qmsg, null, 0, 0))
         ::WinDispatchMsg(hAB_Thread, &qmsg);

      break;
   }

   if (hWndObj != null)
      ::WinDestroyWindow(hWndObj);
}
