/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "lcmd/LCmdFilePanelModeInfo.h"
#include "lcmd/LCmdFilePanel.h"
#include "lcmd/LCmdFilePanelFrame.h"

LCmdFilePanelModeInfo::LCmdFilePanelModeInfo ( LCmdFilePanel& fpanel,
                                               const GString& winName )
                      :GDecoratedWindow(winName,
                                        GString::Empty, 
                                        &fpanel.tabs,
                                        GString::Empty,
                                        WS_VISIBLE,
                                        WS2_OS2Y),
                       LCmdFilePanelModeAbstract(fpanel),
                       fpanel(fpanel),
                       curSelectedItem(-1) // Initially no selected item.
{
   fpanel.items.addFileItemContainerListener(this);

   setHScrollVisible(true);
   setVScrollVisible(true);
}

LCmdFilePanelModeInfo::~LCmdFilePanelModeInfo ()
{
}

void LCmdFilePanelModeInfo::invalidateAll ( bool inclChildren ) const
{
   GDecoratedWindow::invalidateAll(inclChildren);
}

void LCmdFilePanelModeInfo::invalidateRect ( const GRectangle& rect ) const
{
   GDecoratedWindow::invalidateRect(rect);
}

void LCmdFilePanelModeInfo::onViewHasBeenActivated ()
{
   // Both panels can't be in INFO mode at the same time, so toggle the
   // view mode of the opposite panel.
   LCmdFilePanel& oppositePanel = fpanel.getOppositePanel();
   if (oppositePanel.view.viewMode == LCmdFilePanelViewOptions::VIEWMODE_INFO)
      oppositePanel.setViewMode(oppositePanel.previousViewMode);

   layout();
   fpanel.frameWin.layout();
   fpanel.frameWin.headerWin.layout();
   updateScrollBarRange(); // Will call updateScrollBarPos() as well
   fpanel.info.updateUponReread();
}

void LCmdFilePanelModeInfo::allItemsHaveBeenRemoved ()
{
   curSelectedItem = -1;
}

bool LCmdFilePanelModeInfo::isHorizontallyScrollable () const
{
   return GDecoratedWindow::isHScrollbarEnabled();
}

int LCmdFilePanelModeInfo::calcItemIdxFromPos ( int xpos, int ypos ) const
{
   return -1;
}

bool LCmdFilePanelModeInfo::calcItemRect ( int /*itemIndex*/, GRectangle& rect ) const
{
   rect.clear();
   return false;
}

int LCmdFilePanelModeInfo::getCurrentSelectedIndex () const
{
   return curSelectedItem;
}

int LCmdFilePanelModeInfo::getFirstVisibleIndex () const
{
   return -1;
}

void LCmdFilePanelModeInfo::drawItem ( int /*itemIndex*/ )
{
   // Makes no sense on this file panel type, since it contains no file items.
}

void LCmdFilePanelModeInfo::drawItem ( int /*itemIndex*/, class GGraphics& /*g*/, const class GRectangle& /*itemRect*/, bool /*isDragOver*/ )
{
   // Makes no sense on this file panel type, since it contains no file items.
}

int LCmdFilePanelModeInfo::navigateDown ()
{
   onVScrollLineDown();
   return -1;
}

int LCmdFilePanelModeInfo::navigateUp ()
{
   onVScrollLineUp();
   return -1;
}

int LCmdFilePanelModeInfo::navigateEnd ()
{
   int sumh = fpanel.info.sumHeightOfItems();
   GDimension dim = getWindowSize();
   int pos = sumh - dim.height;
   if (pos > 0)
      onVScrollSliderTrack(pos);
   return -1;
}

int LCmdFilePanelModeInfo::navigateHome ()
{
   onVScrollSliderTrack(0);
   return -1;
}

int LCmdFilePanelModeInfo::navigateLeft ()
{
   onHScrollLineUp();
   return -1;
}

int LCmdFilePanelModeInfo::navigateRight ()
{
   onHScrollLineDown();
   return -1;
}

int LCmdFilePanelModeInfo::navigatePageDown ()
{
   onVScrollPageDown();
   return -1;
}

int LCmdFilePanelModeInfo::navigatePageUp ()
{
   onVScrollPageUp();
   return -1;
}

int LCmdFilePanelModeInfo::navigateRandom ( int index )
{
   if (index < 0 || index >= fpanel.items.getCount())
      return -1;
   curSelectedItem = index;
   return curSelectedItem;
}

bool LCmdFilePanelModeInfo::onFontNameSizeChanged ( const GString& fontNameSize )
{
   layout();
   return true;
}

bool LCmdFilePanelModeInfo::onPaint ( GGraphics& g, const GRectangle& rect )
{
   g.drawFilledRectangle(rect, fpanel.colors.itemBck);
   fpanel.info.paint(g, &rect);
   return true;
}

void LCmdFilePanelModeInfo::updateHScrollBarPos ()
{
   setHScrollPos(fpanel.info.xpos0);
}

void LCmdFilePanelModeInfo::updateVScrollBarPos ()
{
   setVScrollPos(fpanel.info.ypos0);
}

void LCmdFilePanelModeInfo::updateScrollBarPos ()
{
   updateHScrollBarPos();
   updateVScrollBarPos();
}

void LCmdFilePanelModeInfo::updateScrollBarRange ()
{
   int thumbLen = 0;
   int scrollLen = 0;

   // The vertical scrollbar
   thumbLen = filesRect.height;
   scrollLen = fpanel.info.sumHeightOfItems() - thumbLen - 1;
   setVScrollRange(scrollLen, thumbLen);

   // The horizontal scrollbar
   thumbLen  = filesRect.width;
   scrollLen = fpanel.info.widthOfWidestItem() - thumbLen - 1;
   setHScrollRange(scrollLen, thumbLen);

   // Also update the current scrollbar(s) thumb position, so that they also
   // always are placed at the correct position. (E.g. when the file list
   // window has been resized.)
   updateScrollBarPos();
}

void LCmdFilePanelModeInfo::itemsListHasBeenRefreshed ()
{
   layout();
}

void LCmdFilePanelModeInfo::layout ()
{
   filesRect = getWindowRect();
   GDecoratedWindow::invalidateAll(true);
   fpanel.info.updateItemRects();
   updateScrollBarRange();
}

void LCmdFilePanelModeInfo::scrollHorizontal ( int add )
{
   int old = fpanel.info.xpos0;
   fpanel.info.xpos0 += add;
   if (fpanel.info.xpos0 < 0)
   {
      fpanel.info.xpos0 = 0;
   }
   else
   {
      int widest = fpanel.info.widthOfWidestItem();
      if (fpanel.info.xpos0 >= (widest - filesRect.width))
         fpanel.info.xpos0 = widest - filesRect.width - 1;
   }
   int diff = old - fpanel.info.xpos0;
   GRectangle scroll(filesRect);
   scrollWindow(diff, 0, true, &scroll);
   updateHScrollBarPos();
   updateWindow();
}

void LCmdFilePanelModeInfo::scrollVertical ( int add )
{
   int old = fpanel.info.ypos0;
   fpanel.info.ypos0 += add;
   if (fpanel.info.ypos0 < 0)
   {
      fpanel.info.ypos0 = 0;
   }
   else
   {
      int sumh = fpanel.info.sumHeightOfItems();
      int max = sumh - filesRect.height;
      if (fpanel.info.ypos0 >= max)
         fpanel.info.ypos0 = max - 1;
   }
   int diff = fpanel.info.ypos0 - old;
   GRectangle scroll(filesRect);
   // scroll.y = scroll.y + scroll.height - sumh;
   scrollWindow(0, diff, true, &scroll);
   updateWindow();

   updateVScrollBarPos();
}

bool LCmdFilePanelModeInfo::onHScrollLineUp ()
{
   scrollHorizontal(-10);
   return true;
}

bool LCmdFilePanelModeInfo::onHScrollLineDown ()
{
   scrollHorizontal(10);
   return true;
}

bool LCmdFilePanelModeInfo::onHScrollPageUp ()
{
   scrollHorizontal(-filesRect.width);
   return true;
}

bool LCmdFilePanelModeInfo::onHScrollPageDown ()
{
   scrollHorizontal(filesRect.width);
   return true;
}

bool LCmdFilePanelModeInfo::onHScrollSliderTrack ( int pos )
{
   scrollHorizontal(pos - fpanel.info.xpos0);
   return true;
}

bool LCmdFilePanelModeInfo::onVScrollLineUp ()
{
   scrollVertical(-(fpanel.info.items.get(0).getHeight()));
   return true;
}

bool LCmdFilePanelModeInfo::onVScrollLineDown ()
{
   scrollVertical(fpanel.info.items.get(0).getHeight());
   return true;
}

bool LCmdFilePanelModeInfo::onVScrollPageUp ()
{
   scrollVertical(-filesRect.height);
   return true;
}

bool LCmdFilePanelModeInfo::onVScrollPageDown ()
{
   scrollVertical(filesRect.height);
   return true;
}

bool LCmdFilePanelModeInfo::onVScrollSliderTrack ( int pos )
{
   scrollVertical(pos - fpanel.info.ypos0);
   return true;
}
