/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __LCMD_FILEITEM
#define __LCMD_FILEITEM

#include "glib/vfs/GFileItem.h"
#include "glib/vfs/GVfs.h"

/**
 * Class used to represent a single file or directory item in the
 * file panel list.
 *
 * The information in our super class {@link GFileItem} all represents 
 * the file item as shown in the file panel. That is, it does not necessarily
 * represent a physical file, because the file items in the file panel might
 * originate from some virtual file system (VFS), such as e.g. ZIP or FTP.
 * In that case, to get the path to a physical file representation of the
 * virtual file represented by this file item object, 
 * use {@link #getPhysicalFileRepresentation}. See the documentation 
 * of that method for more information.
 *
 * @author  Leif Erik Larsen
 * @since   1998.07.19
 */
class LCmdFileItem : public GFileItem
{
   public:

      // ---
      enum FINFO_FLAGS
      {
         /** Dummy placeholder for "no internal flags". */
         FINFO_NONE = 0x0000,

         /** Force size of file to be displayed (usually due to "Calculate dir size" has been run on this file item). */
         FINFO_SHOWSIZE = 0x0001,

         /** Bit is set if the item is already painted with respect to a drag operation. */
         FINFO_EMPHASIZED = 0x0002,
      };

   private:

      /** True if filename is curently marked. */
      bool marked;

      /** The icon of the file item, or null if uninitialized. */
      const class GIcon* icon;

      /** True if the icon was loaded from external file and shuld be destroyed. */
      bool iconMustBeDeleted;

   public:

      /** Total pixels width of item in BRIEF mode only. (Icon, filename, extention, etc.) */
      int briefWidth;

   private:

      /** Flag(s) defined by Larsen Commander (FINFO_*). */
      int internalFlags;

      /** Text to show in the files-panel for the "up-dir" ("%Txt_FP_UpDir"). */
      static GString TxtUpDir;

      /** Text to show in the files-panel for "dub-dirs" ("%Txt_FP_SubDir"). */
      static GString TxtSubDir;

      /**
       * Pointer to the file panel that owns this file item, or null.
       *
       * If the instance variable {@link #path} does not contain a
       * directory (that is if path's {@link GFile#isDirectory} returns false)
       * then we will use the current directory of this file panel as the
       * directory of the file item.
       *
       * This is for speed and memory optimizing reasons. It would have been
       * waste of both time and memory if we had to store and parse the
       * directory once for each file item in the file panels. Espessially
       * if the current directory of the file panel contains a huge number
       * of files.
       *
       * It is an error if this pointer is null while the instance
       * variable {@link #path} does not contain a directory.
       */
      const class LCmdFilePanel* panel;

      /**
       * @author  Leif Erik Larsen
       * @since   2005.01.04
       * @see     #getPhysicalFileRepresentation
       */
      GVfs::File* physicalFileRep;

   public:

      /**
       * Create a new File Item Object by cloning the specified
       * File Item Object. We will not copy the icon.
       */
      explicit LCmdFileItem ( const class LCmdFileItem& src );

      /**
       * Create a new File Item Object by cloning the specified source.
       */
      LCmdFileItem ( class LCmdFilePanel& panel, const GFileItem& src );

      /**
       * Create a new File Item and initialize it using the
       * specified file path.
       *
       * @throws GFileNotFoundException if we fail to access the specified file.
       */
      LCmdFileItem ( class GVfs& vfs, const GString& path );

      virtual ~LCmdFileItem ();

   public:

      /** Return true if and only if the file item is curently marked. */
      bool isMarked () const;

      /** Return true if and only if the file item is curently marked. */
      void setMarked ( bool marked );

      /** Get the total pixels width of the item in BRIEF mode only.  */
      int getBriefWidth () const;

      /** Get the flag(s) defined by Larsen Commander (FINFO_*). */
      int getInternalFlags () const;

      /** Set the flag(s) defined by Larsen Commander (FINFO_*). */
      void setInternalFlags ( int flags );

      /**
       * Paint the icon of the file item, at the specified position on
       * the specified graphics object.
       */
      void drawTheIcon ( class GGraphics& g, 
                         int xpos, 
                         int ypos, 
                         class LCmdFilePanel& fp );

      /** 
       * Draw the cell that normally shows the size of the file items if it 
       * actually is a file, or "Up/Subdir" if the file items is a directory.
       * However, if a "calculate directory size" operation has been done
       * on the directory, then the soze information is shown even if the 
       * file item is a directory.
       *
       * @author  Leif Erik Larsen
       * @since   2004.03.18
       */
      void drawTheFileSizeCell ( class GGraphics& g, 
                                 const class GRectangle& r, 
                                 const class GColor& fc,
                                 bool alwaysLeftAlign );

      /**
       * Check if the specified internal flag is on or not.
       */
      bool isInternalFlag ( int flag ) const;

      /**
       * Test if "this" file item matches the specified filter options.
       *
       * @return True if the file item do match, or else false if it doesn't.
       */
      bool doesItemMatchFilter ( class LCmdDlgFileFilterProperties& filter ) const;

      /**
       * Test if the full name (incl. extension) of "this" file item does
       * match any of the specified semicolon separted filter strings.
       *
       * @return True if filename match, or else false.
       */
      bool doesItemMatchFilterList ( const GString& filterList ) const;

      /**
       * Get a locale-specific string that describes all the file system
       * attributes of the file item. The returned string can be used for
       * instance in the Attributes-field of the "file details" dialog.
       *
       * @author  Leif Erik Larsen
       * @since   2001.03.24
       */
      GString getFileAttributesDescrStr () const;

      /**
       * Fetch out and associate the correct pointer/icon to specified
       * panel item.
       *
       * @see  #unloadIcon
       */
      const GIcon* getIcon ( class LCmdFilePanel& fp );

      /**
       * Get a pointer to the physical file representation of this 
       * file item, if it has been set by a recent call to 
       * {@link #setPhysicalFileRepresentation}. If the return value 
       * is != null it is easy for the caller to perform physical 
       * file operations on the file even if it originates from some 
       * virtual file system such as e.g. FTP or ZIP.
       * 
       * @author  Leif Erik Larsen
       * @since   2005.01.04
       * @see     #setPhysicalFileRepresentation
       */
      GVfs::File* getPhysicalFileRepresentation ();

      /**
       * @author  Leif Erik Larsen
       * @since   2005.01.04
       * @see     #getPhysicalFileRepresentation
       */
      void setPhysicalFileRepresentation ( GVfs::File* f );

      /**
       * Clears the icon property of the file item, and automatically 
       * destroys it from memory if needed.
       *
       * @author  Leif Erik Larsen
       * @since   2004.03.08
       * @see     #getIcon
       */
      void unloadIcon ();

      /**
       * Return true if and only if the file item is associated with
       * any application, program or script.
       */
      bool isDocument ( class LCmdFilePanel& fp ) const;

      /**
       * Get a copy of the full path of the filename item.
       * That is, the path of the file as it shows on its file system.
       *
       * @author  Leif Erik Larsen
       * @since   2005.01.05
       * @see     #getFullPhysicalPath
       */
      virtual GString getFullPath () const;

      /**
       * Get the path of the physical file representation of the file 
       * represented by this file item. That is if the file originates 
       * from some virtual file system (e.g. ZIP or FTP) and has been 
       * already prepared for physical actions on the local file system.
       *
       * If the file originates from the local file system or the file 
       * has not already been prepared for physical actions, we will 
       * return the same as {@link #getFullPath}.
       *
       * @author  Leif Erik Larsen
       * @since   2005.01.05
       */
      GString getFullPhysicalPath () const;

      /**
       * Get a copy of the directory (including drive and terminating
       * slash) of the filename item.
       */
      virtual GString getDirectory () const;
};

#endif // #ifndef __LCMD_FILEITEM
