/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "lcmd/LCmdDlgFNameCase.h"
#include "lcmd/LCmdFileItem.h"
#include "lcmd/LCmdFilePanel.h"

#include "glib/GProgram.h"
#include "glib/gui/GDialogPanel.h"
#include "glib/gui/event/GDialogMessage.h"
#include "glib/vfs/GExtendedAttributes.h"
#include "glib/sys/GSystem.h"

LCmdDlgFNameCase::LCmdDlgFNameCase ( LCmdFilePanel& panel )
                 :LCmdDirProgressBase(panel.vfs.peek(), &panel),
                  nameToWhat(TOLOWER),
                  extToWhat(TOLOWER),
                  changeIfMixed(false)
{
   recalcPanelWhenFinish = true;
}

LCmdDlgFNameCase::~LCmdDlgFNameCase ()
{
}

GString LCmdDlgFNameCase::changeString ( const GString& str, CASETOWHAT toWhat )
{
   GString ret = str;
   switch (toWhat)
   {
      case TOLOWER:
         ret.toLowerCase();
         break;

      case TOUPPER:
         ret.toUpperCase();
         break;

      case TOMIXED:
      {
         bool prevWasAnAlpha = false;
         int num = ret.length();
         for (int index=0; index<num; index++)
         {
            char chr = ret.charAt(index);
            if (prevWasAnAlpha)
               ret.setCharAt(index, (char) tolower(chr));
            else
               ret.setCharAt(index, (char) toupper(chr));
            prevWasAnAlpha = isalpha(chr) != 0;
         }
         break;
      }

      case TONONE:
      default:
         break;
   }

   return ret;
}

bool LCmdDlgFNameCase::handleFileItem ( LCmdFileItem* pFile,
                                        int /*fileIndex*/,
                                        bool* bSkipped,
                                        bool* itemUpdated )
{
   if (!changeIfMixed)
      if (pFile->getName().isMixedCase() || pFile->getExtension().isMixedCase())
         return true;

   GString fname1 = pFile->getFileName();

   if (!changeIfMixed)
      if (fname1.isMixedCase())
         return true;

   pFile->setName(changeString(pFile->getName(), nameToWhat));
   pFile->setExtension(changeString(pFile->getExtension(), extToWhat));

   GString fname2 = pFile->getFileName();

   if (fname1 == fname2)
      return true;

   GString fullDir = curDir;
   for (int i=fullDir.length() - 1;
        i > 0 && !GFile::IsSlash(fullDir.charAt(i));
        i--, fullDir.removeLastChar());

   fullDir += fname2;

   for (;;)
   {
      GError rc = vfs.moveOrRenameFile(curDir, fullDir, false);
      if (rc == GError::Ok)
      {
         // Remove the ".LONGNAME" EA. In case of any error, ignore it, not critical
         GExtendedAttributes::DiscardEA(fullDir, EA_NAME_LONGNAME);
      }
      else
      {
         GMessageBox::Answer answ;
         if (skipAllByDefault)
         {
            answ = GMessageBox::IDSKIP;
         }
         else
         {
            GString msg("Failed on rename file '%s' to '%s'.\nMessage from the system: %s", GVArgs(fname1).add(fname2).add(rc.getErrorMessage()));
            answ = showWorkerMessageBox(msg, GMessageBox::TYPE_ERROR, "Ss!rc");
         }
         switch (answ)
         {
            case GMessageBox::IDCANCEL:
                 statusOK = false;
                 return statusOK;

            case GMessageBox::IDSKIPALL:
                 skipAllByDefault = true; // Don't ask again from now!
            case GMessageBox::IDSKIP:
                 *itemUpdated = false;
                 if (!pFile->isDirectory())
                    *bSkipped = true;
                 break;

            default:
                 continue; // Try again...
         }
      }
      break;
   }

   return true;
}

void LCmdDlgFNameCase::execute ()
{
   const GVfs& vf = panel->vfs.peek();
   if (!vf.supportsChangeFileNameCase())
   {
      GString msg("%VfsErrCommandNotSupported");
      GProgram& prg = GProgram::GetProgram();
      GWindow& mwin = prg.getMainWindow();
      mwin.showMessageBox(msg, GMessageBox::TYPE_ERROR);
      return;
   }

   if (!panel->activatePanelDriveAndDir(true))
      return;

   // Execute the secondary thread as soon as the progress bar has been created
   startTheDirectoryProcessor("DlgFNameCase", 
                              "%Txt_DlgFNameCase_TitleProgBar");
}

bool LCmdDlgFNameCase::handleDialogMessage ( GDialogMessage& msg )
{
   GDialogPanel& dlg = msg.getDialog();
   switch (msg.getID())
   {
      case GM_INITDIALOG:
      {
         LCmdDirProgressBase::handleDialogMessage(msg);
         switch (nameToWhat)
         {
            case TOLOWER: dlg.setComponentValue("110", true); break;
            case TOUPPER: dlg.setComponentValue("111", true); break;
            case TOMIXED: dlg.setComponentValue("112", true); break;
            case TONONE: default: dlg.setComponentValue("113", true); break;
         }
         switch (extToWhat)
         {
            case TOLOWER: dlg.setComponentValue("115", true); break;
            case TOUPPER: dlg.setComponentValue("116", true); break;
            case TOMIXED: dlg.setComponentValue("117", true); break;
            case TONONE: default: dlg.setComponentValue("118", true); break;
         }
         dlg.setComponentValue("105", changeIfMixed);
         dlg.setComponentFocus("DLG_OK");
         return true;
      }

      case GM_COMMAND:
      {
         LCmdDirProgressBase::handleDialogMessage(msg);
         GString cmdID = msg.getParam1String();
         if (cmdID == "DLG_OK")
         {
            changeIfMixed = dlg.getComponentBoolValue("105");

            // Name
            if (dlg.getComponentBoolValue("110"))
               nameToWhat = TOLOWER;
            else
            if (dlg.getComponentBoolValue("111"))
               nameToWhat = TOUPPER;
            else
            if (dlg.getComponentBoolValue("112"))
               nameToWhat = TOMIXED;
            else
               nameToWhat = TONONE;

            // Extension
            if (dlg.getComponentBoolValue("115"))
               extToWhat = TOLOWER;
            else
            if (dlg.getComponentBoolValue("116"))
               extToWhat = TOUPPER;
            else
            if (dlg.getComponentBoolValue("117"))
               extToWhat = TOMIXED;
            else
               extToWhat = TONONE;

            if (nameToWhat == TONONE &&
                extToWhat == TONONE)
            {
               GStringl msg("You have choosen not to change the case of neither the file name nor extension. Is this what you want to do?");
               GMessageBox::Answer answ = dlg.showMessageBox(msg, GMessageBox::TYPE_QUESTION, "Ync");
               if (answ != GMessageBox::IDYES)
                  return true;
               bCanceled = true;
            }

            dlg.dismiss(cmdID);
         }
         return true;
      }

      default:
         return LCmdDirProgressBase::handleDialogMessage(msg);
   }
}

