/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __LCMD_DLGCHOOSEDRIVE
#define __LCMD_DLGCHOOSEDRIVE

#include "glib/GThread.h"
#include "glib/util/GDriveInfo.h"
#include "glib/gui/event/GDialogMessageHandler.h"
#include "glib/gui/event/GListSelectionListener.h"

/**
 * This class forms the definition of the dialog box that is used to let
 * the user select one of the existing drives. This is by the
 * <i>cmdLeftChooseDrive</i> and the <i>cmdRightChooseDrive</i>
 * commands.
 *
 * @since   2000.05.10
 * @author  Leif Erik Larsen
 * @see     LCmdMainWindow#cmdLeftChooseDrive
 * @see     LCmdMainWindow#cmdRightChooseDrive
 * @see     LCmdFilePanel#cmdChangeDrive
 */
class LCmdDlgChooseDrive : private GThread,
                           private GDialogMessageHandler,
                           private GListSelectionListener
{
   private:

      /** Input/Output: 0=None (cancel), 1=A:, 2=B:, 3=C:, etc. */
      int selectedDrive;

      /** Input: Title string of dialog. */
      GString titleStr;

      /** Information about every available drive. */
      GDriveInfo info[GDriveInfo::MAX_DRIVES];

      /** Bit map of which drives are avilable on the system. */
      int disksBitMap;

      /** True if the thread has been requested to stop. */
      volatile bool hasRequestedStop;

      /** A reference to the calling owner program. */
      class GProgram& ownerProg;

      /**
       * The dialog to receive UM_DRIVEINFO_UPDATED messages as is given
       * to {@link #refreshDrivesInfo}.
       */
      class GDialogPanel* dlg;

   public:

      explicit LCmdDlgChooseDrive ( class GProgram& ownerProg );
      virtual ~LCmdDlgChooseDrive ();

   private:

      /** Disable the assignment operator for this class. */
      const LCmdDlgChooseDrive& operator= ( const class LCmdDlgChooseDrive& src ) const;

      /**
       * Get the name of the icon resource that best "describes" a drive of
       * the specified type.
       */
      GString getDriveIconName ( GDriveInfo::DRIVE_TYPE driveType );

      /**
       * Get the description text of the specified drive info.
       */
      GString getDriveText ( const class GDriveInfo& inf );

      /**
       * This is the message handler method that is called to handle
       * messages from the dialog box window.
       */
      virtual bool handleDialogMessage ( class GDialogMessage& msg );

      /** Part of {@link GListSelectionListener}. */
      virtual void onListBoxSelectionChanged ( class GAbstractListBox& listb );

      /** Part of {@link GListSelectionListener}. */
      virtual void onListBoxSelectionChangedByCharKey ( class GAbstractListBox& listb );

      /** Part of {@link GListSelectionListener}. */
      virtual void onListBoxSelectionDblClick ( class GAbstractListBox& listb );

      /**
       * Reread information of every available drive into our drives
       * information block table, but only if <i>force</i> is true or if the
       * bit map of available drives on the system has changed.
       * <p>
       * The refresh can be either synchronous or asynchronous. If it is
       * asynchronous then we will launch a background thread of which will
       * perform the update. Regardsless of this, the <i>disksBitMap</i>
       * variable will always be updated synchrounsouly so that the caller
       * can always know which drives are available on the system.
       *
       * @param  force   True if we shall refresh even if the drives map has
       *                 not changed.
       * @param  dlg     Reference to which dialog panel of which to receive
       *                 UM_DRIVEINFO_UPDATED messages once for each
       *                 refreshed drive info block.
       */
      void refreshDrivesInfo ( bool force, class GDialogPanel& dlg );

      /**
       * The method to be run by a background thread to update drives info.
       */
      virtual void run ();

   public:

      /**
       * Execute the modal dialog that lets the user pick one of the
       * available drives.
       *
       * @param  parentWin  Parent window, or NULL.
       * @param  driveInit  Which drive to initially select (1=A, 2=B, 3=C, etc.).
       * @param  titleStr   String to use in the title bar of the dialog box window.
       * @return 1=A, 2=B, 3=C, etc. or 0 if user canceled the operation (or any error).
       */
      int execute ( class GWindow* parentWin, int driveInit, const GString& titleStr );
};

#endif // #ifndef __LCMD_DLGCHOOSEDRIVE
