/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "lcmd/LCmdDlgChooseDrive.h"

#include "glib/GProgram.h"
#include "glib/exceptions/GIllegalStateException.h"
#include "glib/gui/event/GDialogMessage.h"
#include "glib/gui/GDialogFrame.h"
#include "glib/gui/GDialogPanel.h"
#include "glib/gui/GListBox.h"

LCmdDlgChooseDrive::LCmdDlgChooseDrive ( GProgram& ownerProg )
                   :GThread("Drives-Info-Refresher", 32768),
                    selectedDrive(0),
                    disksBitMap(0),
                    hasRequestedStop(false),
                    ownerProg(ownerProg),
                    dlg(null)
{
}

LCmdDlgChooseDrive::~LCmdDlgChooseDrive ()
{
}

const LCmdDlgChooseDrive& LCmdDlgChooseDrive::operator= ( const class LCmdDlgChooseDrive& src ) const 
{ 
   return *this; 
}

GString LCmdDlgChooseDrive::getDriveIconName ( GDriveInfo::DRIVE_TYPE driveType )
{
   switch (driveType)
   {
      case GDriveInfo::DT_FLOPPY: return "IDP_DRIVE_FLOPPY";
      case GDriveInfo::DT_CDROM: return "IDP_DRIVE_CDROM";
      case GDriveInfo::DT_HARDDISK: return "IDP_DRIVE_HARDDISK";
      case GDriveInfo::DT_NETWORK: return "IDP_DRIVE_NETWORK";
      case GDriveInfo::DT_REMOVABLE:  return "IDP_DRIVE_REMOVABLE";
      case GDriveInfo::DT_RAMDRIVE: return "IDP_DRIVE_RAMDRIVE";
      case GDriveInfo::DT_UNKNOWN: default: return "IDP_DRIVE_UNKNOWN";
   }
}

GString LCmdDlgChooseDrive::getDriveText ( const GDriveInfo& inf )
{
   return GString("%c: %s", GVArgs(char(inf.drive + 'A' - 1)).add(inf.volumeName));
}

void LCmdDlgChooseDrive::run ()
{
   for (int i=0; i<GDriveInfo::MAX_DRIVES; i++)
   {
      if (hasRequestedStop)
         break;
      GDriveInfo& di = info[i];
      if (di.isAvailable)
      {
         di.update(i+1);
         if (hasRequestedStop)
            break;
         di.isAvailable = true; // In case di.update() cleared it.
         if (dlg != null)
         {
            GString istr = GInteger::ToString(i);
            GDialogMessage dm(*dlg, GM_USER, "UM_DRIVEINFO_UPDATED", istr);
            sendGuiUserMessage(*dlg, dm);
         }
      }
   }
}

void LCmdDlgChooseDrive::refreshDrivesInfo ( bool force, GDialogPanel& dlg )
{
   int map = GDriveInfo::GetMapOfAvailableDrives();

   // Update the disks bitmap, which tells us which drives are
   // physically available (mounted). This gives us no drive types
   // information however.
   bool doit = (force || (disksBitMap != int(map)));
   if (doit)
   {
      // If the drives refresher thread is already running then stop it.
      if (isRunning())
         hasRequestedStop = true;

      disksBitMap = map; // Remember the new disks map to the next call

      for (int i=0; i<GDriveInfo::MAX_DRIVES; i++)
      {
         info[i].clear();
         if ((map >> i) & 0x1)
         {
            info[i].isAvailable = true;
            info[i].drive = i+1;
         }
      }
   }

   // Update the GUI listbox with one item for each available drive.
   // Since the drive types are not yet known we will give each listbox item
   // temporary drive type icon, which is something like a question mark.
   GListBox& list = dynamic_cast<GListBox&>(dlg.getComponentByID("101"));
   list.removeAllItems();
   for (int i=0, idx=0; i<GDriveInfo::MAX_DRIVES; i++)
   {
      GDriveInfo& inf = info[i]; // Make this a fast one
      if (inf.isAvailable)
      {
         GString itemText = getDriveText(inf);
         GString iconName = getDriveIconName(inf.type);
         list.addItem(itemText, iconName);
         list.setItemUserData(idx, &inf, false);
         if (inf.drive == selectedDrive)
            list.setSelectedIndex(idx);
         idx++;
      }
   }

   // Request all parts of the listbox to be repainted.
   list.invalidateAll(true);

   // Start the background thread used to read drive types information.
   this->dlg = &dlg;
   if (doit)
   {
      hasRequestedStop = false;
      try {
         start();
      } catch (GIllegalStateException& /*e*/) {
         // Nothing critical about this. It probably means that 
         // the user is sitting and fast-clicking repeatedly on the 
         // Refresh-button, just for for, and starting the thread
         // fails with this exception if it is already running.
         // This can happen if an attempt is made to start the 
         // thread again before the recent one has got time to exit.
      }
   }
}

int LCmdDlgChooseDrive::execute ( GWindow* parentWin, int driveInit, const GString& titleStr )
{
   this->selectedDrive = driveInit;
   this->titleStr = titleStr;

   GProgram& prg = GProgram::GetProgram();
   GString id = prg.executeDialog(parentWin, "DlgChooseDrive", this);

   // Stop the drives info collector thread in case it is still
   // working. In that case we must clear the drives bitmap to force
   // the  drives information blocks to be refreshed the next time the
   // dialog is opened.
   if (isRunning())
   {
      hasRequestedStop = true;
      disksBitMap = 0;
   }

   if (id == "DLG_OK")
      return selectedDrive;
   else
      return 0;
}

void LCmdDlgChooseDrive::onListBoxSelectionChanged ( class GAbstractListBox& listb ) 
{
}

void LCmdDlgChooseDrive::onListBoxSelectionDblClick ( class GAbstractListBox& listb ) 
{
}

void LCmdDlgChooseDrive::onListBoxSelectionChangedByCharKey ( GAbstractListBox& /*listb*/ )
{
   dlg->sendDialogMessage(GM_COMMAND, "DLG_OK");
}

bool LCmdDlgChooseDrive::handleDialogMessage ( GDialogMessage& msg )
{
   GDialogPanel& dlg = msg.getDialog();
   switch (msg.getID())
   {
      case GM_INITDIALOG:
      {
         dlg.getOwnerFrame().setText(titleStr);
         GListBox& lbox = dynamic_cast<GListBox&>(dlg.getComponentByID("101"));
         lbox.addListBoxSelectionListener(this);
         dlg.sendDialogMessage(GM_USER, "UM_DRIVEINFO_INITLIST", "false");
         return true;
      }

      case GM_LISTBOX_DBLCLICK:
      {
         GString id = msg.getParam1String();
         if (id == "101")
            dlg.sendDialogMessage(GM_COMMAND, "DLG_OK");
         return true;
      }

      case GM_COMMAND:
      {
         GString cmdID = msg.getParam1String();
         if (cmdID == "DLG_REFRESH")
         {
            dlg.sendDialogMessage(GM_USER, "UM_DRIVEINFO_INITLIST", "true");
         }
         else
         if (cmdID == "DLG_OK")
         {
            GListBox& list = dynamic_cast<GListBox&>(dlg.getComponentByID("101"));
            int sel = list.getSelectedIndex();
            if (sel >= 0)
            {
               GDriveInfo* inf = (GDriveInfo*) list.getItemUserData(sel);
               if (inf == null) // Should never happen, but in case.
               {
                  if (GLog::Filter(GLog::TEST))
                     GLog::Log(this, "%s: list.getItemUserData(%d) returned null", GVArgs(__FUNCTION__).add(sel));
                  return true;
               }
               selectedDrive = inf->drive;
               dlg.dismiss(cmdID);
            }
         }
         else
         if (cmdID == "DLG_CANCEL")
         {
            selectedDrive = 0;
            dlg.dismiss(cmdID);
         }
         return true;
      }

      case GM_USER:
      {
         GString id = msg.getParam1String();
         if (id == "UM_DRIVEINFO_UPDATED")
         {
            int driveIndex = msg.getParam2Int();
            int listIndex = -1;
            GDriveInfo* inf = null;
            for (int i=0; i<=driveIndex; i++)
            {
               inf = &info[i];
               if (inf->isAvailable)
                  listIndex++;
            }
            if (listIndex >= 0)
            {
               GListBox& list = dynamic_cast<GListBox&>(dlg.getComponentByID("101"));
               GString itemText = getDriveText(*inf);
               GString iconName = getDriveIconName(inf->type);
               list.setItemTextAndIcon(listIndex, itemText, iconName);
            }
            return true;
         }
         else
         if (id == "UM_DRIVEINFO_INITLIST")
         {
            // Refresh drives list in case any new drive has been mounted
            // since we were previously here.
            bool b = msg.getParam2Bool(); // Force?
            refreshDrivesInfo(b, dlg);
            return true;
         }
         return false;
      }

      default:
         return false;
   }
}

