/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __LCMD_DLGCHILDPROCESSSELECTOR
#define __LCMD_DLGCHILDPROCESSSELECTOR

#include "glib/gui/event/GDialogMessageHandler.h"
#include "glib/primitives/GInteger.h"
#include "glib/util/GArray.h"
#include "glib/sys/GSystem.h"

/**
 * This class forms the Dialog Box used by the command line class
 * {@link LCmdCmdLineEntry} to let the user select one of the child
 * processes that are currently running under the control
 * of Larsen Commander.
 *
 * This dialog is used to select which process to break when user
 * has pressed Ctrl+C on keyboard, or by any other code that needs
 * the user to explicitly define a currently running child process.
 *
 * @author  Leif Erik Larsen
 * @since   2000.06.14
 */
class LCmdDlgChildProcessSelector : public GDialogMessageHandler
{
   private:

      /** Reference to the command line object of where we are contained. */
      class LCmdCmdLineEntry& entry;

      /** The break type of which to be initially selected by default. */
      GSystem::BreakType defaultBt;

      /**
       * Each listbox item has an element in this array. The element is
       * an integer that is the index of the corresponding child process,
       * which is actually contained in <i>entry.runningChildren</i>.
       */
      GArray<GInteger> indexMap;

      /**
       * Pointer to the selected process, but only if executed dialog
       * was dismissed with the DLG_OK button. Otherwise, this will be null.
       */
      class LCmdProcessLauncher* selectedProcess;

      /** The user selected break type if dialog was dismissed with OK. */
      GSystem::BreakType selectedBt;

      /** True if we shall break the whole process tree of the selected process. */
      bool breakTree;

      /** See {@link #setPureProcessSelectionMode}. */
      bool pureProcessSelectionMode;

   public:

      explicit LCmdDlgChildProcessSelector ( class LCmdCmdLineEntry& entry,
                                             GSystem::BreakType defaultBt,
                                             bool breakTree );

      virtual ~LCmdDlgChildProcessSelector ();

   private:

      bool handleDialogMessage ( GDialogMessage& msg );

   public:

      /**
       * Execute the dialog to let the user select one of the currently
       * running child processes of Larsen Commander.
       *
       * @author  Leif Erik Larsen
       * @since   2000.06.14
       * @return  Pointer to the launcher of the selected child process,
       *          or null if Cancel was selected or if there are not
       *          currently running child processes to select from.
       */
      class LCmdProcessLauncher* execute ( class GWindow& parentWin );

      /**
       * Get the user selected break type. This method makes sense 
       * only after the dialog has been dismissed with the OK button.
       *
       * @author  Leif Erik Larsen
       * @since   2004.06.25
       */
      GSystem::BreakType getSelectedBreakType () const;

      /**
       * Check if the "Process Tree" switch is toggled ON for the 
       * selected process. If this is true then the selected process an
       * all its child-processes should be killed. Else only the selected
       * process it self (not its child processes) should be killed.
       *
       * @author  Leif Erik Larsen
       * @since   2004.07.26
       */
      bool isBreakTree () const;

      /**
       * By default, this class is used for the user to select a child 
       * process of which to kill or break. In that case the dialog contains 
       * a droplist and a toggle button for some additional properties 
       * to be used with the break-command. However, the dialog can also 
       * operate without these additional components. This is simply 
       * achieved by calling this method with a true argument. This must 
       * be done before {@link #execute} is called.
       *
       * @author  Leif Erik Larsen
       * @since   2005.07.22
       */
      void setPureProcessSelectionMode ( bool flag );
};

#endif // #ifndef __LCMD_DLGCHILDPROCESSSELECTOR
