/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __LCMD_DIRCACHE
#define __LCMD_DIRCACHE

#include "lcmd/LCmdDirProgressBase.h"
#include "lcmd/LCmdDlgFileFilterProperties.h"

/**
 * The directory cache is used to store all directories of a given drive
 * in memory for fast access in a listbox (via {@link LCmdDlgDirCache}) for
 * instance when the user tries to perform a "CD" command that does not
 * actually exist.
 *
 * @author  Leif Erik Larsen
 * @since   1998.06.25
 */
class LCmdDirCache : public LCmdDirProgressBase
{
   public:

      /** For future usage, if we want some information per Cached Directory item. */
      class Item : public GObject
      {
      public:
         Item () {}
      };

      /** Path of text file of where to read/write the content of the directory cache. */
      GString filePath;

      /** */
      LCmdDlgFileFilterProperties Filter;

      /** The dynamic array of directories. The key is the directory. Item is null. */
      GKeyBag<Item> dirs;

   public:

      explicit LCmdDirCache ();
      virtual ~LCmdDirCache ();

   private:

      /**
       * Needed during construction, due to our super constructor.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.09
       */
      static class GVfsLocal& GetLocalVfs ();

   public:

      /**
       * @author  Leif Erik Larsen
       * @since   2007.02.07
       * @return  The user selected directory, or an empty string if the
       *          user cancels the dialog or if the list doesn't contain 
       *          any matching directory.
       */
      GString pickFromList ( const GString& match );

      const GString& get ( int index );

      /**
       * Add the current directory of the Directory Progress to the cache.
       */
      virtual bool handleFileItem ( class LCmdFileItem *pFile, 
                                    int fileIndex, 
                                    bool *bSkipped, 
                                    bool *itemUpdated );

      /**
       * Add the specified directory to the cache.
       *
       * @author  Leif Erik Larsen
       * @since   2007.02.10
       * @param   dir  The fully qualyfied directory of which to add.
       * @param   notifyListeners True if we shall notify any listeners,
       *               eg the statusbar cell showing the current number
       *               of items in the cache. This parameter should 
       *               normally be true.
       */
      void addDirectoryToCache ( const GString& dir, bool notifyListeners );

      /**
       * Clear the content of the cache.
       */
      void removeAllCachedDirectories ();

      /**
       * @author  Leif Erik Larsen
       * @since   2007.02.11
       * @param   dir         The fully qualified directory of which to strip
       *                      out from the Directory Cache container.
       * @param   inclSubDirs True if we shall also remove all 
       *                      subdirectories (if any).
       * @return  Number of items actually removed from the cache.
       */
      int removeDirectoryFromCache ( const GString& dir, bool inclSubdirs );

      /**
       * Rename all occurences of the specified directory in the 
       * directory cache.
       *
       * @author  Leif Erik Larsen
       * @since   2007.02.11
       * @param   dir     The fully qualified directory of which to 
       *                  rename in the Directory Cache container.
       * @param   newDir  The fully qualified new directory name.
       * @return  Number of items actually renamed in the cache.
       */
      int renameDirectoryInCache ( const GString& dir, const GString& newDir );

      /**
       * Let the directory cache object write it self to its
       * corresponding file. The cache will not be written if it hasn't
       * been rescanned, except if the 'force' parameter is true.
       */
      void writeToProfile ( bool force = false );

      /**
       * Let the directory cache object read it self from its
       * corresponding file.
       */
      void readFromProfile ();
};

#endif // #ifndef __LCMD_DIRCACHE
