/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __LCMD_CONSOLEMONITOR
#define __LCMD_CONSOLEMONITOR

#include "glib/gui/GTextViewer.h"

/**
 * This is the console monitor class that also implements the background
 * thread that is constantly reading from the read-end of the pipe
 * of where Larsen Commander's stdout is directed.
 *
 * @author  Leif Erik Larsen
 * @since   1998.10.03
 */
class LCmdConsoleMonitor : public GTextViewer
{
   private:

      /** To be used internally by {@link #appendTextFromGuiThread} only. */
      GString lineBuff;

      int lineBuffPos;
      class LCmdCmdContainer& conMon;

   public:

      LCmdConsoleMonitor ( class LCmdCmdContainer& parentWin, const GString& constraints );
      virtual ~LCmdConsoleMonitor ();

   public:

      /**
       * Append the specified formatted text to the console monitor, 
       * synchronously. This method should not be called by any background 
       * (non-gui) thread!
       *
       * @author  Leif Erik Larsen
       * @since   2004.04.26
       * @param   format  The format string.
       * @param   args    The arguments to merge into the format string.
       * @see     GVArgs#FormatArgs
       * @see     #appendTextFromSecondaryThread
       */
      void appendTextFromGuiThread ( const GString& format, 
                                     const GVArgs& args = GVArgs() );

      /**
       * This method is called by background threads that reads data from 
       * the redirected output stream of a child process, in order to print
       * the read text to the console monitor window of Larsen Commander.
       * More than one thread might call this method at once, so it has to
       * be thread safe.
       *
       * @author  Leif Erik Larsen
       * @since   2004.04.15
       * @param   thread  A reference to the calling background thread.
       * @param   format  The format string.
       * @param   args    The arguments to merge into the format string.
       * @see     GVArgs#FormatArgs
       * @see     #appendTextFromGuiThread
       */
      void appendTextFromSecondaryThread ( class GThread& thread,
                                           const GString& format, 
                                           const GVArgs& args = GVArgs() );

   protected:

      virtual void onCaretPositionChanged ();
      virtual bool onUserMessage ( class GUserMessage& msg );
      virtual bool onKeyDown ( const GKeyMessage& key );
      virtual bool onInitMenu ();

   protected:

      /**
       * This method will be called by our super class when the user
       * has selected the <i>cmdGoto</i> command.
       */
      virtual int getUserSelectedLineNumber ();
};

#endif // #ifndef __LCMD_CONSOLEMONITOR
