/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/primitives/GCharacter.h"
#include "glib/util/GTokenizer.h"
#include "glib/GEnvironment.h"

#include "lcmd/LCmdAliases.h"
#include "lcmd/LCmdFilePanel.h"

LCmdAliases::LCmdAliases ()
            :aliases(64, -3, true)
{
}

bool LCmdAliases::isAlias ( const GString& name )
{
   return aliases.containsKey(name);
}

bool LCmdAliases::defineAlias ( const GString& name, const GString& value )
{
   GString nameStr = name;
   nameStr.trim();
   if (nameStr == "")
      return false;
   if (nameStr[0] == '!') // The dollar prefix is reserved
      return false;
   nameStr.toUpperCase();
   GString* v = new GString(value);
   if (!isAlias(nameStr))
      return aliases.put(nameStr, v);
   else
      return aliases.update(nameStr, v);
}

bool LCmdAliases::defineAlias ( const GString& script )
{
   GString name(64);
   const char* aliasScript = script.cstring();
   while (isspace(*aliasScript))
      aliasScript++;
   for (; *aliasScript != '\0'; aliasScript++)
   {
      if (*aliasScript == '=')
      {
         aliasScript++;
         while (isspace(*aliasScript))
            aliasScript++;
         if (*aliasScript == '\0')
         {
            undefineAlias(name);
            return true;
         }
         else
         {
            return defineAlias(name, aliasScript);
         }
      }
      else
      if (isspace(*aliasScript))
      {
         while (isspace(*aliasScript))
            aliasScript++;
         aliasScript--; // Compansate for the next increment in for-loop
      }
      else
      {
         name += char(GCharacter::ToUpperCase(*aliasScript));
      }
   }
   return false;
}

void LCmdAliases::undefineAlias ( const GString& name )
{
   aliases.remove(name);
}

int LCmdAliases::getCount ()
{
   return aliases.getCount();
}

const GString& LCmdAliases::getAliasName ( int index )
{
   return aliases.getKey(index);
}

const GString& LCmdAliases::getAliasStr ( int index )
{
   return aliases.getIndexedItem(index);
}

const GString* LCmdAliases::getAliasStr ( const GString& name )
{
   return aliases.get(name);
}

void LCmdAliases::writeProfile ( GSectionBag& ini, bool doIt )
{
   if (doIt)
   {
      ini.deleteSection("Aliases");
      int num = aliases.getCount();
      for (int i=0; i<num; i++)
         ini.putString("Aliases", aliases.getKey(i), aliases.getIndexedItem(i), doIt);
   }
}

void LCmdAliases::readProfile ( GSectionBag& ini )
{
   GKeyBag<GString>* sectBag = ini.getSectionBag("Aliases");
   if (sectBag != null)
   {
      int num = sectBag->getCount();
      for (int i=0; i<num; i++)
         defineAlias(sectBag->getKey(i), sectBag->getIndexedItem(i));
   }
}

GString LCmdAliases::processString ( LCmdFilePanel& panel, 
                                     const GString& src, 
                                     GEnvironment* envList, 
                                     const GString& homeDir )
{
   GTokenizer tokenizer(src, GString::Blank, GString::Empty, true);
   bool isFirst = true;
   GString dst(256);

   for (;;)
   {
      const GToken* token = tokenizer.getNextToken();
      if (token->isEmpty())
         break;

      if (isFirst)
         isFirst = false;
      else
         dst += GString::Blank; // Separate each token with a single space

      if (token->isQuoted())
      {
         dst += token->toString(true);
         continue;
      }

      if (*token == "!d") // !d ==> Current directory
      {
         GString curDir = panel.getCurrentSysDirectory(false);
         dst += GToken(curDir).toString(true);
         continue;
      }

      if (*token == "!p") // !p ==> Path of current item
      {
         LCmdFileItem* fitem = panel.getCurItem();
         if (fitem != null)
         {
            GString path = fitem->getFullPhysicalPath();
            dst += GToken(path).toString(true);
         }
         continue;
      }

      if (*token == "!h") // !h ==> Home directory
      {
         dst += GToken(homeDir).toString(true);
         continue;
      }

      if (*token == "!l") // !l ==> Path of left item
      {
         LCmdFilePanel& leftPanel = LCmdFilePanel::GetLeftPanel();
         LCmdFileItem* fitem = leftPanel.getCurItem();
         if (fitem != null)
         {
            GString path = fitem->getFullPhysicalPath();
            dst += GToken(path).toString(true);
         }
         continue;
      }

      if (*token == "!r") // !r ==> Path of right item
      {
         LCmdFilePanel& rightPanel = LCmdFilePanel::GetRightPanel();
         LCmdFileItem* fitem = rightPanel.getCurItem();
         if (fitem != null)
         {
            GString path = fitem->getFullPhysicalPath();
            dst += GToken(path).toString(true);
         }
         continue;
      }

      GString tokenStr = token->toString();
      if (envList != null && 
          tokenStr.length() >= 3 && 
          tokenStr.charAt(0) == '%' && 
          tokenStr.endsWith("%"))
      {
         // Skip first and last character, which is both '%'.
         GString envName = tokenStr.substring(1, tokenStr.length() - 1); 
         const char* envVal = envList->getEnv(envName, null);
         if (envVal != null)
            dst += envVal;
         else
            isFirst = true; // Don't append another " " because no new text was appended
      }
      else
      if (isAlias(tokenStr))
      {
         GString* str = aliases.get(tokenStr);
         dst += *str;
      }
      else
      {
         dst += tokenStr;
      }
   }

   return dst;
}

