/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_VFSZIP
#define __GLIB_VFSZIP

#include "glib/vfs/GVfsArchiveFile.h"
#include "glib/util/GArray.h"
#include "glib/util/GBuffer.h"

/**
 *
 * @author  Leif Erik Larsen
 * @since   2002.07.28
 */
class GVfsZip : public GVfsArchiveFile
{
   public:

#pragma pack(1)

      class LocalFileHeader
      {
         public:

            unsigned int signature;     // Local file header signature 4 bytes  (0x04034b50)
            unsigned short verExtract;  // Version needed to extract   2 bytes
            unsigned short gflag;       // General purpose bit flag    2 bytes
            unsigned short compMethod;  // Compression method          2 bytes
            unsigned short fileTime;    // Last mod file time          2 bytes
            unsigned short fileDate;    // Last mod file date          2 bytes
            unsigned int crc32;         // crc-32                      4 bytes
            unsigned int compSize;      // Compressed size             4 bytes
            unsigned int uncompSize;    // Uncompressed size           4 bytes
            unsigned short fnameLength; // Filename length             2 bytes
            unsigned short extraLength; // Extra field length          2 bytes
            // File name (variable size)
            // Extra field (variable size)
            // File data: Immediately following the local header for a file
            //            is the compressed or stored data for the file. 

         public:

            LocalFileHeader () :
               signature(0),
               verExtract(0),
               gflag(0),
               compMethod(0),
               fileTime(0),
               fileDate(0),
               crc32(0),
               compSize(0),
               uncompSize(0),
               fnameLength(0),
               extraLength(0)
            {
            }
      };

      class CentralDirFileHeader
      {
         public:

            unsigned int signature;     // Central file header signature  4 bytes (0x02014b50)
            unsigned short verMadeBy;   // Version made by  2 bytes
            unsigned short verNeeded;   // Version needed to extract  2 bytes
            unsigned short gflag;       // General purpose bit flag  2 bytes
            unsigned short compMethod;  // Compression method  2 bytes
            unsigned short fileTime;    // Last mod file time  2 bytes
            unsigned short fileDate;    // Last mod file date  2 bytes
            unsigned int crc32;         // crc-32  4 bytes
            unsigned int compSize;      // Compressed size  4 bytes
            unsigned int uncompSize;    // Uncompressed size  4 bytes
            unsigned short fnameLength; // Filename length  2 bytes
            unsigned short extraLength; // Extra field length  2 bytes
            unsigned short comLength;   // File comment length  2 bytes
            unsigned short unused1;     // Disk number start  2 bytes
            unsigned short intFAttr;    // Internal file attributes  2 bytes
            unsigned int extFAttr;      // External file attributes  4 bytes
            unsigned int headOffs;      // Relative offset of local file header  4 bytes
            // File name  (variable size)
            // Extra field  (variable size)
            // File comment  (variable size)

         public:

            CentralDirFileHeader () :
               signature(0),
               verMadeBy(0),
               verNeeded(0),
               gflag(0),
               compMethod(0),
               fileTime(0),
               fileDate(0),
               crc32(0),
               compSize(0),
               uncompSize(0),
               fnameLength(0),
               extraLength(0),
               comLength(0),
               unused1(0),
               intFAttr(0),
               extFAttr(0),
               headOffs(0)
            {
            }
      };

      class EndOfCentralDir
      {
         public:

            unsigned int signature; // End of central dir signature  4 bytes (0x06054b50)
            unsigned short unused1; // Number of this disk  2 bytes
            unsigned short unused2; // Number of the disk with the start of the central directory  2 bytes
            unsigned short unused3; // Total number of entries in the central directory on this disk  2 bytes
            unsigned short count;   // Total number of entries in the central directory  2 bytes
            unsigned int size;      // Size of the central directory  4 bytes
            unsigned int offset;    // Offset of start of central directory with respect to the starting disk number 4 bytes
            unsigned short comLen;  // .ZIP file comment length 2 bytes
            // .ZIP file comment (variable size)

         public:

            EndOfCentralDir () :
               signature(0),
               unused1(0),
               unused2(0),
               unused3(0),
               count(0),
               size(0),
               offset(0),
               comLen(0)
            {
            }
      };

#pragma pack()

      /**
       * @author  Leif Erik Larsen
       * @since   2005.03.14
       */
      class OpenFile : public GObject
      {
         public:

            /** A reference to the VFS of where we are contained. */
            class GVfsZip& ownerVfs;
    
            /** The zip-file stream used to read/write from/to the zipped data. */
            class GRandomAccessFile& zipFile;

            // Information that has to do with the entry inside the zip-file,
            // of where this open virtual file is represented.
            GVfsArchiveFile::Item& item;

            // Information used for emulating standard stream functionality.
            GVfs::FileHandle handle;
            longlong currentStreamSeekPos;

            // Information used in case the file is opened in write-mode.
            // In that case a temporary physical file is created on the 
            // local file system, and all read/write operations takes place
            // on that physical file. When the file is closed the 
            // corresponding zip-file is updated and the temporary local 
            // file is removed.
            bool writableMode; // True if the file is opened in write-mode.
            GVfs::FileHandle writeHandle; // The GVfsLocal file handle for wrapping of all file i/o.
            GVfs::File* writableFileObj;

            // Buffer for reading data from the zipped stream.
            int chunk;
            int compressedDataPos;
            int compressedDataCount;
            GBuffer<BYTE> compressedData;
            int uncompressedDataPos;
            int uncompressedDataCount;
            GBuffer<BYTE> uncompressedData;

            // ZLib-related instance variables.
            LocalFileHeader head;
            longlong remainingZippedBytes;
            struct z_stream_s* strm;
            int zlibInitErrCode;

            OpenFile ( class GVfsZip& ownerVfs,
                       class GRandomAccessFile& zipFile, 
                       GVfsArchiveFile::Item& item, 
                       bool writable, 
                       bool create, 
                       bool append );

            virtual ~OpenFile ();
            
            void initZipStream ();
            void clearZipStream ();
            int readZipStream ( void* buff, int numBytesToRead );
      };

      static int OpenFileCounter;
      static GHashtable<GInteger, GVfsZip::OpenFile> OpenFiles;

   public:

      /**
       * @author  Leif Erik Larsen
       * @since   2005.03.09
       * @param   localVfs  We need the local VFS in order to read the 
       *                    ZIP-file from a VFS that supports all 
       *                    file seek operations for 100% sure.
       * @param   parentVfs The parent file system, as of where the specified 
       *                    archive file is contained.
       * @param   vfile     The ZIP-file for which this new ZIP VFS 
       *                    is to be represented. We will receive the 
       *                    ownership of this object.
       * @param   toolPath  Path of the ZIP tool program (usually ZIP.EXE).
       * @param   paramsDel The arguments to use for performing delete 
       *                    operations via the ZIP tool command. Must contain 
       *                    the following substring tags:<br>
       *                    %1=Will be replaced by the path of the ZIP-file.<br>
       *                    %2=Will be replaced by the path of the temporary 
       *                       file containing the list of which files to 
       *                       delete from the ZIP-file.<br>
       * @throws  GFileNotFoundException if the specified file can not
       *                                 be found.
       */
      GVfsZip ( class GVfsLocal& localVfs, 
                class GVfs& parentVfs, 
                GVfs::File* vfile,
                const GString& toolPath,
                const GString& paramsDel );

      virtual ~GVfsZip ();

   private:

      /** Disable the copy constructor. */
      GVfsZip ( const GVfsZip& src ) : GVfsArchiveFile(src) {}

      /** Disable the assignment operator. */
      GVfsZip& operator= ( const GVfsZip& ) { return *this; }

      /** 
       * Get a pointer to the OpenFile representation of the specified file 
       * handle, or null if the specified file handle is invalid.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.15
       */
      GVfsZip::OpenFile* getOpenFile ( GVfs::FileHandle hfile );

   protected:

      /**
       * Reload all filename items from the central directory
       * of the ZIP-file.
       *
       * @throws  GFileNotFoundException  If the file does not exist, or if
       *                                  the path is a directory rather than
       *                                  a file. <b>Note</b> that this class
       *                                  is a subclass of {@link GOpenFileException}.
       * @throws  GOpenFileException      If the file exist but cannot be
       *                                  opened or created for some
       *                                  other reason.
       * @throws  GIOException            If an I/O error occurs during reading of
       *                                  of the archive file entries.
       */
      virtual void loadItems ( bool* cancelSem, int* preLoadCounter ) const;

   public:

      /**
       * Close the specified file, as recently opened by {@link openFile}.
       * See the declaring {@link GVfs#closeFile} for details.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.09
       */
      virtual GError closeFile ( FileHandle hfile );

      /**
       * Get the current seek position of the specified file (not pipe).
       * See the declaring {@link GVfs#getFileSeekPos} for details.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.11
       */
      virtual longlong getFileSeekPos ( GVfs::FileHandle hfile, GError* err );

      /**
       * Get the size of the specified open file.
       * See the declaring {@link GVfs#getFileSize} for details.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.14
       */
      virtual longlong getFileSize ( GVfs::FileHandle hfile, GError* err );

      /**
       * Load information about the specified file into the specified object.
       * See the declaring {@link GVfs#loadFileInfo} for details.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.29
       */
      virtual GError loadFileInfo ( GVfs::FileHandle hfile, GFileItem& info );

      /**
       * Open the specified file in the VFS.
       * See the declaring {@link GVfs#openFile} for details.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.09
       */
      virtual FileHandle openFile ( const GString& path, 
                                    GError* errorCode = null,
                                    OF_Mode modeOpt = Mode_ReadOnly,
                                    OF_Create createOpt = Create_Never,
                                    OF_Share shareOpt = Share_DenyWrite,
                                    int flagsOpt = OF_FLAG_SEQUENTIAL_ACCESS,
                                    OF_ActionTaken* actionTaken = null );

      /**
       * Reads data from the specified file handle, which must have been 
       * opened for reading by {@link #openFile}, and positioned if needed.
       * See the declaring {@link GVfs#readFromFile} for details.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.09
       */
      virtual GError readFromFile ( GVfs::FileHandle hfile, 
                                    void* buff, 
                                    int numBytesToRead, 
                                    int* numBytesActuallyRead = null );

      /**
       * Set the attributes (readonly, hidden, etc.) of the specified file.
       * See the declaring {@link GVfs#setFileAttributes} for details.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.11
       */
      virtual GError setFileAttributes ( const GString& path, int attr );

      /**
       * Set the seek position of the specified file, relative to the
       * current position of the file.
       * See the declaring {@link GVfs#setFileSeekPosFromCurrent} for details.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.11
       */
      virtual GError setFileSeekPosFromCurrent ( GVfs::FileHandle hfile, 
                                                 longlong distanceToMove );

      /**
       * Set the seek position of the specified file, relative to the
       * end position of the file.
       * See the declaring {@link GVfs#setFileSeekPosFromEnd} for details.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.11
       */
      virtual GError setFileSeekPosFromEnd ( GVfs::FileHandle hfile, 
                                             longlong distanceToMove );

      /**
       * Set the seek position of the specified file, relative to the
       * current position of the file.
       * See the declaring {@link GVfs#setFileSeekPosFromStart} for details.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.11
       */
      virtual GError setFileSeekPosFromStart ( GVfs::FileHandle hfile, 
                                               longlong distanceToMove );

      /**
       * Set the size of the specified open file.
       * The file must have been opened for random write access.
       * See the declaring {@link GVfs#setFileSize} for details.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.11
       */
      virtual GError setFileSize ( GVfs::FileHandle hfile, longlong size );

      /**
       * Write attributes and time stamps of a file or directory.
       * See the declaring {@link GVfs#writeAttrAndTimes} for details.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.29
       */
      virtual GError writeAttrAndTimes ( GVfs::FileHandle hfile, 
                                         const class GFileItem& info,
                                         const GString& path = GString::Empty );

      /**
       * Writes data to the specified file handle, which must have been 
       * opened for writing by {@link #openFile}, and positioned if needed.
       * See the declaring {@link GVfs#writeToFile} for details.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.10
       */
      virtual GError writeToFile ( GVfs::FileHandle hfile, 
                                   const void* buff, 
                                   int numBytesToWrite, 
                                   int* numBytesActuallyWritten = null );
};

#endif
