/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/vfs/GVfsArchiveFile.h"
#include "glib/util/GProcessLauncher.h"
#include "glib/io/GRandomAccessFile.h"

int GVfsArchiveFile::OpenDeletionCounter = 0;
GHashtable<GInteger, GArray<GString> > GVfsArchiveFile::OpenDeletions;

GVfsArchiveFile::HDir::HDir ( const GString& dir, const GString& findPattern ) 
                      :pos(0), 
                       counter(0), 
                       dir(dir),
                       findPattern(findPattern),
                       returnedNames(null)
{
}

GVfsArchiveFile::HDir::~HDir () 
{
   delete returnedNames;
}

GVfsArchiveFile::Item::Item ( longlong headPos, longlong fileDataSize )
                      :GFileItem(), // Content is filled in upon {@link #loadItems}.
                       headPos(headPos),
                       fileDataSize(fileDataSize)
{
}

GVfsArchiveFile::Item::~Item ()
{
}

GVfsArchiveFile::GVfsArchiveFile ( GVfsLocal& localVfs,
                                   GVfs& parentVfs, 
                                   GVfs::File* vfile,
                                   const GString& fsName,
                                   char slashChar,
                                   bool supportsChangeFileNameCase,
                                   bool isFileNameCasePreserved,
                                   bool isFileNameCaseSensitive,
                                   const GString& toolPath,
                                   const GString& paramsDel )
                :GVfs(&parentVfs),
                 archiveFile(null),
                 archiveFileOpenCounter(0),
                 vfile(vfile),
                 _supportsChangeFileNameCase(supportsChangeFileNameCase),
                 _isFileNameCasePreserved(isFileNameCasePreserved),
                 _isFileNameCaseSensitive(isFileNameCaseSensitive),
                 fileSystemName(fsName),
                 toolPath(toolPath),
                 paramsDel(paramsDel),
                 localVfs(localVfs),
                 items(512, -3, false),
                 passwordProtected(false),
                 hasCalledLoadItems(false),
                 slashChar(slashChar),
                 physicalArchiveFilePath(vfile->getPhysicalPath()),
                 archiveFileItem(parentVfs, vfile->getVirtualPath())
{
}

GVfsArchiveFile::~GVfsArchiveFile ()
{
   delete vfile;
   delete archiveFile;
}

bool GVfsArchiveFile::supportsChangeFileNameCase () const 
{ 
   return _supportsChangeFileNameCase; 
}

bool GVfsArchiveFile::isFileNameCasePreserved () const 
{ 
   return _isFileNameCasePreserved; 
}

bool GVfsArchiveFile::isFileNameCaseSensitive () const
{
   return _isFileNameCaseSensitive;
}

const GString& GVfsArchiveFile::getFileSystemName () const
{
   return fileSystemName;
}

longlong GVfsArchiveFile::getFreeDriveSpace () const
{
   if (parentVfs != null)
      return parentVfs->getFreeDriveSpace();
   return 0;
}

bool GVfsArchiveFile::isSlash ( char chr ) const
{
   return chr == slashChar || GFile::IsSlash(chr);
}

GString& GVfsArchiveFile::slash ( GString& dir ) const
{
   if (dir == "")
      return dir;
   char lastChr = dir.lastChar();
   bool slashed = lastChr == slashChar || GFile::IsSlash(lastChr);
   if (!slashed)
      dir += slashChar;
   return dir;
}

const GString& GVfsArchiveFile::getSlashStr () const
{
   return GVfsArchiveFile::SlashStr;
}

bool GVfsArchiveFile::containsAnySlash ( const GString& path ) const
{
   return path.indexOf(slashChar) >= 0 ||
          (slashChar != '/' && path.indexOf('/') >= 0) || 
          (slashChar != '\\' && path.indexOf('\\') >= 0);
}

void GVfsArchiveFile::translateSlashes ( GString& path ) const
{
   if (slashChar == '/')
   {
      path.replaceAll('\\', '/');
   }
   else
   if (slashChar == '\\')
   {
      path.replaceAll('/', '\\');
   }
   else
   {
      path.replaceAll('/', slashChar);
      path.replaceAll('\\', slashChar);
   }
}

int GVfsArchiveFile::getFileAttributes ( const GString& path, GError* errorCode )
{
   GFileItem fitem;
   int fh = findFirst(fitem, path);
   if (fh == 0)
   {
      *errorCode = ERROR_FILE_NOT_FOUND;
      return GVfs::FAttrError;
   }
   findClose(fh);
   return fitem.attr;
}

GVfsArchiveFile::Item* GVfsArchiveFile::getItem ( const GString& path )
{
   GString virtualPath(GFile::MAXPATH);
   if (containsAnySlash(path))
   {
      // File name is already more or less qualified, so "walk" it
      // in order to get the fully qualified path.
      if (getWalkedPath(path, virtualPath) != GError::Ok)
         return null;
   }
   else
   {
      // Make fully qualified filename.
      virtualPath = getCurrentDirectory(true);
      virtualPath += path; 
   }
   translateSlashes(virtualPath);
   return items.get(virtualPath);
}

bool GVfsArchiveFile::isCurrentDirectoryTheRoot () const
{
   return curDir == "";
}

bool GVfsArchiveFile::isFilePreparationPossiblyLengthy () const 
{ 
   return true; 
}

GString GVfsArchiveFile::getLogicalSelfName () const 
{ 
   return archiveFileItem.getFileName(); 
}

GString GVfsArchiveFile::getPhysicalSelfName () const 
{ 
   return archiveFileItem.getFullPath(); 
}

GString GVfsArchiveFile::getRootDir () const 
{ 
   return GString::Empty; 
}

GString GVfsArchiveFile::getCurrentDirectory ( bool slash = false ) const 
{ 
   GString ret(curDir.length() + 1);
   ret = curDir;
   if (slash)
   {
      if (ret.length() > 0)
         this->slash(ret);
   }
   else
   if (GFile::IsSlash(ret.lastChar()))
   {
      ret.removeLastChar();
   }
   return ret; 
}

GError GVfsArchiveFile::setCurrentDirectory ( const GString& dir )
{
   GString slashedDir = dir;
   if (isSlashed(slashedDir))
      slashedDir.removeLastChar();

   if (slashedDir != "")
   {
      slash(slashedDir);

      int pos = 0;
      items.findEntry(slashedDir, &pos);
      if (pos < 0 || pos >= items.getCount())
         return GError::PathNotFound;

      const GString& key = items.getKey(pos);
      if (!key.beginsWith(dir))
         return GError::PathNotFound; // No such directory!
   }

   // Update the content of "curDir".
   curDir = slashedDir;
   return GError::Ok;
}

void GVfsArchiveFile::loadItems ( bool* cancelSem, int* preLoadCounter ) const 
{ 
   // Do nothing by default.
   // Subclasses override this in order to load filename items from the 
   // archive file of some specific format (ZIP, ARJ, etc.).
}

GError GVfsArchiveFile::getWalkedPath ( const GString& srcDir, GString& walkedDir ) const
{
   GString dir = srcDir;
   if (dir != "" && dir[0] != '.' && !isSlash(dir[0]))
      dir.insert(slashChar);
   GError err = GVfs::getWalkedPath(dir, walkedDir);
   if (err != GError::Ok)
      return err;
   if (isSlash(walkedDir.firstChar()))
      walkedDir.removeFirstChar();
   return GError::Ok;
}

bool GVfsArchiveFile::findFirstMightNeedLongTime () const 
{ 
   return true; 
}

int GVfsArchiveFile::findFirst ( GFileItem& fitem, 
                                 const GString& path, 
                                 bool* cancelSem, 
                                 int* preLoadCounter ) const
{
   // Make sure only one thread at a time is executing this method.
   GObject::Synchronizer synch(*this);

   fitem.clear();

   if (!hasCalledLoadItems)
   {
      hasCalledLoadItems = true;
      loadItems(cancelSem, preLoadCounter);
   }

   // --- 
   GString findDir;
   GString findPattern;
   if (path == "")
   {
      findDir = curDir;
      findPattern = "*";
   }
   else
   {
      GString walkedPath(GFile::MAXPATH);
      if (getWalkedPath(path, walkedPath) != GError::Ok)
         return 0;
      GFile f(walkedPath);
      findDir = f.getDirectory();
      findPattern = f.getFileName();
      // The search pattern "*.*" on FAT will match ALL filenames, even 
      // those with no extension. This should be true to us as well.
      if (findPattern == "" || findPattern == "*.*")
         findPattern = "*";
   }

   // Find a hdir that is not already in use.
   int hdir = 0;
   for (;;)
   {
      hdir++;
      GString key = GInteger::ToString(hdir);
      if (!hdirs.containsKey(key))
         break;
   }

   // Make sure the "findDir" contains the correct slash-characters.
   translateSlashes(findDir);

   // Store the hdir in our container so that we can access it
   // from {@link #findNext} when/if it is called with the right hdir.
   HDir* h = new HDir(findDir, findPattern);
   GString key = GInteger::ToString(hdir);
   hdirs.put(key, h);

   // Find the index of the next item of which to return by next
   // call to {@link #findNext}.
   items.findEntry(findDir, &h->pos);

   // The first item must always be the up-dir ("..") because
   // an archive file is often used as an emulation of a
   // file system directory.
   fitem.unsortedIndex = h->counter++;
   fitem.attr = GVfs::FAttrDirectory;
   fitem.timeCreate = archiveFileItem.timeCreate;
   fitem.timeWrite = archiveFileItem.timeWrite;
   fitem.timeAccess = archiveFileItem.timeAccess;
   fitem.fileSize = 0;
   fitem.fileSizeAlloc = 0;
   fitem.path = "..";

   // Set up the bag of where to track the name of which items we returns.
   bool ignoreCase = !isFileNameCaseSensitive();
   h->returnedNames = new GKeyBag<GString>(256, ignoreCase);
   h->returnedNames->put(fitem.path, const_cast<GString*>(&GString::Empty), false);

   // Return the ".." item if requested.
   bool caseSen = isFileNameCaseSensitive();
   if (match(fitem.path, h->findPattern, caseSen))
      return hdir;

   // The ".." item was not requested, so return the next matching filename.
   if (findNext(hdir, fitem))
      return hdir;

   // No matchin item, so close the handle and return zero.
   findClose(hdir);
   return 0;
}

bool GVfsArchiveFile::findNext ( int hdir, GFileItem& fitem ) const
{
   fitem.clear();

   // Find the hdir object.
   GString key = GInteger::ToString(hdir);
   HDir* h = hdirs.get(key);
   if (h == null)
      return false;

   // Make sure not to scan outside items range.
   if (h->pos < 0 || h->pos >= items.getCount())
      return false;

   // ---
   bool caseSen = isFileNameCaseSensitive();
   for (int i=h->pos, num=items.getCount(); i<num; i++)
   {
      const GFileItem& next = items.getIndexedItem(i);
      GString nextDir = next.path.getDirectory();
      if (!nextDir.beginsWith(h->dir))
         break;

      // ---
      nextDir.remove(0, h->dir.length()); 
      if (GFile::IsSlash(nextDir[0]))
         nextDir.removeFirstChar();
      int slashPos = -1;
      for (int ii=0, len=nextDir.length(); ii<len; ii++)
      {
         if (!GFile::IsSlash(nextDir[ii]))
            continue;
         slashPos = ii;
         break;
      }
      if (slashPos >= 0)
         nextDir.cutTailFrom(slashPos);

      // ---
      if (nextDir == GString::Empty)
      {
         const GString& name = next.getName();
         const GString& ext = next.getExtension();
         if (name == GString::Empty && ext == GString::Empty)
            continue;
         fitem.setName(name);
         fitem.setExtension(ext);
         fitem.attr = GVfs::FAttrArchive;
         fitem.fileSize = next.fileSize;
         fitem.fileSizeAlloc = next.fileSizeAlloc;
      }
      else
      {
         if (h->returnedNames->containsKey(nextDir))
            continue;
         GString file;
         GString ext;
         GFile::SplitPath(nextDir, null, null, &file, &ext);
         fitem.setName(file);
         fitem.setExtension(ext);
         fitem.attr = GVfs::FAttrArchive | GVfs::FAttrDirectory;
         fitem.fileSize = 0;
         fitem.fileSizeAlloc = 0;
      }

      // ---
      fitem.unsortedIndex = h->counter++;
      fitem.timeCreate = next.timeCreate;
      fitem.timeWrite = next.timeWrite;
      fitem.timeAccess = next.timeAccess;

      // Track all filenames that we returns.
      GString fitemName = fitem.getFileName();
      h->returnedNames->put(fitemName, const_cast<GString*>(&GString::Empty), false);

      // Update the position, so that the next call to {@link #findNext}
      // will return the next item.
      h->pos = i + 1;

      // We have now found the next filename item within actual directory.
      // However, if this filename does not match the pattern specified 
      // to "findFirst()" then skip it and find the next one instead.
      if (!match(fitemName, h->findPattern, caseSen))
         continue;

      // ---
      return true;
   }

   // ---
   h->pos = items.getCount();
   return false;
}

void GVfsArchiveFile::findClose ( int hdir ) const
{
   if (hdir == 0)
      return;
   GString key = GInteger::ToString(hdir);
   hdirs.remove(key);
}

GRandomAccessFile& GVfsArchiveFile::openArchiveFile () const
{
   synchronized (archiveFileAccessLock)
   {
      archiveFileOpenCounter++;
      if (archiveFile != null)
         return *archiveFile;
      archiveFile = new GRandomAccessFile(localVfs, physicalArchiveFilePath, "r", false);
      return *archiveFile;
   } synchronized_end;
}

void GVfsArchiveFile::closeArchiveFile () const
{
   synchronized (archiveFileAccessLock)
   {
      if (archiveFile == null)
         return;
      if (archiveFileOpenCounter > 0)
         archiveFileOpenCounter--;
      if (archiveFileOpenCounter == 0)
      {
         delete archiveFile; // Close the archive file.
         archiveFile = null;
      }
   } synchronized_end;
}

GArray<GString>* GVfsArchiveFile::getOpenDeletion ( GVfs::DeletionHandle hdel )
{
   synchronized (OpenDeletions) 
   {
      GInteger key(static_cast<int>(hdel));
      return OpenDeletions.get(key);
   } synchronized_end;
}

GVfs::DeletionHandle GVfsArchiveFile::openDeletion ()
{
   synchronized (OpenDeletions) 
   {
      int hdel;
      for (;;)
      {
         OpenDeletionCounter++;
         if (OpenDeletionCounter < 1)
            OpenDeletionCounter = 1;
         GInteger key(OpenDeletionCounter);
         if (!OpenDeletions.containsKey(key))
         {
            hdel = OpenDeletionCounter;
            break;
         }
      }

      hdel = static_cast<GVfs::DeletionHandle>(hdel);
      GInteger* key = new GInteger(hdel);
      GArray<GString>* itemList = new GArray<GString>();
      OpenDeletions.put(key, itemList, true, true);
      return hdel;
   } synchronized_end;
}

GError GVfsArchiveFile::closeDeletion ( GVfs::DeletionHandle hdel )
{
   synchronized (OpenDeletions) 
   {
      GArray<GString>* delItems = getOpenDeletion(hdel);
      if (delItems == null)
         return GError::InvalidHandle;
      GInteger key(static_cast<int>(hdel));
      OpenDeletions.remove(key);
   } synchronized_end;
   return GError::Ok;
}

/**
 * The class used to launch the archiver tool (e.g. ZIP.EXE, ARJ.EXE, etc.).
 *
 * @author  Leif Erik Larsen
 * @since   2005.09.01
 */
class ToolLauncher : public GProcessLauncher
{

public:

   GObject lock;

   ToolLauncher ( const GString& progName, const GString& params )
      :GProcessLauncher(GString::Empty, progName, params)
   {
   }

   virtual ~ToolLauncher ()
   {
   }

private:

   /** Disable the copy-constructor. */
   ToolLauncher ( const ToolLauncher& src )
      :GProcessLauncher(GString::Empty, GString::Empty)
   {
   }

   /** Disable the assignment operator. */
   ToolLauncher& operator= ( const ToolLauncher& ) { return *this; }

protected:

   virtual void run ()
   {
      try {
         GProcessLauncher::run();
         lock.notifyAll();
      } catch (...) {
         lock.notifyAll();
         throw;
      }
   }

public:

   virtual void waitForTheChildProcessToFinish ()
   {
      // Reduce the priority of this thread somewhat, to prevent the 
      // GUI-thread from being CPU-hogged.
      setPriority(GThread::PRIORITY_NORMAL_MINUS);

      // Read pipe data and append to the console monitor.
      const int maxText = 1024;
      GString text(maxText);
      bool anyData = true;
      for (;;)
      {
         bool stillRunning = isRunningChild();
         if (!(anyData || stillRunning))
         {
            if (childStdOut->peekBytesAvailable() <= 0 &&
               childStdErr->peekBytesAvailable() <= 0)
            {
               break;
            }
            anyData = true;
         }

         if (!anyData)
            GThread::Sleep(50);

         anyData = false; // Until the opposite has been proven.

         // Read text from childs STDOUT and append it to the console monitor.
         childStdOut->readUnblocked(text, maxText);
         if (text.length() > 0)
         {
            anyData = true;
            GProgram::GetProgram().printF(text);
         }

         // Read text from childs STDERR and append it to the console monitor.
         childStdErr->readUnblocked(text, maxText);
         if (text.length() > 0)
         {
            anyData = true;
            GProgram::GetProgram().printF(text);
         }
      }
   }
};

GError GVfsArchiveFile::performDeletion ( DeletionHandle hdel )
{
   // Make sure only one thread can perform read/write operations 
   // on the physical archive file at once.
   GObject::Synchronizer synch(archiveFileAccessLock);

   // We cannot perform the deletion if the physical archive file 
   // is open by some other operation.
   if (archiveFile != null)
      return GError::AccessDenied;

   // Create the temporary file containing a linefeed separated list 
   // of which archive file enries to delete.
   GVfsLocal localVfs;
   GString delItemsPath = localVfs.createTemporaryFile("lcmd");
   GFileOutputStream os(localVfs, delItemsPath, true, true, true);
   GArray<GString>* delItems = getOpenDeletion(hdel);
   for (int i=0, num=delItems->getCount(); i<num; i++)
   {
      GString& itm = delItems->get(i);
      os.printf("%s\n", GVArgs(itm));
   }
   os.close();

   // ---
   GString params(paramsDel, GVArgs(physicalArchiveFilePath).add(delItemsPath));
   ToolLauncher tool(toolPath, params);

   // Run the ZIP-program in order to perform the ZIP-operation.
   GProgram::GetProgram().printF("\"%s\" %s\n", GVArgs(toolPath).add(params));
   tool.start();
   tool.lock.wait(); // Wait for the tool to finish in a thread safe manner.

   // Remove the temporary deletion-list file.
   localVfs.removeFile(0, delItemsPath, false);

   // Make sure the launcher thread has exited before we return.
   // This is in order not to destroy the launcher object before the
   // thread has exited (or else we might get some strange exceptions,
   // such as pure virtual function calls, etc.).
   while (tool.isRunning())
      GThread::Sleep(25);

   // Reload archive file entries.
   loadItems(null, null);

   // Return.
   GProcessLauncher::ErrorCode err = tool.getErrorCode();
   if (err != GProcessLauncher::EC_NoError)
   {
      GError ret(GError::BadCommand);
      ret.errorDetails = toolPath + " " + params;
      return ret;
   }
   int exitCode = tool.getExitCodeFromprocess();
   if (exitCode == 0)
      return GError::Ok;
   GError ret(GError::FunctionFailed);
   ret.errorDetails = toolPath + " " + params;
   return ret;
}

GError GVfsArchiveFile::removeFileOrDirectory ( DeletionHandle hdel, 
                                                const GString& path, 
                                                bool isdir )
{
   if (hdel == 0)
   {
      // The caller want us to perform the deletion directly.
      // This is done here simply by calling our self reqursively
      // using a new local temporary deletion handle.
      DeletionHandle hdel_ = openDeletion();
      GVfs::AutoDeletionHandleCloser acloser(*this, hdel_, false);
      GError err = removeFileOrDirectory(hdel_, path, isdir);
      if (err == GError::Ok)
         err = performDeletion(hdel_);
      return err;
   }

   // ---
   GArray<GString>* delItems = getOpenDeletion(hdel);
   if (delItems == null)
      return GError::InvalidHandle;

   GString entryPath;
   if (isdir)
   {
      GError err = getWalkedPath(path, entryPath);
      if (err != GError::Ok)
         return err;
      slash(entryPath);
   }
   else
   {
      Item* item = getItem(path);
      if (item == null)
         return GError::FileNotFound;
      entryPath = item->getFullPath();
   }

   delItems->add(new GString(entryPath));
   items.remove(entryPath);
   return GError::Ok;
}

GError GVfsArchiveFile::removeFile ( DeletionHandle hdel, 
                                     const GString& path, 
                                     bool /*trashCan*/ )
{
   bool isdir = false;
   return removeFileOrDirectory(hdel, path, isdir);
}

GError GVfsArchiveFile::removeDirectory ( DeletionHandle hdel, 
                                          const GString& path, 
                                          bool /*trashCan*/, 
                                          HWND /*ownerWin*/ )
{
   bool isdir = true;
   return removeFileOrDirectory(hdel, path, isdir);
}
