/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/vfs/GFileItem.h"
#include "glib/vfs/GVfsLocal.h"
#include "glib/io/GFileNotFoundException.h"
#include "glib/util/GDriveInfo.h"
#include "glib/sys/GSystem.h"

GFileItem::GFileItem ()
          :unsortedIndex(0),
           attr(GVfs::FAttrNormal),
           fileSize(0),
           fileSizeAlloc(0)
{
}

GFileItem::GFileItem ( const GFileItem& src )
          :unsortedIndex(src.unsortedIndex),
           attr(src.attr),
           timeCreate(src.timeCreate),
           timeWrite(src.timeWrite),
           timeAccess(src.timeAccess),
           fileSize(src.fileSize),
           fileSizeAlloc(src.fileSizeAlloc),
           path(src.path)
{
}

GFileItem::GFileItem ( const FILEFINDBUF3& src )
          :unsortedIndex(0),
           attr(src.attrFile),
           timeCreate(src.fdateCreation, src.ftimeCreation),
           timeWrite(src.fdateLastWrite, src.ftimeLastWrite),
           timeAccess(src.fdateLastAccess, src.ftimeLastAccess),
           fileSize(src.cbFile),
           fileSizeAlloc(src.cbFileAlloc),
           path(GString::Empty, src.achName)
{
}

GFileItem::GFileItem ( const FILEFINDBUF3L& src )
          :unsortedIndex(0),
           attr(src.attrFile),
           timeCreate(src.fdateCreation, src.ftimeCreation),
           timeWrite(src.fdateLastWrite, src.ftimeLastWrite),
           timeAccess(src.fdateLastAccess, src.ftimeLastAccess),
           fileSize(src.cbFile),
           fileSizeAlloc(src.cbFileAlloc),
           path(GString::Empty, src.achName)
{
}

GFileItem::GFileItem ( GVfs& vfs, const GString& path_ )
          :unsortedIndex(0),
           attr(GVfs::FAttrNormal),
           fileSize(0),
           fileSizeAlloc(0),
           path() // Don't initialize yet, since vfs.findFirst() will overwrite!
{
   int h = vfs.findFirst(*this, path_);
   if (h == 0)
      gthrow_(GFileNotFoundException(path_));
   vfs.findClose(h);
   GString fnameWithCorrectCase = getFileName();
   this->path.setFullPath(path_); // Set the directory + maybe incorrect cased filename.
   this->path.setFileName(fnameWithCorrectCase); // Set the correctly cased filename.
}

GFileItem::~GFileItem ()
{
}

const GFileItem& GFileItem::operator= ( const FILEFINDBUF3& src )
{
   this->unsortedIndex = 0;
   this->attr = src.attrFile;
   this->timeCreate = GTimeStamp(src.fdateCreation, src.ftimeCreation);
   this->timeWrite = GTimeStamp(src.fdateLastWrite, src.ftimeLastWrite);
   this->timeAccess = GTimeStamp(src.fdateLastAccess, src.ftimeLastAccess);
   this->fileSize = src.cbFile;
   this->fileSizeAlloc = src.cbFileAlloc;
   this->path = src.achName;
   return *this;
}

const GFileItem& GFileItem::operator= ( const FILEFINDBUF3L& src )
{
   this->unsortedIndex = 0;
   this->attr = src.attrFile;
   this->timeCreate = GTimeStamp(src.fdateCreation, src.ftimeCreation);
   this->timeWrite = GTimeStamp(src.fdateLastWrite, src.ftimeLastWrite);
   this->timeAccess = GTimeStamp(src.fdateLastAccess, src.ftimeLastAccess);
   this->fileSize = src.cbFile;
   this->fileSizeAlloc = src.cbFileAlloc;
   this->path = src.achName;
   return *this;
}

bool GFileItem::operator== ( const GFileItem& src ) const
{
   if (attr != src.attr)
      return false;
   if (timeCreate != src.timeCreate)
      return false;
   if (timeWrite != src.timeWrite)
      return false;
   if (timeAccess != src.timeAccess)
      return false;
   if (fileSize != src.fileSize)
      return false;
   if (path != src.path)
      return false;
   // Yes, ignore value of "unsortedIndex" and "fileSizeAlloc"!
   return true;
}

bool GFileItem::operator!= ( const GFileItem& src ) const
{
   return !operator==(src);
}

void GFileItem::clear ()
{
   unsortedIndex = 0;
   attr = GVfs::FAttrNormal;
   timeCreate.clear();
   timeWrite.clear();
   timeAccess.clear();
   fileSize = 0;
   fileSizeAlloc = 0;
   path = "";
}

void GFileItem::setDirectoryFlag ( bool on )
{
   if (on)
      attr |= GVfs::FAttrDirectory;
   else
      attr &= ~GVfs::FAttrDirectory;
}

void GFileItem::setArchiveFlag ( bool on )
{
   if (on)
      attr |= GVfs::FAttrArchive;
   else
      attr &= ~GVfs::FAttrArchive;
}

void GFileItem::setSystemFlag ( bool on )
{
   if (on)
      attr |= GVfs::FAttrSystem;
   else
      attr &= ~GVfs::FAttrSystem;
}

void GFileItem::setHiddenFlag ( bool on )
{
   if (on)
      attr |= GVfs::FAttrHidden;
   else
      attr &= ~GVfs::FAttrHidden;
}

void GFileItem::setReadOnlyFlag ( bool on )
{
   if (on)
      attr |= GVfs::FAttrReadOnly;
   else
      attr &= ~GVfs::FAttrReadOnly;
}

bool GFileItem::isProgramFileName () const
{
   const GString& fext = path.getExtension();
   return fext.equalsIgnoreCase(".exe") ||
          fext.equalsIgnoreCase(".com") ||
          fext.equalsIgnoreCase(".cmd") ||
          fext.equalsIgnoreCase(".bat");
}

bool GFileItem::isZipOrArchiveFile () const
{
   GString f = path.getFullPath();
   return IsZipOrArchiveFile(f);
}

bool GFileItem::IsZipOrArchiveFile ( const GString& path )
{
   GFile f(path);
   const GString& fext = f.getExtension();
   return fext.equalsIgnoreCase(".zip") || // Standard ZIP file.
          fext.equalsIgnoreCase(".jar") || // Java ARchive (ZIP format).
          fext.equalsIgnoreCase(".war");   // Web ARchive (ZIP format).
}

GString GFileItem::getDirectory () const
{
   if (path.containsDirectory())
      return path.getDrive() + path.getDirectory();
   else
      return GString::Empty;
}

GString GFileItem::getFullPath () const
{
   return path.getFullPath();
}

bool GFileItem::isDirectory () const 
{ 
   return (attr & GVfs::FAttrDirectory) != 0; 
}

bool GFileItem::isArchive () const 
{ 
   return (attr & GVfs::FAttrArchive) != 0; 
}

bool GFileItem::isSystem () const 
{ 
   return (attr & GVfs::FAttrSystem) != 0; 
}

bool GFileItem::isHidden () const 
{ 
   return (attr & GVfs::FAttrHidden) != 0; 
}

bool GFileItem::isReadOnly  () const 
{ 
   return (attr & GVfs::FAttrReadOnly) != 0; 
}

bool GFileItem::isThisDir () const 
{ 
   return path.isThisDir(); 
}

bool GFileItem::isUpDir () const 
{ 
   return path.isUpDir(); 
}

GString GFileItem::getFileName () const 
{ 
   return path.getFileName(); 
}

const GString& GFileItem::getName () const 
{ 
   return path.getName();
}

const GString& GFileItem::getExtension () const 
{ 
   return path.getExtension(); 
}

void GFileItem::setName ( const GString& fname ) 
{ 
   path.setName(fname); 
}

void GFileItem::setExtension ( const GString& fext ) 
{ 
   path.setExtension(fext); 
}
