/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_FILE
#define __GLIB_FILE

#include "glib/primitives/GString.h"
#include "glib/util/GArray.h"

/**
 * This class is used to represent a single filename with or
 * without its drive and/or directory specification.
 *
 * <b>Mark</b> that the two special filanames "." and ".." will be parsed
 * specially by this class. They will be stored in the <i>fname</i> variable,
 * meaning that even if they both contains a dot they will not be treated
 * as the extension part of the filename, for instance by the
 * {@link #splitPath} method.
 *
 * In addition, this class contains a number of static methods that naturally
 * belongs to this class, such as for instance {@link #existFile},
 * {@link #SetCurrentDir} and a number of other file related methods.
 *
 * @since  1999.09.05
 * @author Leif Erik Larsen
 */
class GFile : public GObject
{
   public:

      enum { MAXPATH = 256 };

      /** The character set that is guaranteed to be supported on all FS's. */
      static const GString LegalChars;
      static const GString LegalFNChars;

   private:

      GString drive;
      GString dir;
      GString fname;
      GString fext;

      /**
       * Inner class used to maintain a list of the current directory of 
       * each of the available drives on the system. This is needed because 
       * Win32 maintains the current directory for the current drive only.
       * That is; the "current drive" is part of the "current directory"
       * it self. But we need to support the philosophy of a system that 
       * keeps track of the current directory on each drive individually.
       *
       * @author  Leif Erik Larsen
       * @since   2004.05.04
       */
      class CurDirs : public GArray<GString>
      {
         friend class GFile;
         CurDirs ();
         virtual ~CurDirs ();
         /** @param drive A=0, B=1, C=2, etc. */
         static GString GetSystemDefaultCurrentDriveAndDir ( int drive );
      };

      /** 
       * Holds the current directory for each drive. 
       * Initialized by the constructor of {@link GProgram}. 
       *
       * @author  Leif Erik Larsen
       * @since   2004.05.04
       */
      static CurDirs CurDirForEachDrive;

   public:

      /**
       * The directory separator (slash) character for the current
       * underlying system, as a character.
       */
      static const char SlashChar;

      /**
       * The directory separator (slash) character for the current
       * underlying system, as a string.
       */
      static const GString SlashStr;

   public:

      GFile ();
      GFile ( const GFile& path );
      explicit GFile ( const GString& path );
      GFile ( const GString& dir, const GString& fileName );

      virtual ~GFile ();

   public:

      const GFile& operator= ( const GFile& src );
      const GFile& operator= ( const GString& src );

      /**
       * Return true if and only if all parts of the specified GFile object
       * are equal to "this", respecting the case sensibility of the
       * underlying file system.
       *
       * @see #isCaseSensitiveFS
       */
      const bool operator== ( const GFile& src ) const;
      const bool operator== ( const GString& src ) const;
      const bool operator!= ( const GFile& src ) const { return !operator==(src); };
      const bool operator!= ( const GString& src ) const { return !operator==(src); };

      /**
       * Call {@link #toString} automatically when code needs a <i>GString</i>
       * instead of a <i>GFile</i>.
       */
      operator const GString () const;

   public:

      bool containsDrive () const;
      bool containsDirectory () const;
      bool containsName () const;
      bool containsExtension () const;
      bool containsFileName () const;

      static bool ContainsDrive ( const GString& path );
      static bool ContainsDirectory ( const GString& path );
      static bool ContainsName ( const GString& path );
      static bool ContainsExtension ( const GString& path );
      static bool ContainsFileName ( const GString& path );

      /**
       * Cut the file name, if any, from the specified path.
       *
       * @return  A reference to the same string object as the specified path.
       */
      static GString& CutFileName ( GString& path );

      /**
       * Compare the two files, byte by byte.
       *
       * This is the brute force method, returning false as soon as the
       * first byte difference is found.
       *
       * Return true if and only if the two files are 100% identical, with
       * respect to the contents. OS/2 Extended Attributes are completely
       * ignored by this test.
       *
       * @author  Leif Erik Larsen
       * @since   2005.08.31
       * @param   vfs1    The VFS of where to read file specified by path1.
       * @param   path1   The path to the first file to read and compare.
       * @param   vfs2    The VFS of where to read file specified by path2.
       * @param   path1   The path to the second file to read and compare.
       * @throws  GIOException in case of any error. Possible error is likely
       *                       to be that one of the files couldn't be opened
       *                       in binary read-only mode.
       */
      static bool FilesEquals ( class GVfs& vfs1, const GString& path1, 
                                class GVfs& vfs2, const GString& path2 );

      /**
       * Get the current drive and directory as a standard FS string.
       * For instance: "C:\My\Directory" or "C:\".
       *
       * The returned directory will always have a trailing slash if it 
       * is the root directory that is current.
       *
       * In case of any error, the corresponding error code is thrown.
       *
       * @author  Leif Erik Larsen
       * @since   2003.01.26
       * @param   drive     The drive of which to get current directory,
       *                    where 0=Current, 1=A:, 2=B, 3=C:, etc.
       * @return  A copy of the current directory for the specified drive.
       * @see     #SetCurrentDir
       * @throws  GIllegalArgumentException if the soecified drive is 
       *                    not a valid drive specification.
       * @throws  APIRET    In case of any system dependent error.
       */
      static GString GetCurrentDir ( int drive = 0 ); // throws APIRET

      /**
       * Get the currently active logical drive, where 1=A:, 2=B:, 3=C:, etc.
       * In case of any error, the corresponding error code is thrown.
       */
      static int GetCurrentDrive (); // throws APIRET

      /**
       * Get the drive number of the path, where 1=A, 2=B, 3=C, etc.
       * If drive is unspecified or unknown then we will return 0 (current
       * or default drive, that is).
       */
      int getDriveNum () const;

      /**
       * Same as {@link #getDriveNum}, but for the specified path.
       *
       * @author  Leif Erik Larsen
       * @since   2004.08.12
       */
      static int GetDriveNum ( const GString& path );

      const GString& getDrive () const;
      GString getDirAndFileName ( bool skipInitialSlash = false ) const;
      const GString& getDirectory () const;
      const GString& getName () const;
      const GString& getExtension () const;
      GString getFileName () const;
      GString getFullPath () const;

      static GString GetDrive ( const GString& path );
      static GString GetDirectory ( const GString& path );
      static GString GetName ( const GString& path );
      static GString GetExtension ( const GString& path );

      /**
       * Get the filename+extension part of the specified path.
       */
      static GString GetFileName ( const GString& path );

      /**
       * Return true if and only if the underlying system has case sensitive
       * file systems. In this case, "case sensitive" means that two filenames
       * can have the same name within the same directory, as long as the two
       * filenames does not have the same character case(s). This is true for
       * most Linux/UNIX systems. Else we will return false, such as for the
       * file systems of Windows and OS/2.
       */
      static bool IsCaseSensitiveFS ();

      bool isDirectoryFromRoot () const;

      /**
       * Check if the item is the "." item.
       */
      static bool IsThisDir ( const char* str );
      static bool IsThisDir ( const GString& str );
      bool isThisDir () const;

      /**
       * Check if the item is the ".." item.
       */
      static bool IsUpDir ( const char* str );
      static bool IsUpDir ( const GString& str );
      bool isUpDir () const;

      /**
       * Test if specified filename does match the specified filter,
       * which may contain one or more wildcards.
       */
      static bool IsFileNameInFilter ( const GString& fileName, 
                                       const GString& filter );

      /**
       * Scan specified comma-separated filter list to see if specified
       * filename does match any of them.
       */
      static bool IsFileNameInFilterList ( const GString& fileName, 
                                           const GString& filterList, 
                                           char sep = ';' );

      /**
       * Return true if the specified path is to a file which name means
       * that it is likely to be an executable program file,
       * from the OS point of view.
       *
       * @author  Leif Erik Larsen
       * @since   2003.01.13
       */
      static bool IsProgramFileName ( const GString& path );

      /**
       * Return true if the specified path is to a file which name means
       * that it is likely to be a script file that is executable by
       * the OS dependent shell. On OS/2 and Windows NT this is
       * true if the file extension is ".cmd" or ".bat".
       *
       * @author  Leif Erik Larsen
       * @since   2003.01.13
       */
      static bool IsShellScriptFileName ( const GString& path );

      /**
       * Return true if and only if the specified character is a
       * directory separator (slah-) character.
       *
       * @author  Leif Erik Larsen
       * @since   2004.03.26
       * @see     #Slash
       * @see     #IsSlashed
       */
      static bool IsSlash ( char chr );

      /**
       * Return true if and only if the specified directory is terminated
       * with a directory separator (slah-) character.
       *
       * @author  Leif Erik Larsen
       * @since   2004.03.26
       * @see     #Slash
       * @see     #IsSlash
       */
      static bool IsSlashed ( const GString& dir );

      /**
       * Walk the specified path and return a copy that is correct with
       * respect to the character case on the underlying file system.
       * This method makes sense only on file systems that preserves
       * character case without actually being case sensitive.
       * Examples of such file systems are FAT32, NTFS and HPFS.
       *
       * @author  Leif Erik Larsen
       * @since   2004.05.04
       * @return  A copy of the specified path, but with correct 
       *          character case. In case of any error we will return 
       *          a string that is exactly equal to the specified path.
       */
      static GString GetCorrectFilenameCase ( const GString& path );

      /**
       * If we return false then the caller can assume that we have not tauched
       * the <i>obj</i> parameter value.
       */
      static bool LocateExecutableFileInDir ( const GString& dir, 
                                              const GString& fileName, 
                                              GFile& obj );

      bool removeEA ( const GString& eaname ) const;

      /**
       * Set the current drive and/or directory.
       *
       * The dir can be with or without drive specification.
       * If it has no drive then we will change the directory of the
       * current default drive of the program. If it has a drive
       * specification then that drive will be set to be the default
       * disk of the program, and the directory is of course also
       * activated.
       *
       * @author  Leif Erik Larsen
       * @since   2003.01.26
       * @return  The system dependent error code in case of any error,
       *          or NO_ERRRO (zero) on success.
       * @see     #GetCurrentDir
       */
      static APIRET SetCurrentDir ( const GString& dir );

      /**
       * Set the drive, where 1=A:, 2=B:, 3=C:, etc.
       * Giving drive zero means "No drive specification" or "Current Drive".
       */
      void setDrive ( int drive );

      /**
       * Set drive. The specified drive must have the format "X:", withount
       * any slash character.
       */
      void setDrive ( const GString& drive );

      /**
       * Set the directory part of the path.
       */
      void setDirectory ( const GString& dir );

      void setName ( const GString& fname );
      void setExtension ( const GString& fext );

      /**
       * Set the file name and extension only of the path, from the filename
       * part of the specified path. The specified path can be a full path or
       * just the filename part of a path.
       */
      void setFileName ( const GString& path );

      void setFullPath ( const GString& path );

      /**
       * Append a slash character to the specified path, if needed.
       *
       * A slash will not be appended if the path specified is empty,
       * or if the path specified already has a trailing slash.
       *
       * @author  Leif Erik Larsen
       * @since   2004.03.26
       * @return  A reference to the same string object as specified.
       * @see     #IsSlashed
       * @see     #IsSlash
       * @see     GVfs#slash
       */
      static GString& Slash ( GString& path );

      /**
       * Split the specified path into its components.
       *
       * <b>Mark</b> that the two special filanames "." and ".." will be
       * parsed specially by this method. They will be stored in the
       * <i>file</i> parameter, meaning that even if they both contains a
       * dot they will not be treated as the extension part of the filename.
       *
       * @param   path    The path of which to split.
       * @param   drive   Drive part, e.g. "C:".
       * @param   dir     Dir part, e.g. "\TEST\DIR\".
       * @param   file    File part, e.g. "NAME" and "..".
       * @param   ext     Extention part, e.g. ".EXT".
       */
      static void SplitPath ( const GString& path, 
                              GString* drive = null, 
                              GString* dir = null, 
                              GString* file = null, 
                              GString* ext = null );

      /**
       * Returns the full path of the file.
       * Same as {@link #getFullPath}.
       */
      GString toString () const;
};

#endif
