/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_VECTOR
#define __GLIB_VECTOR

#include "glib/util/GComparator.h"
#include "glib/util/GArrayIndexOutOfBoundsException.h"

/**
 * Dynamic array for objects that are not allocated on the heap.
 * Contained elements can be of any type, including primitives,
 * as long as they support the following operators;<br>
 * <code>operator==()</code><br>
 * <code>operator!=()</code><br>
 *
 * However, for the vector to be sortable by {@link #sortItems}
 * the contained elements must be {@link GObject}s.
 *
 * @author  Leif Erik Larsen
 * @since   2001.03.06
 * @see     GFixedVector
 * @see     GArray
 */
template <class T> class GVector : public GObject
{
   private:

      /** Current number of items in the vector. */
      int num;

      /** Current allocated size of the vector array. */
      int size;

      /** Initial size of the vector array. */
      int initial;

      /** The array grow factor, by number (if positive) or by factor (if negative). A default grow size will be used if this is zero. */
      int incremental;

      /** The array it self, containing items by value. */
      T* items;

   public:

      /**
       * Construct a new vector with the specified initial and incremental
       * size.
       */
      explicit GVector ( int initial = 64, int incremental = -3 )
      {
         this->num = 0;
         this->initial = (initial > 0 ? initial : 64);
         this->size = this->initial;
         this->incremental = incremental;
         this->items = new T[unsigned(this->size)];
      }

      /**
       * Copy constructor.
       */
      GVector ( const GVector<T>& src )
      {
         num = src.num;
         size = src.size;
         initial = src.initial;
         incremental = src.incremental;
         items = new T[unsigned(size)];

         // Copy the new content.
         for (int i=0; i<num; i++)
            items[i] = src.items[i];
      }

      /**
       * Destruct the vector and all of its elements.
       */
      virtual ~GVector ()
      {
         destruct();
      }

   private:

      /**
       * Destruct all elements, and delete the element array it self.
       */
      void destruct ()
      {
         delete [] items;
         items = null;
         num = 0;
      }

   public:

      GVector<T>& operator= ( const GVector<T>& src )
      {
         // Remove the old content.
         destruct();

         // Prepare for the new content.
         num = src.num;
         size = src.size;
         initial = src.initial;
         incremental = src.incremental;
         items = new T[unsigned(size)];

         // Copy the new content.
         for (int i=0; i<num; i++)
            items[i] = src.items[i];

         return *this;
      }

      bool operator== ( const GVector<T>& src ) const
      {
         if (num != src.num)
            return false;
         for (int i=0; i<num; i++)
            if (items[i] != src.items[i])
               return false;
         return true;
      }

      bool operator!= ( const GVector<T>& src ) const
      {
         return !operator==(src);
      }

      /**
       * Get a reference to the indexed element.
       */
      T& operator[] ( int index ) const throw(GArrayIndexOutOfBoundsException)
      {
         if (index < 0 || index >= num)
            gthrow_(GArrayIndexOutOfBoundsException());
         return items[index];
      }

   public:

      /**
       * Alternative method that can be used instead of {@link #operator[]}.
       */
      T& elementAt ( int index ) const throw(GArrayIndexOutOfBoundsException)
      {
         return operator[](index);
      }

      /**
       * Extend the vector with as many additional items as was set by the
       * 'incremental' parameter.
       */
      void enlarge ()
      {
         int newsize = size;
         if (incremental < 0)
            newsize += (size / ((-1) * incremental)) + 1;
         else
         if (incremental == 0)
            newsize += 64;
         else
            newsize += incremental;

         T* tmp = new T[unsigned(newsize)];
         for (int i=0; i<num; i++)
            tmp[i] = items[i];
         delete [] items; // Delete the old items and the old array it self.
         items = tmp;
         size = newsize;
      }

      /**
       * If the current capacity is less than the specified count then
       * reallocate space to make room for exactly the specified count of items.
       */
      void ensureCapacity ( int count )
      {
         if (count > size)
         {
            T* tmp = new T[unsigned(count)];
            for (int i=0; i<num; i++)
               tmp[i] = items[i];
            delete [] items; // Delete the old items and the old array it self.
            items = tmp;
            size = count;
         }
      }

      /**
       * Add another item to the end of the vector.
       * Extends the vector if necessary.
       */
      void add ( const T& anItem )
      {
         if (num >= size)
            enlarge();
         items[num++] = anItem;
      }

      /**
       * Get the index of the first item in the vector that is equal to the
       * specified item, or -1 if it does not exist in the vector.
       */
      int indexOf ( const T& anItem ) const
      {
         for (int i=0; i<num; i++)
            if (items[i] == anItem)
               return i;
         return -1;
      }

      /**
       * Returns true if and only if the specified item is contained 
       * in the vector, using <code>operator==()</code>.
       */
      bool contains ( const T& anItem ) const
      {
         return indexOf(anItem) >= 0;
      }

      /**
       * Insert another item just above the indexed item in the vector.
       * Extends the vector if necessary.
       */
      void insert ( const T& anItem, int index )
      {
         if (num >= size)
            enlarge();

         if (index < num) // If there are any tail to pull at all
         {
            for (int i=num; i>index; i--)
               items[i] = items[i-1];
         }

         items[index] = anItem;
         num++;
      }

      /**
       * Remove all items from the vector without changing the current
       * allocated size of the vector.
       */
      void clear ()
      {
         for (int i=0; i<num; i++)
            items[i] = T();
         num = 0;
      }

      /**
       * Remove the specified number of items from the vector, starting with
       * the item at the indexed position.
       *
       * @param  index  The index of the first item of which to remove.
       * @param  count  The number of items of which to remove.
       */
      void remove ( int index, int count = 1 )
      {
         if (count <= 0)
            return;

         if (index + count > num)
         {
            count = num - index;
            if (count <= 0)
               return;
         }

         // Displace the tail to overwrite the removed items.
         int i;
         for (i=index+count; i<num; i++)
            items[i-count] = items[i];

         int prevNum = num;
         num -= count;

         // Set items in the tail to their default value.
         for (i=num; i<prevNum; i++)
            items[i] = T();
      }

      /**
       * Remove all items from the vector and shrink the array buffer
       * to its initial size.
       */
      void removeAll ()
      {
         num = 0;
         size = initial;
         delete [] items;
         items = new T[unsigned(size)];
      }

      /**
       * Return the current number of items that are contained in the vector.
       */
      int getCount () const
      {
         return num;
      }

      /**
       * Return  True if and only if the vector contain no items at all,
       *         or else return false.
       */
      bool isEmpty () const
      {
         return num <= 0;
      }

      /**
       * Return the allocated size of the vector.
       */
      int getAllocatedSize () const
      {
         return size;
      }

      /**
       * Swap the two indexed items.
       */
      void swap ( int idx1, int idx2 )
      {
         T tmp = items[idx1];
         items[idx1] = items[idx2];
         items[idx2] = tmp;
      }

   private:

      /**
       * This method is to be called by {@link #qsort} only.
       *
       * Compare the two objects, respecting the ascending/descending flag
       * so that the calling sorting algorithm doesn't need to care
       * about that flag.
       */
      int qsort_compare ( const GComparator& comparator, 
                          const GObject& obj1, 
                          const GObject& obj2, 
                          bool ascending ) const
      {
         int res = comparator.compare2Objects(obj1, obj2);
         return ascending ? res : -res;
      }

      /**
       * This method is to be used by {@link #sortItems} only.
       *
       * If you think of a one dimensional array as going from
       * the lowest index on the left to the highest index on the right
       * then the parameters to this function are lowest index at
       * left and highest index at right. The first time you call
       * this method it will be with the parameters lo0=0, hi0=list.length - 1.
       *
       * @param list The array to sort.
       * @param lo0  Left boundary of array partition.
       * @param hi0  Right boundary of array partition.
       * @see   #sortItems
       */
      void qsort ( int lo0, int hi0, const GComparator& comparator, bool ascending )
      {
         int low = lo0;
         int high = hi0;
         if (hi0 > lo0)
         {
            // Arbitrarily establishing partition element as the midpoint of
            // the array.
            const T& mid = items[(lo0 + hi0) / 2];

            // loop through the array until indices cross
            while (low <= high)
            {
               // find the first element that is greater than or equal to
               // the partition element starting from the left Index.
               while ((low < hi0) && (qsort_compare(comparator, &items[low], &mid, ascending) < 0))
                  ++low;

               // find an element that is smaller than or equal to
               // the partition element starting from the right Index.
               while ((high > lo0) && (qsort_compare(comparator, &items[high], &mid, ascending) > 0))
                  --high;

               // if the indexes have not crossed, swap
               if (low <= high)
                  swap(low++, high--);
            }

            // If the right index has not reached the left side of array
            // must now sort the left partition.
            if (lo0 < high)
               qsort(lo0, high, comparator, ascending);

            // If the left index has not reached the right side of array
            // must now sort the right partition.
            if (low < hi0)
               qsort(low, hi0, comparator, ascending);
         }
      }

   public:

      /**
       * Sort the items in the array, using the specified comparator.
       */
      void sortItems ( const GComparator& comparator, bool ascending = true )
      {
         qsort(0, num - 1, comparator, ascending);
      }
};

#endif
