/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/primitives/GString.h"
#include "glib/primitives/GInteger.h"
#include "glib/primitives/GBoolean.h"
#include "glib/primitives/GLong.h"
#include "glib/util/GSectionBag.h"
#include "glib/vfs/GVfsLocal.h"
#include "glib/io/GFileInputStream.h"
#include "glib/io/GFileOutputStream.h"

GSectionBag::GSectionBag ( const GString& path, bool caseSen )
            :path(path),
             changed(false),
             sectionsBag(25, -3, caseSen),
             loadedOK(false),
             caseSen(caseSen)
{
   if (path == "")
      return;

   try {
      GVfsLocal vfs;
      GFileInputStream is(vfs, path, true);
      GString buff(8200);

      // ---
      // Load the whole file into buff.
      BYTE temp[4096+1];
      for (;;)
      {
         int read = is.read(temp, sizeof(temp));
         if (read <= 0)
            break;
         for (int i=0; i<read; i++)
            buff += char(temp[i]);
      }

      // ---
      // Parse the content of the loaded file.

      GString srcBuff(1024);
      GString sectionName(256); // Default section is empty (no section at all)
      GString entryName(256);
      bool isInsideSectionName = false;
      bool isInsideEntryName = false;
      bool isInsideEntryData = false;

      // The bag of entries of the default section, which has no section name.
      GKeyBag<GString>* entriesBag = sectionsBag.get("");
      if (entriesBag == null)
      {
         entriesBag = new GKeyBag<GString>(25);
         // Create the default bag, which is the bag without name.
         sectionsBag.put(sectionName, entriesBag);
      }

      const char* byteBuffPtr = buff.cstring();
      int byteBuffIdx = 0;
      int byteBuffCount = buff.length();

      for (;;)
      {
         // Get next byte, and put it in xbyte.
         int xbyte = EOF;
         if (++byteBuffIdx < byteBuffCount)
            xbyte = *byteBuffPtr++;

         // Parse the read byte.
         if (xbyte == EOF || // If we have reached EOF
             xbyte == '\n')  // Or if we have reached EOL
         {
            if (entryName.length() > 0 && !isInsideEntryName && isInsideEntryData)
            {
               entryName.trim(); // Strip white space from start and end.
               srcBuff.trim();
               GString* s = new GString(srcBuff);
               entriesBag->put(entryName, s);
            }
            srcBuff.clear();
            entryName.clear();
            isInsideSectionName = false;
            isInsideEntryName = false;
            isInsideEntryData = false;
            if (xbyte == EOF)
               break;
         }
         else
         if (xbyte == '[' && 
             !isInsideSectionName && 
             !isInsideEntryName && 
             !isInsideEntryData)
         {
            isInsideSectionName = true;
            entryName.clear();
            srcBuff.clear();
         }
         else
         if (isInsideSectionName)
         {
            if (xbyte == ']')
            {
               isInsideSectionName = false;
               sectionName = srcBuff;
               sectionName.trim(); // Strip white space from start and end.
               entriesBag = sectionsBag.get(sectionName);
               if (entriesBag == null)
               {
                  entriesBag = new GKeyBag<GString>(50, -2);
                  sectionsBag.put(sectionName, entriesBag);
               }
            }
            else
               srcBuff += char(xbyte);
         }
         else
         if (entryName.isEmpty() && !isInsideEntryName)
         {
            srcBuff.clear();
            if (xbyte != '=')
            {
               isInsideEntryName = true;
               srcBuff += char(xbyte);
            }
         }
         else
         if (isInsideEntryName)
         {
            if (xbyte == '=')
            {
               entryName = srcBuff;
               entryName.trim(); // Strip white space from start and end.
               isInsideEntryName = false;
               isInsideEntryData = true;
               srcBuff.clear();
            }
            else
               srcBuff += char(xbyte);
         }
         else
         if (isInsideEntryData)
            srcBuff += char(xbyte);
      }

      // The file was loaded and parsed OK.
      loadedOK = true;
   } catch (GOpenFileException& /*e*/) {
      // File does not exist, or cannot be opened for some other reason.
      loadedOK = false;
   }
}

bool GSectionBag::isDirty () const
{
   return changed;
}

void GSectionBag::write ()
{
   if (changed)
      return write(path);
}

void GSectionBag::write ( const GString& path )
{
   GVfsLocal vfs;
   GFileOutputStream os(vfs, path, true, true, true);
   write(os);
   changed = false;
}

void GSectionBag::write ( const GOutputStream& os )
{
   const int num1 = sectionsBag.getCount();
   for (int i1=0; i1<num1; i1++)
   {
      const GString& sectName = sectionsBag.getKey(i1);
      if (sectName != "")
         os.printf("\n[%s]\n", GVArgs(sectName));

      GKeyBag<GString>* sectBag = sectionsBag.get(sectName);
      const int num2 = sectBag->getCount();
      for (int i2=0; i2<num2; i2++)
      {
         const GString& itemName = sectBag->getKey(i2);
         const GString& itemValue = sectBag->getIndexedItem(i2);
         os.printf("%s=%s\n", GVArgs(itemName).add(itemValue));
      }
   }
}

GKeyBag<GString>* GSectionBag::getSectionBag ( const GString& section )
{
   GKeyBag<GString>* sect = sectionsBag.get(section);
   return sect;
}

GKeyBag<GString>& GSectionBag::getIndexedSectionBag ( int index ) 
{ 
   return sectionsBag.getIndexedItem(index); 
}

GString GSectionBag::getIndexedSectionName ( int index ) 
{ 
   return GString(sectionsBag.getKey(index)); 
}

int GSectionBag::getNumberOfSections () const 
{ 
   return sectionsBag.getCount(); 
}

bool GSectionBag::wasLoadedOK () const 
{ 
   return loadedOK; 
}

const GString& GSectionBag::getPath () const 
{ 
   return path; 
}

bool GSectionBag::deleteSection ( const GString& section )
{
   bool ret = sectionsBag.remove(section);
   if (ret)
      changed = true;
   return ret;
}

bool GSectionBag::deleteItem ( const GString& section, 
                               const GString& item )
{
   GKeyBag<GString>* sect = sectionsBag.get(section);
   if (sect == null)
      return false;

   bool ret = sect->remove(item);
   if (ret)
      changed = true;

   return ret;
}

void GSectionBag::putString ( const GString& section, 
                              const GString& item, 
                              const GString& value, 
                              bool onlyIf )
{
   if (!onlyIf)
      return;

   GKeyBag<GString>* sect = sectionsBag.get(section);
   if (sect == null)
   {
      // The specified section doesn't already exist, so create it now.
      sect = new GKeyBag<GString>(5, 10);
      sectionsBag.put(section, sect);
      changed = true;
   }

   GString* oldValue = sect->get(item);
   if (oldValue != null)
   {
      if (value != *oldValue)
      {
         GString* valCopy = new GString(value);
         sect->update(item, valCopy);
         changed = true;
      }
   }
   else
   {
      GString *valCopy = new GString(value);
      sect->put(item, valCopy);
      changed = true;
   }
}

void GSectionBag::putBool ( const GString& section, 
                            const GString& item, 
                            bool value, 
                            bool onlyIf )
{
   if (onlyIf)
      putInt(section, item, value ? 1 : 0, onlyIf);
}

void GSectionBag::putChar ( const GString& section, 
                            const GString& item, 
                            char value, 
                            bool onlyIf )
{
   if (onlyIf)
      putInt(section, item, int(value), onlyIf);
}

void GSectionBag::putInt ( const GString& section, 
                           const GString& item, 
                           int value, 
                           bool onlyIf )
{
   if (onlyIf)
      putString(section, item, GInteger::ToString(value), onlyIf);
}

void GSectionBag::putLong ( const GString& section, 
                            const GString& item, 
                            longlong value, 
                            bool onlyIf )
{
   if (onlyIf)
      putString(section, item, GLong::ToString(value), onlyIf);
}

void GSectionBag::putColor ( const GString& section, 
                             const GString& item, 
                             const GColor& color, 
                             bool onlyIf )
{
   if (onlyIf)
   {
      putString(section, item, color.toString(), onlyIf);
   }
}

GString GSectionBag::getString ( const GString& section, 
                                 const GString& item, 
                                 const GString& def )
{
   GKeyBag<GString>* sect = sectionsBag.get(section);
   if (sect == null)
      return GString(def);
   GString* str = sect->get(item);
   if (str == null)
      return GString(def);
   else
      return GString(*str);
}

int GSectionBag::getInt ( const GString& section, 
                          const GString& item, 
                          int def )
{
   GKeyBag<GString>* sect = sectionsBag.get(section);
   if (sect == null)
      return def;
   GString* str = sect->get(item);
   if (str == null)
      return def;
   try {
      return GInteger::ParseInt(*str);
   } catch (GNumberFormatException& /*e*/) {
      return def;
   }
}

int GSectionBag::getInt ( const GString& section, 
                          const GString& item, 
                          int min, 
                          int max, 
                          int def )
{
   int ret = getInt(section, item, def);
   if (ret < min)
      return min;
   else
   if (ret > max)
      return max;
   else
      return ret;
}

longlong GSectionBag::getLong ( const GString& section, 
                                const GString& item, 
                                longlong def )
{
   GKeyBag<GString>* sect = sectionsBag.get(section);
   if (sect == null)
      return def;
   GString* str = sect->get(item);
   if (str == null)
      return def;
   try {
      return GLong::ParseLong(*str);
   } catch (GNumberFormatException& /*e*/) {
      return def;
   }
}

bool GSectionBag::getBool ( const GString& section, 
                            const GString& item, 
                            bool def )
{
   GKeyBag<GString>* sect = sectionsBag.get(section);
   if (sect == null)
      return def;
   GString* str = sect->get(item);
   if (str == null)
      return def;
   return GBoolean::ParseBoolean(*str, def);
}

char GSectionBag::getChar ( const GString& section, 
                            const GString& item, 
                            char def )
{
   GKeyBag<GString>* sect = sectionsBag.get(section);
   if (sect == null)
      return def;
   GString* str = sect->get(item);
   if (str == null)
      return def;
   return char(GInteger::ParseInt(*str));
}

GColor GSectionBag::getColor ( const GString& section, 
                               const GString& item, 
                               const GColor& def )
{
   GString str = getString(section, item, "");
   if (str == "")
      return def;
   else
      return GColor::ParseColor(str, def);
}
