/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_MATH
#define __GLIB_MATH

#include "glib/primitives/GObject.h"

#undef max
#undef min
#undef clip

/**
 * This class is a collection of various static mathematical methods,
 * inspired by the Java class; <i>java.lang.Math</i>.
 *
 * @author  Leif Erik Larsen
 * @since   2000.04.26
 */
class GMath
{
   public:

      static double PI;

   private:

      /** Prevent this class from being instantiated. */
      GMath ();

   public:

#if __HAS_FUNCTION_TEMPLATES__
      template <class T> static T Max ( const T& a, const T& b ) { return a > b ? a : b; }
#else
      static int Max ( const int& a, const int& b ) { return a > b ? a : b; }
      static unsigned Max ( const unsigned& a, const unsigned& b ) { return a > b ? a : b; }
      static longlong Max ( const longlong& a, const longlong& b ) { return a > b ? a : b; }
      static ulonglong Max ( const ulonglong& a, const ulonglong& b ) { return a > b ? a : b; }
      static float Max ( const float& a, const float& b ) { return a > b ? a : b; }
      static double Max ( const double& a, const double& b ) { return a > b ? a : b; }
#endif

#if __HAS_FUNCTION_TEMPLATES__
      template<class T> static T Min ( const T& a, const T& b ) { return a < b ? a : b; }
#else
      static int Min ( const int& a, const int& b ) { return a < b ? a : b; }
      static unsigned Min ( const unsigned& a, const unsigned& b ) { return a < b ? a : b; }
      static longlong Min ( const longlong& a, const longlong& b ) { return a < b ? a : b; }
      static ulonglong Min ( const ulonglong& a, const ulonglong& b ) { return a < b ? a : b; }
      static float Min ( const float& a, const float& b ) { return a < b ? a : b; }
      static double Min ( const double& a, const double& b ) { return a < b ? a : b; }
#endif
      
#if __HAS_FUNCTION_TEMPLATES__
      template<class T> static T Abs ( const T& a ) { return a < 0 ? -a : a; }
#else
      static int Abs ( const int& a, const int& b ) { return a < 0 ? -a : a; }
      static unsigned Abs ( const unsigned& a, const unsigned& b ) { return a < 0 ? -a : a; }
      static longlong Abs ( const longlong& a, const longlong& b ) { return a < 0 ? -a : a; }
      static ulonglong Abs ( const ulonglong& a, const ulonglong& b ) { return a < 0 ? -a : a; }
      static float Abs ( const float& a, const float& b ) { return a < 0 ? -a : a; }
      static double Abs ( const double& a, const double& b ) { return a < 0 ? -a : a; }
#endif
      
#if __HAS_FUNCTION_TEMPLATES__
      template<class T> static T Clip ( const T& x, const T& min, const T& max ) { return (x > max) ? max : ((x < min) ? min : x); }
#else
      static int Clip ( const int& x, const int& min, const int& max ) { return (x > max) ? max : ((x < min) ? min : x); }
      static unsigned Clip ( const unsigned& x, const unsigned& min, const unsigned& max ) { return (x > max) ? max : ((x < min) ? min : x); }
      static longlong Clip ( const longlong& x, const longlong& min, const longlong& max ) { return (x > max) ? max : ((x < min) ? min : x); }
      static ulonglong Clip ( const ulonglong& x, const ulonglong& min, const ulonglong& max ) { return (x > max) ? max : ((x < min) ? min : x); }
      static float Clip ( const float& x, const float& min, const float& max ) { return (x > max) ? max : ((x < min) ? min : x); }
      static double Clip ( const double& x, const double& min, const double& max ) { return (x > max) ? max : ((x < min) ? min : x); }
#endif

      static double Acos ( double a );
      static double Asin ( double a );
      static double Atan ( double a );
      static double Ceil ( double a );
      static double Cos ( double a );
      static double Exp ( double a );
      static double Floor ( double a );
      static double Log ( double a );
      static int Rand ();
      static double Sin ( double a );
      static double Sqrt ( double a );
      static double Tan ( double a );
      static double Pow ( double a, double b );
};

#endif
