/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_LOG
#define __GLIB_LOG

#include <stdio.h>
#include "glib/primitives/GVArgs.h"

/**
 * Class with static context only that can be used for debug
 * logging to a text file.
 *
 * @author  Leif Erik Larsen
 * @since   1999.10.24
 * @see     GProgram#printF
 */
class GLog
{
   friend class GProgram;

   public:

      enum FilterId
      {
         DEBUG,
         TEST,
         PROD,
         LOGGING_IS_OFF
      };

   private:

      /** In order to prevent us from retrying to open the log file more than once, we use this variable. */
      static GString LogFilePath;

      /** Count the number of printed log statements. */
      static int Count;

      /** The current logging filter level. */
      static FilterId FilterLevel;

      /** The open text file of where to write to. */
      static FILE* Logfile;

      /** The lock used to ensure that only one thread is logging at a time. */
      static GObject Lock;

   private:

      /** Prevent this class from being instantiated. */
      GLog ();

   private:

      /** To be called by the constructor of GProgram only. */
      static void OpenLogFile ( const class GProgramParameter& logFileOpts );

      /** To be called when the log-file is reopened after being temporarily closed. */
      static const GString& ReOpenLogFile ( const GString& path, bool append );

      /** Get the human readable name of the specified log-filter level ID. */
      static GString GetFilterLevelName ( GLog::FilterId filterID );

   public:

      static bool Filter ( GLog::FilterId level );

      /**
       * Get the current log filter level.
       *
       * @author  Leif Erik Larsen
       * @since   2005.01.14
       * @see     #SetFilterLevel
       * @see     #GetFilterLevelName
       */
      static FilterId GetFilterLevel ();

      /**
       * Get the current log filter level as a human readable name.
       *
       * @author  Leif Erik Larsen
       * @since   2006.04.30
       * @see     #GetFilterLevel
       */
      static GString GetFilterLevelName ();

      /**
       * Get a copy of the path to the current log file.
       *
       * @author  Leif Erik Larsen
       * @since   2005.01.14
       */
      static GString GetLogFilePath ();

      /**
       * Print the specified message to the log file, but only if the
       * current log filter is the same as of weaker than the specified
       * filter level.
       *
       * @author  Leif Erik Larsen
       * @since   2001.03.15
       * @param   obj  The object context of the caller. This is typically 
       *               the "this" pointer of the object doing the call.
       *               Can be null, in which case we assume we are called
       *               by some static (non-object) context.
       * @param   msg  The message string of which to log.
       */
      static void Log ( const GObject* obj,
                        const char* msg,
                        const GVArgs& = GVArgs() );

      /**
       * Print the specified formatted string to the log-file "as-is", 
       * without any decorations other than a terminating linefeed (if not 
       * already contained in the specified string).
       *
       * @author  Leif Erik Larsen
       * @since   2006.04.29
       */
      static void PrintLn ( const char* formatStr, const GVArgs& args = GVArgs() );

      /**
       * Print the stack trace of the calling thread, with the specified 
       * formatted message as the message string at the top of the 
       * stack trace.
       *
       * @author  Leif Erik Larsen
       * @since   2006.05.02
       */
      static void PrintStackTrace ( const GString& formatStr, const GVArgs& args = GVArgs() );

      /**
       * Set the current log filter level.
       *
       * @author  Leif Erik Larsen
       * @since   2005.01.14
       * @see     #GetFilterLevel
       */
      static void SetFilterLevel ( FilterId fid );
};

#endif
