/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include <stdio.h>
#include <stdlib.h>
#include "glib/GProgram.h"
#include "glib/GProgramParameter.h"
#include "glib/util/GLog.h"
#include "glib/util/GTime.h"
#include "glib/exceptions/GIllegalStateException.h"

void GLog::OpenLogFile ( const GProgramParameter& logFileOpts )
{
   // Only the first thread should attempt to open the file.
   GObject::Synchronizer synch(Lock);
   if (Logfile != null)
      return; // Log-file has already been opened!
   if (LogFilePath != "")
      return; // Path of log-file already specified!

   static bool IsHere = false;
   if (IsHere) // Prevent recursive calls to ours self.
      return;

   IsHere = true;
   GProgram& prg = GProgram::GetProgram();
   GString path;
   if (logFileOpts.paramName != "")
   {
      const GCmdLineParameters& params = prg.getCmdLineParameters();
      path = params.getString(logFileOpts.paramName, "");
   }
   if (logFileOpts.envName != "" && path == "")
   {
      path = prg.getEnvironmentVar(logFileOpts.envName, "");
   }
   if (path == "")
   {
      GString root = prg.getRootDirectory();
      path = root + logFileOpts.defaultValue;
   }

   if (path != "")
   {
      GVfsLocal vfsLocal;
      GString fullPath = vfsLocal.getCurrentDirectory(false);
      if (vfsLocal.walkPath(fullPath, path))
         path = fullPath;
      LogFilePath = ReOpenLogFile(path, false);
   }

   IsHere = false;
}

const GString& GLog::ReOpenLogFile ( const GString& path, bool append )
{
   if (Logfile != null)
      return path; // Log-file has already been opened!
   Logfile = ::fopen(path, append ? "ab" : "wb");
   return path;
}

GString GLog::GetLogFilePath ()
{
   return LogFilePath;
}

GString GLog::GetFilterLevelName ( GLog::FilterId filterID )
{
   switch (filterID)
   {
      case DEBUG: return "Debug";
      case TEST: return "Test";
      case PROD: return "Prod";
      case LOGGING_IS_OFF: return "Off";
      default: gthrow_(GIllegalArgumentException("Unknown filter level: " + GInteger::ToString(filterID)));
   }
}

GString GLog::GetFilterLevelName ()
{
   return GLog::GetFilterLevelName(GLog::FilterLevel);
}

GLog::FilterId GLog::GetFilterLevel ()
{
   return FilterLevel;
}

void GLog::SetFilterLevel ( GLog::FilterId fid )
{
   if (fid == GLog::FilterLevel)
      return; // Nothing to do.
   if (Logfile == null && fid != LOGGING_IS_OFF)
   {
      // Re-open the log-file, in appending mode.
      GLog::ReOpenLogFile(LogFilePath, true);
   }
   GString filterName = GLog::GetFilterLevelName(fid);
   GString logStr("GLog Filter Level set to: %s", GVArgs(filterName));
   if (fid == LOGGING_IS_OFF)
   {
      // Log the new filter level BEFORE we activate it.
      GLog::Log(null, logStr);
      GLog::FilterLevel = fid;
      // Close the log-file.
      ::fclose(Logfile);
      Logfile = null;
   }
   else
   {
      // Log the new filter level AFTER we have activated it.
      GLog::FilterLevel = fid;
      GLog::Log(null, logStr);
   }
}

bool GLog::Filter ( GLog::FilterId level ) 
{ 
   return level >= FilterLevel && 
          FilterLevel < LOGGING_IS_OFF; 
}

void GLog::PrintLn ( const char* formatStr, const GVArgs& args )
{
   if (Logfile == null)
      return; // Something went wrong when attempting to open the log file.

   GString str(formatStr, args);

   // Make sure only one thread can get into this code at once.
   GObject::Synchronizer synch(Lock);

   // Write the string to the log-file.
   ::fwrite(str.cstring(), str.length(), 1, Logfile);
   if (!str.endsWithEol())
      ::fwrite(GString::EOL.cstring(), GString::EOL.length(), 1, Logfile);
   ::fflush(Logfile);
}

void GLog::PrintStackTrace ( const GString& formatStr, const GVArgs& args )
{
   GString msg(formatStr, args);
   GString stackTrace = GSymbolEngine::GetStackTrace(msg);
   GLog::PrintLn(stackTrace);
}

void GLog::Log ( const GObject* obj, 
                 const char* msg,
                 const GVArgs& args )
{
   if (Logfile == null)
      return; // Something went wrong when attempting to open the log file.

   if (FilterLevel == LOGGING_IS_OFF)
      return;

   GString className;
   const std::type_info* tinf = null;
   if (obj != null)
   {
      tinf = &typeid(*obj);
      className = tinf->name();
      for (;;)
      {
         int pos = className.indexOf("class ");
         if (pos < 0)
            break;
         className.remove(pos, 6);
      }
   }
   else
   {
      className = "{Static}";
   }

   GTime time;
   GString threadName;
   try {
      GThread& thread = GThread::GetCurrentThread();
      threadName = thread.getName();
   } catch (GException&) {
      threadName = GString("TID:0x%X", GVArgs(GThread::GetCurrentThreadID()));
   }

   // Format the message.
   //           Idx  Time                Th Cls Msg
   GString str("%07d %02d:%02d:%02d.%03d %s %s: %s\n",
               GVArgs(++Count).
               add(time.getHours()).add(time.getMinutes()).add(time.getSeconds()).add(time.getMillis()).
               add(threadName).
               add(className).
               add(args.num() == 0 ? GString(msg) : GString(msg, args)));

   // Make sure only one thread can get into this code at once.
   GObject::Synchronizer synch(Lock);

   // Print the message.
   static bool WasHere = false;
   if (!WasHere)
   {
      WasHere = true;
      GString head("Index   Time         Thread Class: Message\n");
      ::fwrite(head.cstring(), head.length(), 1, Logfile);
   }
   ::fwrite(str.cstring(), str.length(), 1, Logfile);
   ::fflush(Logfile);
}
