/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/util/GLocaleData.h"
#include "glib/util/GTime.h"
#include "glib/util/GLog.h"
#include "glib/primitives/GVArgs.h"

GLocaleData::GLocaleData ()
            :dateFormat(DATEFMT_MM_DD_YY),
             timeFormat(TIMEFMT_12HOUR),
             timeAMString("am"),
             timePMString("pm"),
             dateSeparator("/"),
             timeSeparator(":"),
             thousandsSeparator(","),
             decimalChar(".")
{
   dateFormat = DATEFMT(GLocaleData::GetProfInt("iDate", dateFormat));
   timeFormat = TIMEFMT(GLocaleData::GetProfInt("iTime", timeFormat));
   timeAMString = GLocaleData::GetProfStr("s1159", timeAMString);
   timePMString = GLocaleData::GetProfStr("s2359", timePMString);
   dateSeparator = GLocaleData::GetProfStr("sDate", dateSeparator);
   timeSeparator = GLocaleData::GetProfStr("sTime", timeSeparator);
   thousandsSeparator = GLocaleData::GetProfStr("sThousand", thousandsSeparator);
   decimalChar = GLocaleData::GetProfStr("sDecimal", decimalChar);
}

GString GLocaleData::GetProfStr ( const char* id, const GString& defaultVal )
{
   char buff[128];
   if (::PrfQueryProfileString(HINI_USER, "PM_National", id, defaultVal, buff, sizeof(buff)) > 0)
      return GString(buff);
   else
      return defaultVal;
}

int GLocaleData::GetProfInt ( const char* id, int def )
{
   GString temp = GLocaleData::GetProfStr(id, GInteger::ToString(def));
   try {
      return GInteger::ParseInt(temp);
   } catch (GNumberFormatException& e) {
      return def;
   }
}

GString GLocaleData::getTimeStringFromMillis ( ulonglong millis, bool inclMillis ) const
{
   int hour = int(millis / 3600000);
   millis -= hour * 3600000;
   int min = int(millis / 60000);
   millis -= min * 60000;
   int sec = int(millis / 1000);
   GString ret("%02d%s%02d%s%02d", GVArgs(hour).add(timeSeparator).add(min).add(timeSeparator).add(sec));
   if (inclMillis)
   {
      millis -= sec * 1000;
      ret += GString("%s%03d", GVArgs(decimalChar).add(millis));
   }
   return ret;
}

GString GLocaleData::getTimeString ( const GTime& time, bool inclSeconds ) const
{
   int hours = time.getHours();
   if (timeFormat == TIMEFMT_12HOUR)
      hours %= 12;
   GString ret("%02d%s%02d", GVArgs(hours).add(timeSeparator).add(time.getMinutes()));
   if (inclSeconds)
      ret += GString("%s%02d", GVArgs(timeSeparator).add(time.getSeconds()));
   if (timeFormat == TIMEFMT_12HOUR)
   {
      if (time.getHours() < 12)
         ret += timeAMString;
      else
         ret += timePMString;
   }
   return ret;
}

GString GLocaleData::getCurrentTimeString ( bool inclSeconds ) const
{
   GTime time;
   return getTimeString(time, inclSeconds);
}

GString GLocaleData::getDateString ( const GDate& date ) const
{
   if (date.getMonth() <= 0)
      return GString::Empty; // Not a valid date, so return blank string.

   switch (dateFormat)
   {
      case DATEFMT_DD_MM_YY:
           return GString("%02d%s%02d%s%04d", GVArgs(date.getDay()).add(dateSeparator).add(date.getMonth()).add(dateSeparator).add(date.getYear()));

      case DATEFMT_YY_MM_DD:
           return GString("%04d%s%02d%s%02d", GVArgs(date.getYear()).add(dateSeparator).add(date.getMonth()).add(dateSeparator).add(date.getDay()));

      case DATEFMT_YY_DD_MM:
           return GString("%04d%s%02d%s%02d", GVArgs(date.getYear()).add(dateSeparator).add(date.getDay()).add(dateSeparator).add(date.getMonth()));

      case DATEFMT_MM_DD_YY:
      default:
           return GString("%02d%s%02d%s%04d", GVArgs(date.getMonth()).add(dateSeparator).add(date.getDay()).add(dateSeparator).add(date.getYear()));
   }
}
