/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_DRIVEINFO
#define __GLIB_DRIVEINFO

#include "glib/primitives/GString.h"

/**
 * An instance of this class contains some detailed information
 * about one single logical drive.
 *
 * @author  Leif Erik Larsen
 * @since   2000.01.09
 */
class GDriveInfo : public GObject
{
   public:

      /** Maximum number of logical drives available to the system. */
      enum { MAX_DRIVES = 'Z' - 'A' + 1 };

      /**
       * Logical Drive Types.
       */
      enum DRIVE_TYPE
      {
         /** Unknown drive type (?). */
         DT_UNKNOWN,

         /** Standard floppy drive (typically A: or B:). */
         DT_FLOPPY,

         /** Standard CD-ROM. */
         DT_CDROM,

         /** Standard hard-disk. */
         DT_HARDDISK,

         /** Network (remote) drive. */
         DT_NETWORK,

         /** Some removable type (Zip-Drive, EZ-Drive, Jaz-Drive, etc.). */
         DT_REMOVABLE,

         /** RAM-Drive (virtual disk in memory). */
         DT_RAMDRIVE
      };

   public:

      /** 1=A:, 2=B:, 3=C:, etc. */
      int drive;

      /** Nonzero if this drive is avalible. */
      bool isAvailable;

      /** Type of the drive. */
      DRIVE_TYPE type;

      /** Volume name of the drive. */
      GString volumeName;

      /** Name of the File System on the drive. */
      GString fsName;

      /** Serial Number of the drive. */
      unsigned int serialNo;

      /** File Allocation Unit Size (512 for HPFS). */
      unsigned int bytesPerCluster;

      /** Number of bytes per sector. */
      unsigned int bytesPerSector;

      /** Number of sectors per allocation unit. */
      unsigned int numSectorsPerUnit;

      /** Number of allocation units. */
      unsigned int numUnits;

      /** Number of free allocation units. */
      unsigned int numAvailUnits;

      /** bytesPerSector * numSectorsPerUnit * numUnits. */
      longlong diskSize;

      /** bytesPerSector * numSectorsPerUnit * numAvailUnits. */
      longlong diskFree;

   public:

      /**
       * Objects created with this constructor will not contain any valid
       * information for any valid logical drive.
       *
       * All fields will be cleared. In order to get some drive information
       * into the object you must use the method <i>update()</i>.
       *
       * @see #update
       */
      GDriveInfo ();

   public:

      /**
       * Update the object with information about the specified
       * logical drive (0=Current, 1=A, 2=B, 3=C, etc.).
       *
       * @return NO_ERROR on success or else the Error Code of the error.
       *         In case of any error we will clear all items of the object
       *         automatically.
       */
      APIRET update ( int drive );

      /**
       * Clear all items of the object.
       */
      void clear ();

   public:

      /**
       * Get the number of bytes per sector for the specified logical
       * drive (0=Current, 1=A, 2=B, 3=C, etc.).
       *
       * @author  Leif Erik Larsen
       * @since   2004.08.12
       * @return  Number of bytes per sector, or 0 on any error.
       */
      static int GetBytesPerSector ( int drive );

      /**
       * Get the label of the specified logical
       * drive (0=Current, 1=A, 2=B, 3=C, etc.).
       *
       * @param  drive The drive of which to query (1=A, 2=B, 3=C, etc.).
       * @param  label Returns the label for the drive, or an empty string
       *               in case of any error (return value != NO_ERROR).
       * @return NO_ERROR on success or else the Error Code of the error.
       */
      static APIRET GetDriveLabel ( int drive, GString& label );

      /**
       * Get the serial number of the specified logical
       * drive (0=Current, 1=A, 2=B, 3=C, etc.).
       *
       * @return The serial number of the drive, or 0 on any error.
       */
      static unsigned int GetDriveSerialNo ( int drive );

      /**
       * Get a bitmap that represent which drives are currently available 
       * on the system. Bit 0 represents drive A:, but 1 drive B:, 
       * bit 2 drive C:, etc.
       */
      static int GetMapOfAvailableDrives ();

      /**
       * Test if the specified drive letter represent a logical drive
       * that is currently available to the system.
       */
      static bool IsAValidDriveLetter ( char cDrive );

      /**
       * Test if the specified logical drive
       * (0=Current, 1=A, 2=B, 3=C, etc.) contains
       * a removable media (floppy, zip, jaz, etc.).
       */
      static bool IsMediaRemoveable ( int drive );

      /**
       * Test if the specified logical drive
       * (0=Current, 1=A, 2=B, 3=C, etc.) currently
       * contains a media.
       */
      static bool IsMediaInDrive ( int drive );

      /**
       * Determining whether a the specified logical
       * drive (0=Current, 1=A, 2=B, 3=C, etc.) is a network drive.
       */
      static bool IsANetworkDrive ( int drive );

      /**
       * Determining whether a the specified logical
       * drive (0=Current, 1=A, 2=B, 3=C, etc.) is a CD-ROM drive.
       *
       * This method works even if there is no CD currently inserted
       * in the drive as well as if the current CD is an audio disk.
       *
       * @author  Leif Erik Larsen
       * @since   2001.07.13
       */
      static bool IsACdRomDrive ( int drive );
};

#endif


