/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_ABSTRACTTOKEN
#define __GLIB_ABSTRACTTOKEN

#include "glib/primitives/GString.h"


/**
 * This interface forms the abstract basis declarations for any
 * token in general.
 *
 * @author  Leif Erik Larsen
 * @since   2000.05.23
 * @see     GAbstractTokenizer
 * @see     GToken
 * @see     GRcToken
 * @see     GExpressionParser
 */
class GAbstractToken : public GObject
{
   public:

      /**
       * Test if the current token was enclosed in a pair of quotes
       * in the tokenizer stream.
       *
       * @author  Leif Erik Larsen
       * @since   2000.07.29
       */
      virtual bool isQuoted () const = 0;

      /**
       * Get a copy of the character that caused {@link #isQuote} to
       * return true.
       *
       * If {@link #isQuote} does not return true then a default quote
       * character is returned. This is typically the character; '"'.
       *
       * @author  Leif Erik Larsen
       * @since   2000.07.29
       */
      virtual char getQuoteCharacter () const = 0;

      /**
       * Return true if and only if the token is 100% empty.
       *
       * A quoted token is never to be considered empty by this method,
       * even if the quoted string is empty in it self.
       *
       * @author  Leif Erik Larsen
       * @since   2000.08.01
       */
      virtual bool isEmpty () const = 0;

      /**
       * Get a copy of the indexed character within the token.
       *
       * @author  Leif Erik Larsen
       * @since   2000.07.29
       */
      virtual char getCharacter ( int idx ) const = 0;

      /**
       * Get the number of characters in the token.
       *
       * @author  Leif Erik Larsen
       * @since   2000.07.29
       */
      virtual int getLength () const = 0;

      /**
       * Test if the token is 100% equal to the specified string token.
       *
       * @author  Leif Erik Larsen
       * @since   2000.07.29
       */
      virtual bool operator== ( const GString& token ) const = 0;
      virtual bool operator!= ( const GString& token ) const = 0;

      /**
       * Get the current token as a string.
       *
       * The returned string will not be enclosed in a pair of quotes,
       * even if the current token was originally enclosed in a pair of quotes.
       * The calling code can use {@link #isQuoted} in order to determine
       * whether or not to add the quote characters to the string returned
       * from this method, if needed.
       *
       * @author  Leif Erik Larsen
       * @since   2000.07.29
       */
      virtual GString toString () const = 0;
};

#endif


