/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_ERROR
#define __GLIB_ERROR

#include "glib/primitives/GString.h"

/**
 * Class used to represent system dependent error code in a system 
 * independent (portable) way.
 *
 * @author  Leif Erik Larsen
 * @since   2005.03.30
 */
class GError
{
   public:

      static const GError Ok; // No error.

      static const GError AccessDenied;
      static const GError BadCommand;
      static const GError BrokenPipe;
      static const GError CircularityRequested;
      static const GError CorruptDataFormat;
      static const GError ExtendedAttributesNotSupported;
      static const GError FileExists;
      static const GError FilenameExceedRange;
      static const GError FileNotFound;
      static const GError FunctionFailed;
      static const GError InvalidArgument;
      static const GError InvalidHandle;
      static const GError InvalidName;
      static const GError InvalidParameter;
      static const GError OutOfMemory;
      static const GError NotSupported;
      static const GError PathNotFound;

   public:

      /** The system dependent error code for this error. */
      int sysErrorCode;

      /** Optional additional details about the error. */
      GString errorDetails;

   public:

      GError ();
      GError ( int sysErrorCode );
      GError ( const GError& src );
      ~GError ();

   public:

      /**
       * Returns true if and only if the error code is the same as of the 
       * specified object.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.30
       */
      bool operator== ( const GError& o ) const;

      /**
       * Returns true if and only if the error code is not the same as of the 
       * specified object.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.30
       */
      bool operator!= ( const GError& o ) const;

      /**
       * @author  Leif Erik Larsen
       * @since   2005.03.30
       */
      void operator= ( int sysErrorCode_ );

      /**
       * Get the optional additional user code defined error details string.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.30
       */
      GString getErrorDetails () const;

      /**
       * Get the system dependent default translated error message text 
       * for this error.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.30
       * @see     GSystem#GetApiRetString
       */
      GString getErrorMessage () const;
};

#endif
