/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/resource/GToolbarButtonParams.h"
#include "glib/resource/GResourceTable.h"
#include "glib/resource/GRcCompiler.h"
#include "glib/resource/GRcException.h"

GToolbarButtonParams::GToolbarButtonParams ( const GString& id )
                     :GAbstractCommand(id),
                      itemType(Separator),
                      isIconDef(false),
                      isTextDef(false),
                      isHintDef(false),
                      isUserDataDef(false)
{
}

GToolbarButtonParams::GToolbarButtonParams ()
                     :GAbstractCommand(),
                      itemType(Separator),
                      isIconDef(false),
                      isTextDef(false),
                      isHintDef(false),
                      isUserDataDef(false)
{
}

void GToolbarButtonParams::loadFromScript ( GRcTokenizer& tokenizer, 
                                            GResourceTable* table )
{
   const GRcToken* token = tokenizer.nextPreCompiledToken(false);
   if (*token != GRcTokenizer::Token_lpar)
      gthrow_(GRcException(tokenizer, GRcException::ERR_EXPECTED_X_FOUND_Y, GVArgs("(").add(token->getString()))); // Expected '(' but found '%2' in statement!

   for (;;)
   {
      token = tokenizer.nextPreCompiledToken(false);
      if (*token == GRcTokenizer::Token_rpar)
         break;
      else
      if (*token == GRcTokenizer::Token_comma)
         continue;
      else
      if (*token == GRcTokenizer::Token_icon)
      {
         if (isIconDef)
            gthrow_(GRcException(tokenizer, GRcException::ERRICONARDEF)); // ICON already defined!

         token = tokenizer.queryArgValue();

         // Make sure the specified ICON actually exists.
         if (table != null)
            if (!table->getIconBag().containsKey(token->getString()))
               gthrow_(GRcException(tokenizer, GRcException::ERRUNDEFICON, GVArgs(token->getString()))); // Undefined ICON: '%s'

         iconID = token->getString();
         isIconDef = true;
      }
      else
      if (*token == GRcTokenizer::Token_id)
      {
         if (getIDString() != "")
            gthrow_(GRcException(tokenizer, GRcException::ERRIDARDEF)); // ID already defined!

         token = tokenizer.queryArgValue();
         setIDString(token->getString());
      }
      else
      if (*token == GRcTokenizer::Token_text)
      {
         if (isTextDef)
            gthrow_(GRcException(tokenizer, GRcException::ERRTEXTARDEF)); // TEXT already defined!

         token = tokenizer.queryArgValue();
         textID = token->getString();
         isTextDef = true;
      }
      else
      if (*token == GRcTokenizer::Token_hint)
      {
         if (isHintDef)
            gthrow_(GRcException(tokenizer, GRcException::ERRHINTARDEF)); // HINT already defined!

         token = tokenizer.queryArgValue();
         hintID = token->getString();
         isHintDef = true;
      }
      else
      if (*token == GRcTokenizer::Token_userdata)
      {
         if (isUserDataDef)
            gthrow_(GRcException(tokenizer, GRcException::ERRUSERDATA1ARDEF)); // Userdata already defined!

         token = tokenizer.queryArgValue();
         setUserData1(token->getString());
         isUserDataDef = true;
      }
      else
         gthrow_(GRcException(tokenizer, GRcException::ERRWUNKNOWN, GVArgs(token->getString()))); // Unknown token: %s
   }

   if (itemType != Separator)
   {
      if (getIDString() == "")
         gthrow_(GRcException(tokenizer, GRcException::ERRITEMMISSIDTB)); // No ID specified for this toolbar button!

      if (!isTextDef)
      {
         if (isHintDef)
            gthrow_(GRcException(tokenizer, GRcException::ERRITEMMISSTEXTTB)); // No TEXT specified for this toolbar button!

         // This toolbar button is defined without TEXT and without HINT.
         // Such items should try to fetch TEXT from the texts
         // resource table by default, just as for menu items.

         textID = '%'; // Make this a reference to the other TEXT
         textID += getIDString();
      }
   }
}

