/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/resource/GTextResource.h"
#include "glib/resource/GRcCompiler.h"
#include "glib/GProgram.h"
#include "glib/primitives/GCharacter.h"
#include "glib/primitives/GUInteger.h"
#include "glib/util/GTokenizer.h"

GTextResource::GTextResource ( const GString& id )
              :GAbstractResource(id)
{
}

GString GTextResource::MakeTextLiteral ( const GString& str, char quote )
{
   int strLen = str.length();
   GString buff(strLen + 10);

   if (quote != '\0')
      buff += quote;

   for (int i=0; i<strLen; i++)
   {
      unsigned char chr = str[i];
      switch (chr)
      {
         case '\0': // Null-character
              buff += '\\';
              buff += '0';
              break;

         case '\b': // Back-space
              buff += '\\';
              buff += 'b';
              break;

         case '\n': // New-line
              buff += '\\';
              buff += 'n';
              break;

         case '\r': // Carriage-return
              buff += '\\';
              buff += 'r';
              break;

         case '\t': // Tab
              buff += '\\';
              buff += 't';
              break;

         case '\\': // Back-slash
              buff += '\\';
              buff += '\\';
              break;

         case '"': // Double-quote
              buff += '\\';
              buff += '"';
              break;

         default:
              if (chr < 32 || chr > 127)
              {
                 buff += '\\';
                 buff += GUInteger::ToString(chr, 8, 3, '0');
              }
              else
              {
                 buff += chr;
              }
              break;
      }
   }

   if (quote != '\0')
      buff += quote;

   return buff;
}

GString GTextResource::ParseTextLiteral ( const GString& str, bool isQuoted )
{
   GString tmp;
   const GString* quotedStr;
   if (isQuoted)
   {
      quotedStr = &str;
   }
   else
   {
      // Since the string is not already quoted, we must quote it now.
      tmp.ensureAllocatedSize(str.length() + 2);
      tmp += '"';
      tmp += str;
      tmp += '"';
      quotedStr = &tmp;
   }

   // Do the parsing and character translation.
   GTokenizer tknzr(*quotedStr, false);
   const GToken* token = tknzr.getNextToken();
   return token->toString(false);
}

void GTextResource::loadFromScript ( GRcTokenizer& tokenizer, GResourceTable* table )
{
   const GRcToken* token = tokenizer.nextPreCompiledToken(false);
   if (*token != GRcTokenizer::Token_lpar)
      gthrow_(GRcException(tokenizer, GRcException::ERR_EXPECTED_X_FOUND_Y, GVArgs("(").add(token->toString()))); // Expected '(' but found '%2' in statement!

   bool textIsDef = false; // True if TEXT parameter was defined
   bool hintIsDef = false; // True if HINT parameter was defined
   bool alt1IsDef = false; // True if ALT1 parameter was defined
   bool alt2IsDef = false; // True if ALT2 parameter was defined
   bool alt3IsDef = false; // True if ALT3 parameter was defined
   bool alt4IsDef = false; // True if ALT4 parameter was defined
   bool alt5IsDef = false; // True if ALT5 parameter was defined
   bool alt6IsDef = false; // True if ALT6 parameter was defined
   bool alt7IsDef = false; // True if ALT7 parameter was defined
   bool alt8IsDef = false; // True if ALT8 parameter was defined
   bool alt9IsDef = false; // True if ALT9 parameter was defined
   GString* lastTextRef = &textID;
   for (;;)
   {
      token = tokenizer.nextPreCompiledToken(false);
      if (token->isQuoted())
      {
         lastTextRef->operator+=(token->getString());
         if (lastTextRef == &textID)
            textIsDef = true; // In case text was given withouth "TEXT=".
      }
      else
      if (*token == GRcTokenizer::Token_text)
      {
         if (textIsDef)
            gthrow_(GRcException(tokenizer, GRcException::ERRTEXTARDEF)); // TEXT already defined!
         token = tokenizer.queryArgValue();
         textID = token->getString();
         textIsDef = true;
         lastTextRef = &textID;
      }

      else
      if (*token == GRcTokenizer::Token_hint)
      {
         if (hintIsDef)
            gthrow_(GRcException(tokenizer, GRcException::ERRHINTARDEF)); // HINT already defined!
         token = tokenizer.queryArgValue();
         hintID = token->getString();
         hintIsDef = true;
         lastTextRef = &hintID;
      }

      else
      if (*token == GRcTokenizer::Token_rpar)
      {
         if (!textIsDef)
            gthrow_(GRcException(tokenizer, GRcException::ERRMENTXTNOTXT)); // No TEXT-parameter specified for this TEXT!
         break;
      }

      else
      if (GResourceTable::IsLoadableText(token->getString()))
      {
         GTextResource *text = null;
         if (table != null)
            text = table->getTextResource(token->getString());
         if (text == null)
         {
            textID = token->getString();
         }
         else
         {
            textID = text->textID;
            hintID = text->hintID;
            alt1 = text->alt1;
            alt2 = text->alt2;
            alt3 = text->alt3;
         }
      }

      else
      if (*token == GRcTokenizer::Token_comma)
      {
         continue;
      }

      else
      if (*token == GRcTokenizer::Token_alt1)
      {
         if (alt1IsDef)
            gthrow_(GRcException(tokenizer, GRcException::ERRTXTARDEFALT1)); // ALT1 Already defined!
         token = tokenizer.queryArgValue();
         alt1 = token->getString();
         alt1IsDef = true;
         lastTextRef = &alt1;
      }

      else
      if (*token == GRcTokenizer::Token_alt2)
      {
         if (alt2IsDef)
            gthrow_(GRcException(tokenizer, GRcException::ERRTXTARDEFALT2)); // ALT2 Already defined!
         token = tokenizer.queryArgValue();
         alt2 = token->getString();
         alt2IsDef = true;
         lastTextRef = &alt2;
      }

      else
      if (*token == GRcTokenizer::Token_alt3)
      {
         if (alt3IsDef)
            gthrow_(GRcException(tokenizer, GRcException::ERRTXTARDEFALT3)); // ALT3 Already defined!
         token = tokenizer.queryArgValue();
         alt3 = token->getString();
         alt3IsDef = true;
         lastTextRef = &alt3;
      }

      else
      if (*token == GRcTokenizer::Token_alt4)
      {
         if (alt4IsDef)
            gthrow_(GRcException(tokenizer, GRcException::ERRTXTARDEFALT4)); // ALT4 Already defined!
         token = tokenizer.queryArgValue();
         alt4 = token->getString();
         alt4IsDef = true;
         lastTextRef = &alt4;
      }

      else
      if (*token == GRcTokenizer::Token_alt5)
      {
         if (alt5IsDef)
            gthrow_(GRcException(tokenizer, GRcException::ERRTXTARDEFALT5)); // ALT5 Already defined!
         token = tokenizer.queryArgValue();
         alt5 = token->getString();
         alt5IsDef = true;
         lastTextRef = &alt5;
      }

      else
      if (*token == GRcTokenizer::Token_alt6)
      {
         if (alt6IsDef)
            gthrow_(GRcException(tokenizer, GRcException::ERRTXTARDEFALT6)); // ALT6 Already defined!
         token = tokenizer.queryArgValue();
         alt6 = token->getString();
         alt6IsDef = true;
         lastTextRef = &alt6;
      }

      else
      if (*token == GRcTokenizer::Token_alt7)
      {
         if (alt7IsDef)
            gthrow_(GRcException(tokenizer, GRcException::ERRTXTARDEFALT7)); // ALT7 Already defined!
         token = tokenizer.queryArgValue();
         alt7 = token->getString();
         alt7IsDef = true;
         lastTextRef = &alt7;
      }

      else
      if (*token == GRcTokenizer::Token_alt8)
      {
         if (alt8IsDef)
            gthrow_(GRcException(tokenizer, GRcException::ERRTXTARDEFALT8)); // ALT8 Already defined!
         token = tokenizer.queryArgValue();
         alt8 = token->getString();
         alt8IsDef = true;
         lastTextRef = &alt8;
      }

      else
      if (*token == GRcTokenizer::Token_alt9)
      {
         if (alt9IsDef)
            gthrow_(GRcException(tokenizer, GRcException::ERRTXTARDEFALT9)); // ALT9 Already defined!
         token = tokenizer.queryArgValue();
         alt9 = token->getString();
         alt9IsDef = true;
         lastTextRef = &alt9;
      }

      else
      {
         gthrow_(GRcException(tokenizer, GRcException::ERRWUNKNOWN, GVArgs(token->getString()))); // Unknown token: %s
      }
   }
}

