/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_RESOURCETABLE
#define __GLIB_RESOURCETABLE

#include "glib/resource/GTextResource.h"
#include "glib/resource/GIconResource.h"
#include "glib/resource/GMenuResource.h"
#include "glib/resource/GToolbarResource.h"
#include "glib/resource/GDialogResource.h"
#include "glib/resource/GAccelResource.h"
#include "glib/util/GKeyBag.h"

/**
 * The resource table class.
 */
class GResourceTable : public GObject
{
   public:

      /** Flags that can be used with {@link #loadText}. */
      enum LoadTextFlags
      {
         /** This is the default. */
         LT_PREFER_NONE = 0x0000,

         /** Prefer the TEXT text parameter. */
         LT_PREFER_TEXT = 0x0001,

         /** Prefer the HINT text parameter. */
         LT_PREFER_HINT = 0x0002,

         /** Prefer the ALT1 text parameter. */
         LT_PREFER_ALT1 = 0x0004,

         /** Prefer the ALT2 text parameter. */
         LT_PREFER_ALT2 = 0x0008,

         /** Prefer the ALT3 text parameter. */
         LT_PREFER_ALT3 = 0x0010,

         /** Prefer ALT1, ALT2 or ALT3. Which of them doesn't really care. */
         LT_PREFER_ALTERNATIVE = 0x0020,

         /** Clip if text is longer than the specified max length. */
         LT_CLIP_IF_NEEDED = 0x0040
      };

   private:

      /** Bag of all allocated text's, indexed by ID. */
      GKeyBag<GTextResource> texts;

      /** Bag of all allocated icon's indexed by ID. */
      GKeyBag<GIconResource> icons;

      /** Bag of all allocated menu's indexed by ID. */
      GKeyBag<GMenuResource> menus;

      /** Bag of all allocated toolbar's indexed by ID. */
      GKeyBag<GToolbarResource> tbars;

      /** Bag of all allocated dialog's indexed by ID. */
      GKeyBag<GDialogResource> dialogs;

      /** Bag of all allocated acellerator tables indexed by ID. */
      GKeyBag<GAccelResource> accels;

      /** Used to prevent endless recursive calls to {@link #loadText}. */
      mutable int loadText_Counter;

   public:

      GResourceTable ();
      virtual ~GResourceTable ();

   private:

      /**
       * Disable the assignment operator.
       */
      const GResourceTable& operator= ( const GResourceTable& src ) const { return *this; }

      /**
       * Used internally by {@link #loadText} only.
       */
      static const GString& GetBestMatch ( int maxLength, 
                                           const GString& s1, 
                                           const GString& s2, 
                                           const GString& s3 = GString::Empty, 
                                           const GString& s4 = GString::Empty, 
                                           const GString& s5 = GString::Empty,
                                           const GString& s6 = GString::Empty,
                                           const GString& s7 = GString::Empty,
                                           const GString& s8 = GString::Empty,
                                           const GString& s9 = GString::Empty,
                                           const GString& s10 = GString::Empty,
                                           const GString& s11 = GString::Empty );

   public:

      GKeyBag<GTextResource>& getTextBag ();
      GKeyBag<GIconResource>& getIconBag ();
      GKeyBag<GMenuResource>& getMenuBag ();
      GKeyBag<GToolbarResource>& getToolbarBag ();
      GKeyBag<GDialogResource>& getDialogBag ();
      GKeyBag<GAccelResource>& getAcceleratorBag ();

      GTextResource* getTextResource ( const GString& name ) const;
      GIconResource* getIconResource ( const GString& name ) const;
      GMenuResource* getMenuResource ( const GString& name ) const;
      GToolbarResource* getToolbarResource ( const GString& name ) const;
      GDialogResource* getDialogResource ( const GString& name ) const;
      GAccelResource* getAcceleratorResource ( const GString& name ) const;

      static bool IsLoadableText ( const GString& textID );

      /**
       * Allocate a string buffer and copy the specified source string
       * into the allocated buffer. If the source string is marked as a
       * reference to another string (that is if it contains a
       * percentage prefix character) then we will rather copy the
       * referenced string.
       * <p>
       * <b>Mark</b> that this method will search for the refernced text
       * using a recursive algorithm, meaning that if the text resource
       * it self is a reference to another text resource then we will
       * return the referenced text at the deepest end.
       *
       * @since   1999.11.20
       * @param   sourceStr  Source string to copy from.
       * @param   flags      Or'ed combinations of
       *                     {@link #LT_PREFER_HINT},
       *                     {@link #LT_PREFER_ALTERNATIVE},
       *                     {@link #LT_CLIP_IF_NEEDED}, etc.
       * @param   maxLength  Max length of the returned string, or 0 if
       *                     there are no limit in the length of the
       *                     returned string.
       * @see     GProgram#LoadText
       */
      GString loadText ( const char* sourceStr,
                         LoadTextFlags flags = LT_PREFER_TEXT,
                         int maxLength = 0 ) const;
};

#endif


