/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include <stdio.h>
#include "glib/primitives/GObject.h"
#include "glib/primitives/GString.h"
#include "glib/util/GCriticalSection.h"
#include "glib/util/GEventSemaphore.h"
#include "glib/exceptions/GIllegalStateException.h"

GObject::GObject ()
        :lock(null),
         monitor(null)
{
}

GObject::GObject ( const GObject& src ) 
        :lock(null),
         monitor(null)
{ 
}

GObject::~GObject () 
{
   delete lock;
   delete monitor;
}

GObject& GObject::operator= ( const GObject& src ) 
{ 
   return *this; 
}

GCriticalSection& GObject::GetSynchLock ()
{
   static GCriticalSection* LockSynch = null;
   if (LockSynch == null)
      LockSynch = new GCriticalSection();
   return *LockSynch;
}

int GObject::compareObj ( const GObject& obj ) const
{
   GString str1 = toString();
   GString str2 = obj.toString();
   return str1.compare(str2);
}

bool GObject::equals ( const GObject& obj ) const
{
   return this == &obj;
}

void GObject::finalize ()
{
   GObject* self = this;
   self->~GObject(); // Virtually call destructor of subclass (if any).
}

int GObject::hashCode () const
{
   return int(this);
}

void GObject::ensureLockObjectIsCreated () const
{
   GCriticalSection& lockSynch = GetSynchLock();
   GCriticalSection::LocalLock autoLock(lockSynch);
   if (lock == null)
      lock = new GCriticalSection();
}

void GObject::ensureMonitorObjectIsCreated () const
{
   GCriticalSection& lockSynch = GetSynchLock();
   GCriticalSection::LocalLock autoLock(lockSynch);
   if (monitor == null)
      monitor = new GEventSemaphore();
}

void GObject::wait () const
{
   ensureMonitorObjectIsCreated();
   monitor->wait();
}

void GObject::wait ( int timeout ) const
{
   ensureMonitorObjectIsCreated();
   monitor->wait(timeout);
}

void GObject::notifyAll () const
{
   ensureMonitorObjectIsCreated();
   monitor->notifyAll();
}

void GObject::enterSynchronizationLock () const
{
   ensureLockObjectIsCreated();
   lock->enter();
}

void GObject::exitSynchronizationLock () const
{
   ensureLockObjectIsCreated();
   lock->exit();
}

GString GObject::toString () const
{
   GString str(128);
   str += typeid(*this).name();
   char buff[32];
   sprintf(buff, "#0x%X", int(this));
   str += buff;
   return str;
}
