/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_BOOLEAN
#define __GLIB_BOOLEAN

#include "glib/primitives/GNumber.h"

/**
 * A general boolean representation class.
 *
 * @author  Leif Erik Larsen
 * @since   1999.09.18
 */
class GBoolean : public GNumber
{
   private:

      bool value;

   public:

      /** A static instance that represents a boolean value that is true. */
      static const GBoolean True;

      /** A static instance that represents a boolean value that is false. */
      static const GBoolean False;

      /** A static string that represents a boolean value that is "true". */
      static const GString TrueStr;

      /** A static string that represents a boolean value that is "false". */
      static const GString FalseStr;

   public:

      GBoolean ( bool val ) : value(val) {}
      GBoolean ( int val ) : value(val ? true : false) {}
      GBoolean ( const GBoolean& val ) : value(val.value) {}
      GBoolean () : value(false) {}

      virtual ~GBoolean ();

   public:

      operator bool () const { return value; }

   public:

      virtual bool boolValue () const { return value ? true : false; }
      virtual char charValue () const { return value ? '1' : '0'; }
      virtual short shortValue () const { return value ? short(1) : short(0); }
      virtual int intValue () const { return value ? 1 : 0; }
      virtual longlong longValue () const { return value ? 1L : 0L; }
      virtual unsigned short ushortValue () const { return (unsigned short) (value ? 1 : 0); }
      virtual unsigned int uintValue () const { return (unsigned int) (value ? 1 : 0); }
      virtual ulonglong ulongValue () const { return (ulonglong) (value ? 1 : 0); }
      virtual float floatValue () const { return value ? 1.0f : 0.0f; }
      virtual double doubleValue () const { return value ? 1.0 : 0.0; }

      virtual int hashCode () const;
      virtual bool equals ( const GObject& obj ) const;

      /**
       * Compares two GBoolean's for sorting algorithms, etc., using 
       * the method {@link #boolValue}.
       *
       * @author  Leif Erik Larsen
       * @since   2004.03.14
       * @param   obj  The object of which to compare to.
       *               Must be an instance of {@link GNumber}.
       * @see     GArrayImpl#compare2Objects
       * @see     GComparator
       */
      virtual int compareObj ( const GObject& obj ) const;

   public:

      static const GBoolean& GetBoolean ( bool b );
      static const GString& GetBooleanStr ( bool b );

   public:

      /**
       * Return a <i>GString</i> representation of the current boolean value.
       * We will return "true" if the value is true, or "false" if the value
       * is false.
       */
      virtual GString toString () const;

      /**
       * Return a <i>GString</i> representation of the specified boolean value.
       * We will return "true" if the value is true, or "false" if the value
       * is false.
       */
      static GString ToString ( bool val );

      /**
       * @author  Leif Erik Larsen
       * @since   2000.12.03
       */
      static bool ParseBoolean ( const GString& str, bool def = false );
};

#endif // #ifndef __GLIB_BOOLEAN


