/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_INPUTSTREAM
#define __GLIB_INPUTSTREAM

#include "glib/io/GIOException.h"

/**
 * This is an abstract base class for all classes that provides
 * reading from an input stream.
 *
 * Only one method is pure virtual in this class, and that is the method
 * for reading one single byte from the stream ({@link #readByte}).
 * This method is the only one that is absolutely required to be
 * implemented by the subclass.
 *
 * We provide a default implementation for all the other virtual methods
 * of this class. These default implementations dependends on the
 * {@link #readByte} method only. Because of this, be aware that the
 * default implementations probably have relatively poor speed performance.
 * Thus, it makes sense for most subclasses to override every method from
 * this class, so that the concrete implementation can run at a maximized
 * speed.
 *
 * @author  Leif Erik Larsen
 * @since   2000.07.29
 * @see     GOutputStream
 * @see     GRandomAccessFile
 * @see     GStringStream
 */
class GInputStream : public GObject
{
   public:

      /**
       * Reads a byte of data from this file.
       *
       * The byte is returned as an integer in the range 0 to
       * 255 (<code>0x00-0x0FF</code>). This method blocks if no input
       * is yet available.
       *
       * @author  Leif Erik Larsen
       * @since   2000.07.29
       * @return  The next byte of data, or less than zero if the end
       *          of the stream has been reached.
       * @throws  GIOException  If an I/O error occurs. Not thrown if
       *                        end-of-stream has been reached.
       */
      virtual int readByte () const = 0;

      /**
       * Read from the stream until, and including, the next
       * linefeed character.
       *
       * You can use the method {@link GString#stripTrailingEol} as a
       * convenient way to strip out the trailing linefeed (if any) from
       * the returned string object.
       *
       * Return a reference to the specified string, which will contain the
       * read string upon return. The buffer is cleared before starting to
       * read.
       *
       * @author  Leif Erik Larsen
       * @since   2000.07.29
       * @param   str       The string of where to put the read characters.
       * @return  A reference to the same string as is specified in
       *          the parameter str. In case of end-of-stream or an
       *          error the returned string will be empty.
       */
      virtual GString& readString ( GString& str ) const;

      /**
       * Read maximum the specified number of bytes from the file stream.
       *
       * @author  Leif Erik Larsen
       * @since   2000.07.29
       * @param   buff    Buffer of where to store the read data.
       * @param   count   The number of bytes to read, maximum.
       * @return  The number of bytes actually read, or < 0 on EOF.
       * @throws  GIOException  In case of any error, except EOF.
       *                        If EOF is reached then we will return a
       *                        value that is less than <i>count</i>,
       *                        instead of throwing a GIOException.
       */
      virtual int read ( void* buff, int count ) const;

      /**
       * Read exactly the specified number of bytes from the file stream,
       * blocking if needed.
       *
       * @author  Leif Erik Larsen
       * @since   2004.04.15
       * @param   buff    Buffer of where to store the read data.
       * @param   count   The number of bytes to read, exactly.
       * @throws  GIOException  In case of any error, including EOF.
       */
      void readExact ( void* buff, int count ) const;
};

#endif

