/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/io/GFileInputStream.h"
#include "glib/sys/GSystem.h"
#include "glib/util/GLog.h"

const int GFileInputStream::BuffSize = 512;

GFileInputStream::GFileInputStream ( GVfs& vfs,
                                     const GString& name, 
                                     bool textMode )
                 :vfs(vfs),
                  path(name),
                  hfile(null),
                  autoClose(true),
                  textMode(textMode),
                  nextByte(-1),
                  buffPos(0),
                  buffCount(0),
                  buff(null)
{
   init();
}

GFileInputStream::GFileInputStream ( GVfs& vfs,
                                     const GFile& file, 
                                     bool textMode )
                 :vfs(vfs),
                  path(file.getFullPath()),
                  hfile(null),
                  autoClose(true),
                  textMode(textMode),
                  nextByte(-1),
                  buffPos(0),
                  buffCount(0),
                  buff(null)
{
   init();
}

GFileInputStream::GFileInputStream ( GVfs& vfs,
                                     GVfs::FileHandle hfile, 
                                     bool autoClose, 
                                     bool textMode,
                                     const GString& path )
                 :vfs(vfs),
                  path(path),
                  hfile(hfile),
                  autoClose(autoClose),
                  textMode(textMode),
                  nextByte(-1),
                  buffPos(0),
                  buffCount(0),
                  buff(null)
{
}

GFileInputStream::~GFileInputStream ()
{
   if (buff != null)
      delete [] buff;
   if (autoClose && hfile != null)
      vfs.closeFile(hfile);
}

void GFileInputStream::init ()
{
   if (!vfs.existFile(path))
      gthrow_(GFileNotFoundException("Filename: " + path));
   GError err;
   hfile = vfs.openFile(path, &err, GVfs::Mode_ReadOnly, GVfs::Create_Never, GVfs::Share_DenyNone);
   if (hfile == null)
   {
      GString sysMsg = err.getErrorMessage();
      GString msg("%s\n\nFilename:\n%s", GVArgs(sysMsg).add(path));
      gthrow_(GOpenFileException(msg));
   }
}

GVfs::FileHandle GFileInputStream::getFileHandle ()
{
   return hfile;
}

int GFileInputStream::readByte () const
{
   BYTE byte_;

   if (nextByte != -1)
   {
      byte_ = BYTE(nextByte);
      nextByte = -1;
   }
   else
   {
      if (buffPos < buffCount)
      {
         byte_ = buff[buffPos++];
      }
      else
      {
         if (buff == null)
            buff = new BYTE[BuffSize];

         // Read next block of buffered data.
         int numRead = 0;
         GError rc = vfs.readFromFile(hfile, buff, BuffSize, &numRead);
         if (rc != GError::Ok)
            gthrow_(GIOException("Filename: " + path));
         if (numRead < 1)
            return -1; // End of file stream.

         // ---
         byte_ = buff[0];
         buffCount = numRead;
         buffPos = 1;
      }
   }

   if (textMode)
   {
      if (byte_ == '\r')
      {
         nextByte = readByte();
         if (nextByte == '\n')
         {
            nextByte = -1;
            byte_ = '\n';
         }
      }
      else
      if (byte_ == 0x00 || byte_ == 0x1A) // If end-of-stream (null or ctrl+Z).
      {
         return -1;
      }
   }

   return byte_;
}

GString& GFileInputStream::readString ( GString& str ) const
{
   return GInputStream::readString(str);
}

int GFileInputStream::read ( void* buff, int count ) const
{
   if (count <= 0)
      return 0;
   int numRead = 0;
   if (textMode)
   {
      BYTE* byteBuff = (BYTE*) buff;
      while (numRead < count)
      {
         int bt = GFileInputStream::readByte();
         if (bt == -1) // If end of file stream.
         {
            if (numRead == 0)
               return -1; // End of file stream.
            break;
         }
         *byteBuff++ = BYTE(bt);
         numRead++;
      }
   }
   else
   {
      GError rc = vfs.readFromFile(hfile, buff, count, &numRead);
      if (rc != GError::Ok)
         if (rc != GError::BrokenPipe)
            gthrow_(GIOException(GString("%s (Filename: %s)", GVArgs(rc.getErrorMessage()).add(path))));
      if (numRead < 0)
         return -1; // End of file stream.
   }
   return numRead;
}

void GFileInputStream::readExact ( void* buff, int count ) const
{
   GInputStream::readExact(buff, count);
}
