/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/gui/tree/GTreePath.h"
#include "glib/gui/tree/GTreeNode.h"
#include "glib/exceptions/GIllegalArgumentException.h"

GTreePath::GTreePath ( GTreeNode& singlePath ) 
          :parentPath(null),
           lastPathComponent(&singlePath)
{
}

GTreePath::GTreePath ( const GArray<GTreeNode>& path ) 
          :parentPath(null),
           lastPathComponent(null)
{
   int s = path.getCount();
   if (s == 0)
      gthrow_(GIllegalArgumentException("path in GTreePath must be non null and not empty."));
   lastPathComponent = &path[s - 1];
   if (s > 1)
   {
      s -= 1;
      GArray<GTreeNode> parent(s);
      for (int i=0; i<s; i++)
      {
         GTreeNode& node = path.get(i);
         parent.add(&node, false);
      }
      parentPath = new GTreePath(parent);
   }
}

GTreePath::GTreePath ( const GTreePath& src )
          :parentPath(null),
           lastPathComponent(null)
{
   operator=(src);
}

GTreePath::~GTreePath ()
{
   delete parentPath;
}

GTreePath& GTreePath::operator= ( const GTreePath& src )
{
   this->parentPath = src.parentPath == null ? null : new GTreePath(*src.parentPath);
   this->lastPathComponent = src.lastPathComponent;
   return *this;
}

GTreeNode& GTreePath::getLastPathComponent () 
{
   return *lastPathComponent;
}

int GTreePath::getPathCount () const 
{
   int ret = 0;
   for (const GTreePath* path = this; path != null; path = path->parentPath)
      ret++;
   return ret;
}

GTreeNode& GTreePath::getPathComponent ( int idx ) 
{
   int pathLength = getPathCount();
   if (idx < 0 || idx >= pathLength)
      gthrow_(GIllegalArgumentException("Invalid index=" + GInteger::ToString(idx)));
   const GTreePath* path = this;
   for (int i = pathLength-1; i != idx; i--)
      path = path->parentPath;
   return *path->lastPathComponent;
}

bool GTreePath::isDescendant ( const GTreePath& aTreePath ) const
{
   const GTreePath* p = &aTreePath;
   if (p == this)
      return true;
   int pathLength = getPathCount();
   int oPathLength = p->getPathCount();
   if (oPathLength < pathLength)
      return false; // Can't be a descendant, has fewer components in the path.
   while (oPathLength-- > pathLength)
      p = p->getParentPath();
   return equals(*p);
}

const GTreePath* GTreePath::getParentPath () const
{
   return parentPath;
}
