/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_MUTABLETREENODE
#define __GLIB_MUTABLETREENODE

#include "glib/gui/tree/GTreeNode.h"
#include "glib/util/GArray.h"

/**
 * A <code>GMutableTreeNode</code> is a general-purpose node in a 
 * tree data structure. 
 *
 * A tree node may have at most one parent and 0 or more children.
 * <code>GMutableTreeNode</code> provides operations for examining 
 * and modifying a node's parent and children and also operations for 
 * examining the tree that the node is a part of. A node's tree is the set 
 * of all nodes that can be reached by starting at the node and following 
 * all the possible links to parents and children. A node with no parent 
 * is the root of its tree; a node with no children is a leaf. A tree may 
 * consist of many subtrees, each node acting as the root for its 
 * own subtree.
 * 
 * A <code>GMutableTreeNode</code> may also hold a reference to a 
 * user object, the use of which is left to the user. Asking a 
 * <code>GMutableTreeNode</code> for its string representation 
 * with {@link #toString} returns the string representation of its 
 * user object.
 * 
 * <b>This is not a thread safe class.</b>If you intend to use
 * a <code>GMutableTreeNode</code> (or a tree of {@link GTreeNode}s) 
 * in more than one thread, you need to do your own synchronizing. 
 * A good convention to adopt is synchronizing on the root node of a tree.
 *
 * @author  Leif Erik Larsen
 * @since   2006.01.06
 */
class GMutableTreeNode : public GTreeNode
{

private:

   /** This node's parent, or null if this node has no parent. */
   GMutableTreeNode* parent;

   /** Array of children, may be null if this node has no children. */
   GArray<GMutableTreeNode>* children;

   /** True if the node is able to have children. */
   bool allowsChildren;

   /** The icon to use if the node is closed (collapsed), or if it cannot be opened (allows no children). */
   class GIcon* iconClosed;

   /** The icon to use if the node is open (expanded). */
   class GIcon* iconOpened;

   bool autoDeleteIconClosed;
   bool autoDeleteIconOpened;

public:

   /**
    * Creates a tree node that has no initial parent and no children.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.06
    * @param   allowsChildren True if the new node should allow any child 
    *                      nodes to be added and contained.
    * @param   userObject  The optional user object to be associated with 
    *                      this node.
    * @param   autoDeleteUserObject True if we shall automatically destroy
    *                      the specified user object when the tree node it
    *                      self is destructed.
    * @param   iconClosedID The name of which icon resource to show when 
    *                      the node is closed/collapsed, or if the node 
    *                      contains no children.
    * @param   iconOpenedID The name of which icon resource to show when 
    *                      the node is opened/expanded.
    */
   explicit GMutableTreeNode ( bool allowsChildren = true,
                               GObject* userObject = null, 
                               bool autoDeleteUserObject = true,
                               const GString& iconClosedID = GString::Empty,
                               const GString& iconOpenedID = GString::Empty );

   virtual ~GMutableTreeNode ();

private:

   /** Disable the copy-constructor. */
   GMutableTreeNode ( const GMutableTreeNode& src ) : GTreeNode(null, false) {}

   /** Disable the assignment operator. */
   GMutableTreeNode& operator= ( const GMutableTreeNode& src ) { return*this; }

public:

   /**
    * Removes the specified child from its present parent (if it already has 
    * a parent), sets the child's parent to this node, and then adds the 
    * child to this node's child array at the specified index.
    * The specified child must not be null and must not be an ancestor of
    * this node.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.06
    * @param	child The node of which to insert under this node.
    * @param	index The index in this node's child array
    *				      where this node is to be inserted
    * @throws  GArrayIndexOutOfBoundsException If the specified 
    *                index is invalid.
    * @throws  GIllegalArgumentException If the specified child is 
    *                null or is an ancestor of this node.
    * @throws  GIllegalStateException If this node does not allow children.
    * @see	#isNodeDescendant
    */
   virtual void insert ( GMutableTreeNode* child, int index, bool autoDelete );

   /**
    * Return true if and only if the indexed child is to be automatically 
    * destroyed by us. That is if the <code>autoDelete</code> argument 
    * was true when inserting the child.
    *
    * @author  Leif Erik Larsen
    * @since   2006.02.02
    */
   int isAutoDeleteChildAt ( int index ) const;

   /**
    * Removes the indexed child from this node's children and sets that 
    * node's parent to null.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.06
    * @param	index The index of which child node to remove.
    * @throws  GArrayIndexOutOfBoundsException If the index is out of bounds.
    */
   virtual void remove ( int index, bool doDestroy = true );

   /**
    * Removes the specified child node from this node's child array, 
    * giving it a null parent.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.06
    * @param	node    The child of this node, of which to remove.
    * @throws  GIllegalArgumentException If the specified child is null
    *                  or is not a child of this node.
    */
   virtual void remove ( GMutableTreeNode* node, bool doDestroy = true );

   /**
    * Blindly sets this node's parent to the one specified, but does not 
    * change the parent's child array. This method is called from
    * {@link insert} and {@link remove} to reassign a child's parent, 
    * and it should not be called from anywhere else.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.06
    * @param	newParent	this node's new parent
    */
   virtual void setParent ( GMutableTreeNode* newParent );

   /**
    * Returns this node's parent or null if this node has no parent.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.06
    */
   virtual GTreeNode* getParent ();

   /**
    * Returns the child at the specified index in this node's child array.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.06
    * @param	index	  The index of which child node to get.
    * @throws  GArrayIndexOutOfBoundsException If the index is out of bounds.
    */
   virtual GTreeNode& getChildAt ( int index ) const;

   /**
    * Returns the number of children of this node.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.06
    */
   virtual int getChildCount () const;

   /**
    * Returns the index of the specified child in this node's child array.
    * If the specified node is not a child of this node, returns
    * <code>-1</code>.  This method performs a linear search and is O(n)
    * where n is the number of children.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.06
    * @param	node    The child node of which index to get.
    * @return	The index of the specified node in this node's child 
    *          array, or -1 if the specified node is a not
    *          a child of this node.
    */
   virtual int getIndex ( const GTreeNode& node ) const;

   /**
    * Determines whether or not this node is allowed to have children. 
    * If <code>allows</code> is false, all of this node's children are
    * removed.
    * 
    * Note: By default, a node allows children.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.06
    * @param	allows  True if this node is allowed to have children.
    */
   void setAllowsChildren ( bool allows );

   /**
    * Returns true if this node is allowed to have children.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.06
    */
   virtual bool getAllowsChildren () const;

   /**
    * Sets the user object for this node to the object specified.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.06
    * @param	newUserObject The Object that constitutes this node's 
    *                        user-specified data.
    * @see	   GTreeNode#getUserObject
    * @see	   #toString
    */
   virtual void setUserObject ( GObject* newUserObject, bool autoDelete );

   /**
    * Removes the subtree rooted at this node from the tree, giving this
    * node a null parent. Does nothing if this node is the root of its tree.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.06
    */
   virtual void removeFromParent ();

   /**
    * Removes all of this node's children, setting their parents to null.
    * If this node has no children, this method does nothing.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.06
    */
   void removeAllChildren ();

   /**
    * Removes <code>newChild</code> from its parent and makes it a child of
    * this node by adding it to the end of this node's child array.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.06
    * @see		#insert
    * @param	child   The node of which to add as a new child of this node.
    * @param   autoDelete True if we shall automatically destroy the 
    *                  specified child when this parent node is destructed.
    * @throws  GIllegalArgumentException If the specified child is null.
    * @throws  GIllegalStateException If this node does not allow	children.
    */
   void add ( GMutableTreeNode* child, bool autoDelete );

   /**
    * Returns true if <code>anotherNode</code> is an ancestor of this node
    * -- if it is this node, this node's parent, or an ancestor of this
    * node's parent. (Note that a node is considered an ancestor of itself.)
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.06
    * @see		#isNodeDescendant
    * @see		#getSharedAncestor
    * @param	anotherNode	The node to test as an ancestor of this node.
    */
   bool isNodeAncestor ( const GTreeNode& anotherNode ) const;

   /**
    * Returns true if <code>anotherNode</code> is a descendant of this node
    * -- if it is this node, one of this node's children, or a descendant of
    * one of this node's children.  Note that a node is considered a
    * descendant of itself.  If <code>anotherNode</code> is null, returns
    * false.  This operation is at worst O(h) where h is the distance from the
    * root to <code>anotherNode</code>.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.06
    * @see	   #isNodeAncestor
    * @see	   #getSharedAncestor
    * @param	anotherNode	The node to test as descendant of this node.
    */
   bool isNodeDescendant ( GMutableTreeNode& anotherNode ) const;

   /**
    * Returns the nearest common ancestor to this node and <code>aNode</code>.
    * Returns null if no such ancestor exists or if this node and
    * <code>aNode</code> are in different trees.
    * A node is considered an ancestor of itself.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.06
    * @see	   #isNodeAncestor
    * @see	   #isNodeDescendant
    * @param	aNode	The node of which to find common ancestor with.
    * @return	The nearest ancestor common to this node and <code>aNode</code>,
    *		      or null if none.
    */
   GTreeNode* getSharedAncestor ( GMutableTreeNode& aNode );

   /**
    * Returns true if and only if <code>aNode</code> is in the same tree
    * as this node.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.06
    * @see	   #getSharedAncestor
    * @see	   #getRoot
    */
   bool isNodeRelated ( const GMutableTreeNode& aNode ) const;

   /**
    * Returns the number of levels above this node -- the distance from
    * the root to this node. If this node is the root, returns 0.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.06
    */
   int getLevel () const;

   /**
    * Returns the root of the tree that contains this node.
    * The root is the ancestor with a null parent.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.06
    * @see	   #isNodeAncestor
    */
   GTreeNode& getRoot ();

   /**
    * Returns true if this node is the root of the tree.  The root is
    * the only node in the tree with a null parent; every tree has exactly
    * one root.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.06
    */
   bool isRoot () const;

   /**
    * Returns the node that follows this node in a preorder traversal of this
    * node's tree. Returns null if this node is the last node of the 
    * traversal.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.06
    */
   GMutableTreeNode* getNextNode ();

   /**
    * Returns the node that precedes this node in a preorder traversal of
    * this node's tree. Returns <code>null</code> if this node is the
    * first node of the traversal -- the root of the tree. 
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.06
    */
   GMutableTreeNode* getPreviousNode ();

   /**
    * Returns this node's first child.  
    * If this node has no children, throws {@link GNoSuchElementException}.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.06
    * @return	The first child of this node.
    * @throws  GNoSuchElementException If this node has no children.
    */
   GTreeNode& getFirstChild ();

   /**
    * Returns this node's last child. If this node has no children,
    * throws {@link GNoSuchElementException}.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.06
    * @throws  GNoSuchElementException If this node has no children.
    */
   GTreeNode& getLastChild ();

   /**
    * Returns the child in this node's child array that immediately
    * follows <code>aChild</code>, which must be a child of this node.  
    * If <code>aChild</code> is the last child, returns null.  
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.06
    * @throws  GIllegalArgumentException If <code>aChild</code> is
    *					not a child of this node.
    */
   GTreeNode* getChildAfter ( GTreeNode& aChild );

   /**
    * Returns the child in this node's child array that immediately
    * precedes <code>aChild</code>, which must be a child of this node.  
    * If <code>aChild</code> is the first child, returns null.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.06
    * @throws  GIllegalArgumentException If <code>aChild</code> is
    *						not a child of this node
    */
   GTreeNode* getChildBefore ( GTreeNode& aChild );

   /**
    * Returns true if <code>anotherNode</code> is a sibling of 
    * (has the same parent as) this node. A node is its own sibling. 
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.06
    * @param	anotherNode	Node to test as sibling of this node.
    */
   bool isNodeSibling ( const GTreeNode& anotherNode ) const;

   /**
    * Returns the number of siblings of this node. A node is its own sibling
    * (if it has no parent or no siblings, this method returns
    * <code>1</code>).
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.06
    */
   int getSiblingCount () const;

   /**
    * Returns the next sibling of this node in the parent's children array.
    * Returns null if this node has no parent or is the parent's last child.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.06
    * @return	The sibling of this node that immediately follows this node.
    */
   GMutableTreeNode* getNextSibling ();

   /**
    * Returns the previous sibling of this node in the parent's children
    * array. Returns null if this node has no parent or is the parent's
    * first child.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.06
    * @return	The sibling of this node that immediately precedes this node.
    */
   GMutableTreeNode* getPreviousSibling ();

   /**
    * Finds and returns the first leaf that is a descendant of this node --
    * either this node or its first child's first leaf.
    * Returns this node if it is a leaf.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.06
    * @see	   #isLeaf
    * @see	   #isNodeDescendant
    * @return	The first leaf in the subtree rooted at this node.
    */
   GMutableTreeNode& getFirstLeaf ();

   /**
    * Finds and returns the last leaf that is a descendant of this node --
    * either this node or its last child's last leaf. 
    * Returns this node if it is a leaf.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.06
    * @see	   #isLeaf
    * @see	   #isNodeDescendant
    * @return	The last leaf in the subtree rooted at this node.
    */
   GMutableTreeNode& getLastLeaf ();

   /**
    * Returns the leaf after this node or null if this node is the
    * last leaf in the tree.
    * 
    * In this implementation this operation is very inefficient. 
    * In order to determine the next node, this method first performs 
    * a linear search in the parent's child-list in order to find the 
    * current node. 
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.06
    * @see	   #isLeaf
    */
   GMutableTreeNode* getNextLeaf ();

   /**
    * Returns the leaf before this node or null if this node is the
    * first leaf in the tree.
    * 
    * In this implementation this operation is very inefficient. 
    * In order to determine the previous node, this method first performs 
    * a linear search in the parent's child-list in order to find the 
    * current node. 
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.06
    * @see		#isLeaf
    */
   GMutableTreeNode* getPreviousLeaf ();

   /**
    * Returns the result of calling <code>toString()</code> to this node's
    * user object, or an empty string if this node has no user object.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.06
    * @see	   #getUserObject
    */
   virtual GString toString () const;

   const class GIcon* getIconClosed () const;
   void setIconClosed ( const GString& iconID );
   void setIconClosed ( class GIcon* icon, bool autoDelete );
   const class GIcon* getIconOpened () const;
   void setIconOpened ( const GString& iconID );
   void setIconOpened ( class GIcon* icon, bool autoDelete );
};

#endif
