/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_DEFAULTTREEMODEL
#define __GLIB_DEFAULTTREEMODEL

#include "glib/gui/tree/GTreeModel.h"
#include "glib/gui/tree/GTreeModelListener.h"
#include "glib/util/GVector.h"
#include "glib/util/GArray.h"

/**
 * A simple tree data model that uses {@link GTreeNode}s.
 *
 * @author  Leif Erik Larsen
 * @since   2006.01.04
 */
class GDefaultTreeModel : public GTreeModel
{

protected:

   /** Root of the tree. */
   class GMutableTreeNode* root;

   /** True if we shall delete {@link #root} automatically upon destruction. */
   bool autoDeleteRoot;

   /**
    * Determines how the <code>isLeaf</code> method figures
    * out if a node is a leaf node. If true, a node is a leaf 
    * node if it does not allow children. (If it allows 
    * children, it is not a leaf node, even if no children
    * are present.) That lets you distinguish between <i>folder</i>
    * nodes and <i>file</i> nodes in a file system, for example.
    * 
    * If this value is false, then any node which has no 
    * children is a leaf node, and any node may acquire 
    * children.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.04
    * @see     GTreeNode#getAllowsChildren
    * @see     GTreeModel#isLeaf
    * @see     #setAsksAllowsChildren
    */
   bool asksAllowsChildren;

   /** Listeners. */
   GVector<GTreeModelListener*> listenerList;

public:

   /**
    * Creates a tree specifying whether any node can have children,
    * or whether only certain nodes can have children.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.04
    * @param   root The object that is the root node of the tree.
    *               Must be != null.
    * @param   autoDeleteRoot True if we receives the ownership 
    *               of the root object.
    * @see #asksAllowsChildren
    * @throws  GIllegalArgumentException if the specified root is null.
    */
   GDefaultTreeModel ( class GMutableTreeNode* root, bool autoDeleteRoot );

   /**
    * @author  Leif Erik Larsen
    * @since   2006.01.04
    */
   virtual ~GDefaultTreeModel ();

public:

   /**
    * Sets whether or not to test leafness by asking {@link #getAllowsChildren}
    * or {@link GTreeNodes#isLeaf}.  If newValue is true, 
    * {@link #getAllowsChildren} is called, otherwise {@link GTreeNodes#isLeaf} 
    * is called.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.04
    * @see     #getAsksAllowsChildren
    * @see     #asksAllowsChildren
    */
   void setAsksAllowsChildren ( bool newValue );

   /**
    * Tells how leaf nodes are determined.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.04
    * @return  True if only nodes which do not allow children are
    *          leaf nodes. False if nodes which have no children
    *          (even if allowed) are leaf nodes.
    * @see     #setAsksAllowsChildren
    * @see     #asksAllowsChildren
    */
   bool getAsksAllowsChildren () const;

   /**
    * Sets the root to <code>root</code>. A null <code>root</code> implies
    * the tree is to display nothing, and is legal.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.04
    */
   void setRoot ( class GMutableTreeNode* root, bool autoDeleteRoot );

   /**
    * Returns the root of the tree. 
    * Returns null only if the tree has no nodes.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.04
    */
   class GTreeNode* getRoot ();

   /**
    * Returns the index of child in parent.
    * If either the parent or child is <code>null</code>, returns -1.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.04
    * @param   parent  A node in the tree, obtained from this data source.
    * @param   child   The node we are interested in.
    * @return  The index of the child in the parent.
    */
   virtual int getIndexOfChild ( const class GTreeNode& parent, const class GTreeNode& child ) const;

   /**
    * Returns the child of <i>parent</i> at index <i>index</i> in the parent's
    * child array. The <i>parent</i> must be a node previously obtained from
    * this data source.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.04
    * @param   parent  A node in the tree, obtained from this data source.
    * @return  The child of <i>parent</i> at index <i>index</i>.
    * @throws  GArrayIndexOutOfBoundsException if the specified index is 
    *                                          outside range.
    */
   class GTreeNode& getChild ( class GTreeNode& parent, int index );

   /**
    * Returns the number of children of <I>parent</I>.  Returns 0 if the node
    * is a leaf or if it has no children.  <I>parent</I> must be a node
    * previously obtained from this data source.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.04
    * @param   parent    A node in the tree, obtained from this data source.
    * @return  The number of children of the node <i>parent</i>.
    */
   int getChildCount ( const class GTreeNode& parent ) const;

   /** 
    * Returns whether the specified node is a leaf node.
    * The way the test is performed depends on the
    * <code>askAllowsChildren</code> setting.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.04
    * @param   node    The node of which to check.
    * @return  True if the node is a leaf node.
    * @see     #asksAllowsChildren
    * @see     GTreeModel#isLeaf
    */
   bool isLeaf ( const class GTreeNode& node ) const;

   /**
    * This sets the user object of the TreeNode identified by path
    * and posts a node changed.  If you use custom user objects in
    * the TreeModel you're going to need to subclass this and
    * set the user object of the changed node to something meaningful.
    */
   void valueForPathChanged ( class GTreePath& path, GObject* newValue, bool autoDelete );

   /**
    * Invoked this to insert newChild at location index in parents children.
    * This will then message {@link #nodesWereInserted} to create the 
    * appropriate event. This is the preferred way to add children as it 
    * will create the appropriate event.
    */
   void insertNodeInto ( class GMutableTreeNode* newChild, 
                         class GMutableTreeNode& parent, 
                         int index, 
                         bool autoDeleteChild );

   /**
    * Remove the specified node from its parent. 
    * This will message nodesWereRemoved to create the appropriate event. 
    * This is the preferred way to remove a node as it handles the event 
    * creation for you.
    *
    * @author  Leif Erik Larsen
    * @since   2006.02.02
    * @param   node    The node of which to remove from its parent.
    * @param   doDestroyNode True if we shall delete the specified node
    *                  before returning from this method, or else false.
    *                  Note that we will <i>never</i> delete the node if 
    *                  the argument <code>autoDeleteChild</code> was false
    *                  upon node insertion.
    */
   void removeNodeFromParent ( class GTreeNode& node, bool doDestroyNode = true );

   /**
    * Invoke this method after you've changed how node is to be
    * represented in the tree.
    */
   void nodeChanged ( const class GTreeNode& node );

   /**
    * Invoke this method after you've inserted some {@link GTreeNode}s 
    * into the specified parent. childIndexes should be the index of the 
    * new elements and must be sorted in ascending order.
    */
   void nodesWereInserted ( class GTreeNode& parent, const GVector<int>& childIndexes );
    
   /**
    * Invoke this method after you've removed some TreeNodes from
    * node. childIndexes should be the index of the removed elements and
    * must be sorted in ascending order. And removedChildren should be
    * the array of the children objects that were removed.
    */
   void nodesWereRemoved ( const class GTreeNode& node, 
                           const GVector<int>& childIndexes, 
                           GArray<class GTreeNode>& removedChildren );

   /**
    * Invoke this method after you've changed how the children identified by
    * childIndexes are to be represented in the tree.
    */
   void nodesChanged ( const class GTreeNode& node, const GVector<int>& childIndexes );

   /**
    * Adds a listener for the events on this Tree Model.
    *
    * @see     #removeTreeModelListener
    * @param   l   The listener to add.
    */
   virtual void addTreeModelListener ( class GTreeModelListener* l );

   /**
    * Removes a listener previously added with {@link #addTreeModelListener}.
    *
    * @see     #addTreeModelListener
    * @param   l   The listener to remove.
    */  
   virtual void removeTreeModelListener ( class GTreeModelListener* l );

protected:

   /**
    * Notifies all listeners that have registered interest for
    * notification on this event type.  The event instance 
    * is lazily created using the parameters passed into 
    * the fire method.
    *
    * @param path the path to the root node
    * @param childIndexes the indices of the changed elements
    * @param children the changed elements
    * @see EventListenerList
    */
   void fireTreeNodesChanged ( const class GTreePath& path, 
                               const GVector<int>& childIndexes,  
                               const GArray<class GTreeNode>& children );

   /**
    * Notifies all listeners that have registered interest for
    * notification on this event type.  The event instance 
    * is lazily created using the parameters passed into 
    * the fire method.
    *
    * @param path the path to the root node
    * @param childIndexes the indices of the new elements
    * @param children the new elements
    * @see EventListenerList
    */
   void fireTreeNodesInserted ( const class GTreePath& path, 
                                const GVector<int>& childIndexes,  
                                const GArray<class GTreeNode>& children );

   /**
    * Notifies all listeners that have registered interest for
    * notification on this event type.  The event instance 
    * is lazily created using the parameters passed into 
    * the fire method.
    *
    * @param path the path to the root node
    * @param childIndexes the indices of the removed elements
    * @param children the removed elements
    * @see EventListenerList
    */
   void fireTreeNodesRemoved ( const class GTreePath& path, 
                               const GVector<int>& childIndexes,  
                               const GArray<class GTreeNode>& children );
};

#endif
