/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_KEYMESSAGE
#define __GLIB_KEYMESSAGE

#include "glib/primitives/GString.h"
#include "glib/gui/event/GKey.h"
#include "glib/gui/event/GWindowMessage.h"

/**
 * Handly class to represent a keyboard event.
 *
 * @author  Leif Erik Larsen
 * @since   2000.01.26
 * @see     GWindow#onKey
 */
class GKeyMessage : public GObject, public GKey
{
   friend class GWindow;

   private:

      bool up;
      bool alt;
      bool ctrl;
      bool shift;
      bool numKeypadKey;
      int repeat;
      int scanCode;
      char chr;
      int vk;

      mutable KeyCode keyCode;
      mutable GString name;
      mutable bool nameIsDirty;
      mutable bool keyCodeIsDirty;
      mutable GWindowMessage originalMsg;

   private:

      /**
       * Make a key message object based on the specified window message.
       */
      explicit GKeyMessage ( const GWindowMessage& msg );

      virtual ~GKeyMessage ();

   public:

      /**
       * Get the character code of this key-message.
       * The return character will be a zero-character if the key does not
       * include any character code (e.g. F1, HOME, INSERT, NUMLOCK, etc.).
       * We will return the character code of the key even if the key is 
       * not a pure character only (e.g. ALT+A will return 'A' and 
       * CTRL+C will return 'C').
       *
       * @author  Leif Erik Larsen
       * @since   2006.04.27
       * @see     #isPureCharacter
       */
      char getCharacter () const;

      GKey::KeyCode getCode () const;

      /** 
       * @author  Leif Erik Larsen
       * @since   2004.08.03
       */
      const GString& getName () const;

      /**
       * Get a reference to the system dependent message that 
       * was used to construct this key message.
       *
       * @author  Leif Erik Larsen
       * @since   2004.09.07
       */
      const GWindowMessage& getOriginalMessage () const;

      int getRepeat () const;

      int getScanCode () const;

      int getVirtualKey () const;

      bool isAlphaChar ( bool evenIfShift = false ) const;

      /**
       * Return true if and only if the ALT key is currently down.
       *
       * @author  Leif Erik Larsen
       * @since   2000.01.26
       * @see     #isAnyShiftKeyDown
       */
      bool isAltDown () const;

      /** 
       * Return true if and only if at least one of the so-called
       * shift-keys (e.g. ALT, CTRL, SHIFT) are currently down.
       *
       * @author  Leif Erik Larsen
       * @since   2004.07.13
       * @see     #isAltDown
       * @see     #isCtrlDown
       * @see     #isShiftDown
       */
      bool isAnyShiftKeyDown () const;

      /**
       * Return true if the key message represents a pure character,
       * without any shift-key except than SHIFT (not ALT or CONTROL).
       *
       * @author  Leif Erik Larsen
       * @since   2006.04.27
       * @see     #getCharacter
       */
      bool isPureCharacter () const;

      /**
       * Return true if and only if the CTRL key is currently down.
       *
       * @author  Leif Erik Larsen
       * @since   2000.01.26
       * @see     #isAnyShiftKeyDown
       */
      bool isCtrlDown () const;

      bool isDigitChar ( bool evenIfShift = false ) const;
      bool isNumericKeypadKey () const;
      bool isKeyDown () const;
      bool isKeyUp () const;

      /**
       * Return true if and only if the left and/or right SHIFT key 
       * is currently down.
       *
       * @author  Leif Erik Larsen
       * @since   2000.01.26
       * @see     #isAnyShiftKeyDown
       */
      bool isShiftDown () const;
};

#endif
