/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_WORKERTHREADADAPTER
#define __GLIB_WORKERTHREADADAPTER

#include "glib/primitives/GString.h"

/**
 * This class declares the methods that can be defined by user code in order
 * to interact with a worker thread that is managed by {@link GWorkerThread}.
 *
 * All methods of this adapter has a default implementation that does 
 * nothing but return, so user defined subclass needs only to override 
 * those methods of interest.
 *
 * Se the documentation of {@link GWorkerThread} for more information
 * and a working sample of how to write a background worker thread
 * using this adapter.
 *
 * This class is intended for subclassing and this is the reason why
 * the constructor and all our methods are protected. Only the worker
 * thread implementation should be able to call these methods, and this
 * is the reason why the class {@link GWorkerThread} is declared to
 * be a "friend" of ours.
 *
 * @author  Leif Erik Larsen
 * @since   2004.01.23
 * @see     GWorkerThread
 */
class GWorkerThreadAdapter : public GObject
{
   friend class GWorkerThread;

   protected:

      GWorkerThreadAdapter ();

   public:

      virtual ~GWorkerThreadAdapter ();

   protected:

      /**
       * This method is automatically called by the GUI-thread when the 
       * monitor dialog receives the 
       * {@link GDialogMessageHandler#GM_COMMAND} message.
       *
       * @author  Leif Erik Larsen
       * @since   2004.01.23
       * @param   worker  A reference to the calling GWorkerThread-instance.
       * @param   monitor A reference to the monitor dialog panel.
       * @param   cmdID   The command name ID string.
       */
      virtual void onWorkerThreadCommand ( class GWorkerThread& worker,
                                           class GDialogPanel& monitor, 
                                           const GString& cmdID );

      /**
       * This method is automatically called by the GUI-thread when the 
       * monitor dialog receives the 
       * {@link GDialogMessageHandler#GM_CTRLCHANGED} message.
       *
       * @author  Leif Erik Larsen
       * @since   2004.01.23
       * @param   worker   A reference to the calling GWorkerThread-instance.
       * @param   monitor  A reference to the monitor dialog panel.
       * @param   compID   Component name ID string.
       * @param   newValue The new component value object.
       */
      virtual void onWorkerThreadCtrlChanged ( class GWorkerThread& worker,
                                               class GDialogPanel& monitor, 
                                               const GString& compID, 
                                               const GString& newValue );

      /**
       * This method is automatically called by the GUI-thread when the 
       * monitor dialog receives the 
       * {@link GDialogMessageHandler#GM_DISMISSDIALOG} message.
       *
       * @author  Leif Erik Larsen
       * @since   2004.01.23
       * @param   worker  A reference to the calling GWorkerThread-instance.
       * @param   monitor A reference to the monitor dialog panel.
       * @see     #onWorkerThreadInitDialog
       */
      virtual void onWorkerThreadDismissDialog ( class GWorkerThread& worker,
                                                 class GDialogPanel& monitor );

      /**
       * This method is automatically called by the GUI-thread when the 
       * monitor dialog receives the 
       * {@link GDialogMessageHandler#GM_FOCUSLOSS} message.
       *
       * @author  Leif Erik Larsen
       * @since   2004.01.23
       * @param   worker  A reference to the calling GWorkerThread-instance.
       * @param   monitor A reference to the monitor dialog panel.
       * @param   focusLossID The ID string of which component has lost focus.
       * @param   focusSetID  The ID string of which component has got focus.
       */
      virtual void onWorkerThreadFocusLoss ( class GWorkerThread& worker,
                                             class GDialogPanel& monitor, 
                                             const GString& focusLossID, 
                                             const GString& focusSetID );

      /**
       * This method is automatically called by the GUI-thread when the 
       * monitor dialog receives the 
       * {@link GDialogMessageHandler#GM_FOCUSSET} message.
       *
       * @author  Leif Erik Larsen
       * @since   2004.01.23
       * @param   worker  A reference to the calling GWorkerThread-instance.
       * @param   monitor A reference to the monitor dialog panel.
       * @param   focusSetID  The ID string of which component has got focus.
       * @param   focusLossID The ID string of which component has lost focus.
       */
      virtual void onWorkerThreadFocusSet ( class GWorkerThread& worker,
                                            class GDialogPanel& monitor, 
                                            const GString& focusSetID, 
                                            const GString& focusLossID );

      /**
       * This method is automatically called by the GUI-thread as soon as 
       * the worker thread has finished, regardless of how it was finished.
       * It is called before the progress dialog (monitor) is dismissed.
       *
       * @author  Leif Erik Larsen
       * @since   2004.01.23
       * @param   worker  A reference to the calling GWorkerThread-instance.
       * @param   monitor A reference to the monitor dialog panel.
       * @param   success The success flag of the worker thread.
       */
      virtual void onWorkerThreadHasFinished ( class GWorkerThread& worker,
                                               class GDialogPanel& monitor, 
                                               bool success );

      /**
       * This method is automatically called by the GUI-thread as soon as 
       * the worker thread has been started.
       *
       * @author  Leif Erik Larsen
       * @since   2004.01.23
       * @param   worker  A reference to the calling GWorkerThread-instance.
       * @param   monitor A reference to the monitor dialog panel.
       */
      virtual void onWorkerThreadHasStarted ( class GWorkerThread& worker,
                                              class GDialogPanel& monitor );

      /**
       * This method is automatically called by the GUI-thread when the 
       * monitor dialog receives the 
       * {@link GDialogMessageHandler#GM_INITDIALOG} message.
       *
       * @author  Leif Erik Larsen
       * @since   2004.01.23
       * @param   worker  A reference to the calling GWorkerThread-instance.
       * @param   monitor A reference to the monitor dialog panel.
       * @see     #onWorkerThreadDismissDialog
       */
      virtual void onWorkerThreadInitDialog ( class GWorkerThread& worker, 
                                              class GDialogPanel& monitor );

      /**
       * This method will be called by the GUI-thread when 
       * {@link GWorkerThread#requestStop} is called for the first 
       * time since the worker thread was started.
       *
       * @author  Leif Erik Larsen
       * @since   2004.01.23
       * @param   worker  A reference to the calling GWorkerThread-instance.
       * @param   monitor A reference to the monitor dialog panel.
       * @param   success The success-flag that was given
       *                  to {@link GWorkerThread#requestStop}.
       */
      virtual void onWorkerThreadRequestStop ( class GWorkerThread& worker, 
                                               class GDialogPanel& monitor, 
                                               bool success );

      /**
       * This method is automatically called by the GUI-thread at a regular 
       * frequency as long as the worker thread is still running. The sub-
       * class can override this method and implement e.g. the code that
       * updates something like a progress bar on the monitor dialog.
       *
       * In addition, this method is also automatically called once just 
       * before the worker thread returns from its main entry point, but 
       * before method {@link #onWorkerThreadHasFinished} is called. This 
       * makes sense for instance to make sure that the progress bar is 
       * updated with something like "100%" before the worker thread 
       * actually exits.
       *
       * We will not call this method while the worker thread is idle
       * waiting for the user to respond to a message box displayed
       * via {@link GWorkerThread#showMessageBox}.
       *
       * This method is not called at all if the "monitorUpdtMillis"
       * parameter of the {@link GWorkerThread}-constructor was given 
       * an argument value that was less than or equal to zero.
       *
       * @author  Leif Erik Larsen
       * @since   2004.01.23
       * @param   worker  A reference to the calling GWorkerThread-instance.
       * @param   monitor A reference to the monitor dialog panel.
       */
      virtual void onWorkerThreadUpdateMonitor ( class GWorkerThread& worker,
                                                 class GDialogPanel& monitor );

      /**
       * This method is called by the GUI-thread when the worker thread 
       * has called {@link GWorkerThread#sendUserMessageToMonitor}.
       *
       * @author  Leif Erik Larsen
       * @since   2004.01.23
       * @param   worker  A reference to the calling GWorkerThread-instance.
       * @param   monitor A reference to the monitor dialog panel.
       * @param   msgID   A copy of the string as was given to
       *                  {@link GWorkerThread#sendUserMessageToMonitor}.
       * @param   userParam A reference to the same object as was given to
       *                  {@link GWorkerThread#sendUserMessageToMonitor}.
       * @see     #sendUserMessageToMonitor
       */
      virtual void onWorkerThreadUserEvent ( class GWorkerThread& worker,
                                             class GDialogPanel& monitor, 
                                             const GString& msgID, 
                                             GObject* userParam );

      /**
       * This is the entry point of the worker thread. The sub-class should
       * override this method in order to implement the code of which to be
       * executed by the background worker thread.
       *
       * The worker thread is considered to be finsihed when this
       * method returns.
       *
       * @author  Leif Erik Larsen
       * @since   2004.01.23
       * @param   worker  A reference to the calling GWorkerThread-instance.
       * @throws  GException It is valid for the implementing code to 
       *                     throw an exception to signal an error.
       */
      virtual void runTheWorkerThread ( class GWorkerThread& worker );
};

#endif

