/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_WORKERTHREAD
#define __GLIB_WORKERTHREAD

#include "glib/primitives/aptr.h"
#include "glib/gui/GMessageBox.h"
#include "glib/gui/GDialogFrame.h"
#include "glib/gui/GWorkerThreadAdapter.h"
#include "glib/util/GEventSemaphore.h"
#include "glib/GThread.h"

/**
 * This class is indented to be used as the base for something like a
 * background worker thread with a modal progress bar that can be canceled 
 * by the user. When the user cancels the progress bar the background worker 
 * thread is requested to stop and the modal progress bar is dismissed as 
 * soon as the worker thread has actually exited.
 *
 * <a name=TheGUIThread><b><i>The GUI Thread</i></b></a>
 * There are several topics to discuss regarding the need for this class and
 * other classes regarding the same mechanism. First, in most GUI programs
 * there is one and only one thread that is used to execute the GUI
 * events that are produced as a result of the user interactivities on
 * the various windows, dialogs and components of the program. This is
 * typically the main thread of the program, which is also often referred
 * to as <i>the GUI thread</i>.
 *
 * Since there is usually only one such GUI thread active at a time the
 * following problem typically occurs in most GUI programs: When the user
 * clicks on a menu item or a push button to perform some lengthy operation
 * then the GUI thread is used to dispatch that command event. As long as
 * this GUI thread is busy performing the command code the user is unable
 * to perform some other task within the same program. This happens because
 * the GUI thread is not free to dispatch the user events that can occur
 * while the command is about being executed, because the one and only GUI
 * thread is already busy. This includes the component repaint mechanism as
 * well. You can often see this effect visually in that while the GUI
 * thread is busy the user interface of the program is not updated
 * properly.
 *
 * Behaviour like this gives a bad impression to the user. 
 * The program seems to be unprofessional and poorly implemented. 
 * What the user want is to be able to perform some other task
 * while the initiated task performs in the background. The users also wants
 * to be able to cancel the initiated lengthy command. This can be done only
 * if the GUI thread is free to receive the GUI events that are required to
 * handle the cancel operation it self.
 *
 * <b>Therefore, the one and only GUI thread is recomended not to be busy for
 * any period longer than a few milliseconds.</b> This means that whenever
 * a lengthy command is to be initiated (upon request from the GUI thread)
 * the command should be executed in a background thread so that the GUI
 * thread it self can remain free to dispatch other GUI events.
 *
 * This class is implemented to make it easy to respect and implement the
 * above requirement in our GUI programs. There are basically two ways 
 * to use this class to get what you want; 1) by adapting a user defined 
 * {@link GWorkerThreadAdapter} object to the GWorkerThread object, or
 * 2) by subclassing GWorkerThread. Method 1 is usually the one that is 
 * easiest and safest to implement in programs, so let us se a working 
 * example of that way of doing it...
 *
 * <h2>Sample of worker-thread-by-adapter</h2>
 *
 * <pre>
 *
 * class MyWorker : public GWorkerThreadAdapter
 * {
 *    private:
 *
 *       GAtomicCounter<int> workerThreadCount;
 *       int workerThreadMax;
 *
 *    public:
 *
 *       MyWorker ()
 *       {
 *          workerThreadCount = 0;
 *          workerThreadMax = 200;
 *       }
 *
 *    private:
 *
 *       virtual void runTheWorkerThread ( GWorkerThread& worker )
 *       {
 *          while (!worker.isStopRequested())
 *          {
 *             // Do some work, but periodically check and respect if we 
 *             // are requested to cancel.
 *             GThread::Sleep(50);  
 *             if (worker.isStopRequested())
 *                break; // We are requested by user to cancel!
 *             if (workerThreadCount++ >= workerThreadMax)
 *                break; // Ok, we are finished!
 *             if (workerThreadCount == workerThreadMax/2) 
 *             {
 *                GString txt = "Do you want to continue?";
 *                GMessageBox::Answer answ = worker.showWorkerMessageBox(txt, GMessageBox::TYPE_QUESTION, "Yn2");
 *                if (answ != GMessageBox::IDYES)
 *                   break;
 *             }
 *          }
 *       }
 *
 *       virtual void onWorkerThreadInitDialog ( GWorkerThread& worker, 
 *                                               GDialogPanel& monitor )
 *       {
 *          GWindow& comp = monitor.getComponentByID("PROGRESSBAR");
 *          GProgressBar& p = dynamic_cast<GProgressBar&>(comp);
 *          p.setMinValue(0);
 *          p.setMaxValue(workerThreadMax);
 *          p.setCurrentValue(workerThreadCount);
 *       }
 *
 *       virtual void onWorkerThreadUpdateMonitor ( GWorkerThread& worker, 
 *                                                  GDialogPanel& monitor )
 *       {
 *          monitor.setComponentValue("PROGRESSBAR", workerThreadCount);
 *       }
 *
 *       virtual void onWorkerThreadCommand ( GWorkerThread& worker, 
 *                                            GDialogPanel& monitor, 
 *                                            const GString& cmdID )
 *       {
 *          if (cmdID == "DLGCANCEL")
 *          {
 *             monitor.setComponentEnabled(cmdID, false);
 *             worker.requestStop(false);
 *          }
 *       }
 * };
 *
 * </pre>
 *
 * To use the above class in your code, you can use a code fragment
 * like this:
 *
 * <pre>
 *
 * void cmdMyWorker ()
 * {
 *    MyWorker adapter;
 *    bool autoDeleteAdapter = false;
 *    GWorkerThread wt("MyProgressDialogResourceID", 50, &adapter, autoDeleteAdapter);
 *    if (wt.workModal(mainWin))
 *       mainWin.showMessageBox("The worker thread has finished.");
 *    else
 *       mainWin.showMessageBox("The worker thread was canceled by the user.");
 * }
 *
 * </pre>
 *
 * @author  Leif Erik Larsen
 * @since   2000.09.26
 */
class GWorkerThread : public GWorkerThreadAdapter
{
   private:

      /**
       * This will automatically be updated with the total number of
       * milliseconds that are spent waiting for user to respond to a
       * message displayed via the method {@link #showMessageBox}.
       *
       * Can be used by the subclass for instance when it needs to
       * calculate how much real time the whole task took, ignoring
       * all the (if any) idle time used to wait for the user.
       *
       * @author  Leif Erik Larsen
       * @since   2000.10.12
       * @see     #getTimeMillisUsedWaitingForUserAnswers
       */
      ulonglong timeUsedWaitingForUserAnswersMillis;

      /**
       * The system time in milliseconds when the worker thread
       * begun. That is the time when the worker thread is about
       * to call the method {@link #runTheWorkerThread}.
       *
       * @author  Leif Erik Larsen
       * @since   2001.03.13
       * @see     #getTimeMillisSinceStart
       */
      ulonglong timeStartMillis;

      /**
       * Number of milliseconds between each automatic call
       * to {@link #onWorkerThreadUpdateMonitor}.
       */
      int updtMillis;

      /** False until worker thread has proven a success. */
      bool success;

      /** True when worker thread is requested to stop. */
      bool stopIsRequested;

      /** Will be true while inside {@link #workModal}. */
      bool executing;

      /** Will be true while inside {@link #runTheWorkerThread}. */
      bool workerIsRunning;

      /** Will be true while inside {@link #showMessageBox}. */
      bool isInMsgBox;

      /** Will be true if the worker thread message box has been shown at all. */
      bool msgBoxHasBeenShownAtLeastOnce;

      /** True if the parent window of the progress dialog was active when the worker thread was requested to start. */
      bool parentWinWasActiveUponStartWorker;

      /** True if the monitor dialog frame was visible and active when the worker thread was about to stop. */
      bool monitorDlgWasActiveUponEndWorker;

      /** In case of an exception in the worker thread, this contains the exception text. */
      GString workerThreadExceptionMessage;

      /** In case of an exception in the worker thread, this contains the exception stack trace (if available). */
      GString workerThreadExceptionStackTrace;

      /** True if we shall not display any message box to the user in case of an exception in the worker thread. */
      bool noExceptionMsgBox;

      /**
       * This inner class implements the worker thread object.
       * This is the thread that is used to do the worker job,
       * which is to be done outside of the GUI-thread.
       * 
       * @author  Leif Erik Larsen
       * @since   2004.01.23
       */
      class BckThread : public GThread
      {
         friend class GWorkerThread;
         GWorkerThread& owner;
         BckThread ( const GString& threadName, GWorkerThread& owner );
         virtual ~BckThread ();
         /** Disable the copy constructor. */
         BckThread ( const BckThread& src ) : owner(src.owner) {}
         /** Disable the assignment operator. */
         BckThread& operator= ( const BckThread& ) { return *this; }
         virtual void run ();
      };

      /** The background worker thread object. */
      BckThread bckThread;

      /**
       * This is the message handler of the monitor dialog panel.
       * It should not be overridden or visible to the out side world,
       * because it contains some critical and private logic that can
       * easily be broken if it is overridden. Therefore this class
       * is declared private.
       *
       * If you need to override the handling of various
       * messages you should rather override the corresponding event
       * handler methods, such as {@link #onWorkerThreadInitDialog},
       * {@link #onWorkerThreadCommand} and 
       * {@link #onWorkerThreadCtrlChanged}.
       *
       * @author  Leif Erik Larsen
       * @since   2000.09.27
       */
      class MsgProc : public GDialogMessageHandler
      {
         friend class GWorkerThread;
         private:
            GWorkerThread& wthread;
            MsgProc ( GWorkerThread& wthread );
            virtual bool handleDialogMessage ( GDialogMessage& msg );
      };

      /** The message handler for the progress bar dialog panel. */
      MsgProc mproc;

      /** The user defined worker thread adapter, or null if no such adapter. */
      GWorkerThreadAdapter* adapter;

      /** True if we shall automatically destroy the adapter. */
      bool autoDelAdptr;

      /** @see #workModal */
      int stayInvisibleTimerMillis;

      /**
       * The dialog panel used to monitor the worker thread.
       * Will be null when the worker thread is not
       * within {@link #runTheWorkerThread}.
       */
      GDialogPanel* monitorDlg;

      /** Used to prevent flashing of owner window frame. */
      GFrameWindow* topFrame;

      /** The progress monitor dialog frame window. */
      aptr<GDialogFrame> monitorFrame;

      /** Counts the number of worker thread instances created by the program. */
      static int InstanceCounter;

   protected:

      /**
       * An instance of this inner class will be created when the
       * worker thread is calling {@link #showMessageBox}.
       *
       * @author  Leif Erik Larsen
       * @since   2000.09.27
       * @see     #showGuiMessageBoxImpl
       */
      class MessageBoxInfo : public GObject
      {
         friend class GWorkerThread;

         public:

            const GString msg;
            const GMessageBox::Type type;
            const GString flags;
            const GString title;
            const bool monoFont;
            const GString userText1;
            const GString userText2;
            const GString userText3;

         private:

            GMessageBox::Answer answer;

         private:

            MessageBoxInfo ( const GString& msg, 
                             GMessageBox::Type type, 
                             const GString& flags, 
                             const GString& title, 
                             bool monoFont,
                             const GString& userText1,
                             const GString& userText2,
                             const GString& userText3 );

            virtual ~MessageBoxInfo ();
      };

   private:

      /**
       * An instance of this inner class will be created when the
       * worker thread is calling {@link #sendUserMessageToMonitor}.
       *
       * @author  Leif Erik Larsen
       * @since   2003.11.27
       */
      class UserMsgData : public GObject
      {
         friend class GWorkerThread;

         private:

            GString id;
            GObject* userData;

         private:

            UserMsgData ( const GString& id, GObject* userData );
            virtual ~UserMsgData ();
      };

   friend class MsgProc;
   friend class BckThread;

   public:

      /**
       * The case sensitive name of the timer that is used to call 
       * {@link #onWorkerThreadUpdateMonitor}
       * at a fixed frequency.
       *
       * @author  Leif Erik Larsen
       * @since   2001.05.05
       */
      static const GString PROGRESS_TIMER_ID;

      /**
       * The case sensitive name of the timer that is used make the 
       * progress dialog visible after some small time.
       *
       * @author  Leif Erik Larsen
       * @since   2004.05.19
       */
      static const GString STAY_INVISIBLE_TIMER_ID;

   public:

      /**
       * Create a new Worker Thread object. To execute the worker thread,
       * use {@link #workModal} on the new object.
       *
       * @author  Leif Erik Larsen
       * @since   2000.09.26
       * @param   dlgResourceID The ID of which dialog resource to use as the
       *                      template for the monitor dialog. The specified
       *                      dialog must exist in the default resource table
       *                      of the calling application.
       * @param   monitorUpdtMillis  Number of milliseconds between each 
       *                      automatic call to {@link #onWorkerThreadUpdateMonitor}. 
       *                      If a value less than or equal to zero is 
       *                      specified then we will not call 
       *                      onWorkerThreadUpdateMonitor() automatically at all.
       *                      In that case the worker thread should probably 
       *                      manually call {@link #updateMonitor} regularly.
       * @param   adapter     The user defined worker thread adapter,
       *                      which we will automatically call by 
       *                      corresponding methods of this class if our
       *                      subclass (if any) does not override them.
       * @param   autoDeleteAdapter  True if we shall automatically destroy 
       *                      the specified worker thread adapter when 
       *                      we are destroyed.
       * @param   threadName  The name of the worker thread. The name 
       *                      might contain one %d-tag, which will be 
       *                      replaced by an instance counter if defined.
       *                      This makes sure each worker thread instance
       *                      get its own unique thread name by default.
       */
      explicit GWorkerThread ( const GString& dlgResourceID,
                               int monitorUpdtMillis = 0, 
                               GWorkerThreadAdapter* adapter = null,
                               bool autoDeleteAdapter = true,
                               const GString& threadName = "WorkerThread%d" );

      virtual ~GWorkerThread ();

   private:

      /** Disable the copy constructor. */
      GWorkerThread ( const GWorkerThread& src ) 
         : bckThread(GString::Empty, *this), mproc(*this) {}

      /** Disable the assignment operator. */
      GWorkerThread& operator= ( const GWorkerThread& ) { return *this; }

   public:

      /**
       * Show the progress dialog, start the worker thread and wait
       * for it to finish.
       *
       * This will create and execute the specified dialog resource as a modal
       * dialog, and use that dialog as the monitor of the worker thread.
       * This dialog is typically containing something like a progress bar
       * and a cancel button, making it possible for the user to stop the
       * thread on request. However, it is up to the sub-class to override
       * {@link #onWorkerThreadUpdateMonitor} and update the progress bar.
       *
       * <b>Mark</b> that this method should be used by the
       * <a HREF="#TheGUIThread">GUI thread</a> only.
       *
       * @author  Leif Erik Larsen
       * @since   2000.09.26
       * @param   parentWin      The window panel of which to be used as the
       *                         parent of the modal monitor dialog.
       * @param   stayInvisibleTimerMillis The number of milliseconds to wait 
       *                         until the progress dialog is actually 
       *                         displayed. This is handy e.g. in order to 
       *                         prevent the progress dialog from being 
       *                         displayed at all when the worker thread 
       *                         takes only a very short time to finish.
       *                         <p>
       *                         This is a minimum only. In some situations 
       *                         it might take longer until the progress 
       *                         dialog is displayed. For instance if the 
       *                         worker thread code requests a message box
       *                         before the progress dialog is displayed.
       *                         In that case we will reset the timer so 
       *                         that the progress dialog will not show 
       *                         before a new amount of this time has 
       *                         passed.
       *                         <p>
       *                         If a value of zero is specified then we 
       *                         will display the progress dialog 
       *                         immediately.
       * @param   noExceptionMsgBox If this parameter is false then we will
       *                         automatically display a GUI error message box
       *                         to the user in case of an unhandled exception 
       *                         in the worker thread. Else, the worker 
       *                         thread will end silently but the methods 
       *                         {@link #getWorkerThreadExceptionMessage} and 
       *                         {@link #getWorkerThreadExceptionStackTrace}
       *                         can be used regardless of this parameter 
       *                         value.
       * @throws  GThreadStartException if we fail to create and start the
       *                                background worker thread.
       * @throws  GIllegalStateException if the worker thread is already running,
       *                                 or if the calling thread is not the
       *                                 GUI thread.
       * @return  True if the worker thread finished without any error, or else
       *          false. False will be returned if {@link #runTheWorkerThread}
       *          returned false or if it did throw an exception or if
       *          {@link #requestStop} was given false. Else we will return true.
       */
      bool workModal ( GWindow& parentWin,
                       int stayInvisibleTimerMillis = 800,
                       bool noExceptionMsgBox = false );

      /**
       * Get the system real time in milliseconds since the worker thread
       * begun. That is the time that has passed since the worker thread
       * called the method {@link #runTheWorkerThread}.
       * 
       * The returned value is typically used by something like a 
       * "time estimate" implemented by the subclass.
       *
       * If the method {@link #runTheWorkerThread} has not yet been
       * called then this method will return zero.
       *
       * This method should not be called while the worker thread is 
       * waiting for the user to answer some message box shown by 
       * {@link #showWorkerMessageBox}.
       *
       * @author  Leif Erik Larsen
       * @since   2001.03.13
       * @param   exclusiveMsgBox True if we shall not include the time used 
       *                          by the worker thread to wait for the user 
       *                          answering some message box, or else false.
       * @see     #getTimeMillisUsedWaitingForUserAnswers
       */
      ulonglong getTimeMillisSinceStart ( bool exclusiveMsgBox ) const;

      /**
       * Get the number of milliseconds spent by the progress thread to 
       * wait for the user answering some message box shown with 
       * {@link #showWorkerMessageBox}. 
       * 
       * The returned value may be usable 
       * e.g. in something like a "time estimate" formula implemented 
       * by the subclass.
       *
       * If the method {@link #runTheWorkerThread} has not yet been
       * called then this method will return zero.
       *
       * @author  Leif Erik Larsen
       * @since   2001.03.13
       * @see     #getTimeMillisSinceStart
       */
      ulonglong getTimeMillisUsedWaitingForUserAnswers () const;

      /**
       * Return true if and only if the {@link #workModal} 
       * method has not yet finished, regardless if the worker thread has
       * actually finished or not.
       *
       * @author  Leif Erik Larsen
       * @since   2001.01.03
       */
      bool isExecuting () const;

      /**
       * To be used only after the worker thread has finished and 
       * {@link #isSuccess} returns false.
       *
       * @author  Leif Erik Larsen
       * @since   2006.10.28
       * @see     #getWorkerThreadExceptionStackTrace
       */
      const GString& getWorkerThreadExceptionMessage () const;

      /**
       * To be used only after the worker thread has finished and 
       * {@link #isSuccess} returns false.
       *
       * @author  Leif Erik Larsen
       * @since   2006.10.28
       * @see     #getWorkerThreadExceptionMessage
       */
      const GString& getWorkerThreadExceptionStackTrace () const;

      /**
       * Get a pointer to the dialog panel that is used to monitor the
       * worker thread.
       *
       * The returned pointer will be null if the worker thread is not
       * currently inside method {@link #runTheWorkerThread}.
       *
       * @author  Leif Erik Larsen
       * @since   2001.05.05
       */
      class GDialogPanel* getMonitorDialog ();

      /**
       * Return true if {@link #requestStop} has been called since the worker
       * thread was executed.
       *
       * @author  Leif Erik Larsen
       * @since   2000.09.26
       * @see     #requestStop
       */
      bool isStopRequested () const;

      /**
       * Return true if the worker thread has run with no error or 
       * exception until now. This method is usually to be called only 
       * when after the worker thread has finished, but it can be 
       * called also while the worker thread is still running.
       * <p>
       * Will return false if an exception has occured on the worker
       * thread, or if {@link #requestStop} has been called with a false
       * argument.
       *
       * @author  Leif Erik Larsen
       * @since   2006.10.28
       * @see     #requestStop
       */
      bool isSuccess () const;

      /**
       * The GUI thread can use this method to request the worker thread to
       * stop. The worker thread will not be killed the hard way. We will
       * just set the stop request flag to true, so
       * that {@link #isStopRequested} will return true.
       *
       * If this method has already been called since the worker thread was
       * started then we will do nothing but return. This means that only the
       * first call will set the success-flag as specified to this method.
       *
       * <b>Mark</b> that this method should be used by the
       * <a HREF="#TheGUIThread">GUI thread</a> only.
       *
       * @author  Leif Erik Larsen
       * @since   2000.09.27
       * @param   success If false is given then the 
       *                  {@link #workModal} method that was
       *                  used to start the worker thread will
       *                  return false regardless if
       *                  {@link #runTheWorkerThread} did throw 
       *                  an exception or not.
       * @throws  GIllegalStateException if the calling thread is not the
       *                                 GUI thread.
       * @see     #onWorkerThreadRequestStop
       */
      void requestStop ( bool success = true );

      /**
       * The worker thread it self can use this method to synchronously send
       * a message to the GUI thread that manages the worker thread.
       *
       * The message will be posted to the monitor dialog panel and the
       * calling thread will suspend until the GUI thread has received and
       * processed the message. The synchronisation issues regarding this
       * suspend/resume are all handled automatically by this class.
       *
       * The only thing that the sub-class need to do is to override
       * the {@link #onWorkerThreadUserEvent} method to handle the message.
       *
       * @author  Leif Erik Larsen
       * @since   2000.09.27
       * @param   id       Program defined string that is typically used as
       *                   a message id/name.
       *                   This object will be received by the GUI thread as the
       *                   second parameter to {@link #onWorkerThreadUserEvent}.
       * @param   userData Any object to be sent to the GUI thread.
       *                   This object will be received by the GUI thread as the
       *                   third parameter to {@link #onWorkerThreadUserEvent}.
       * @see     #onWorkerThreadUserEvent
       * @throws  GIllegalStateException if the calling thread is not the
       *                                 worker thread.
       */
      void sendUserMessageToMonitor ( const GString& id, GObject* userData = null );

      /**
       * Show the specified message to the user, in a GUI message box.
       *
       * The calling thread must be the worker thread, or else we will
       * throw a {@link GIllegalStateException}.
       *
       * The calling worker thread will be suspended until the user has
       * answered the message box in the GUI thread. Therefore, this can
       * be assumed to be a modal call that can be used by the worker
       * thread to get answers from the user synchronously.
       *
       * @author  Leif Erik Larsen
       * @since   2000.09.27
       * @return  See {@link GMessageBox#show}.
       * @see     GWindow#showMessageBox
       * @see     GMessageBox
       * @throws  GIllegalStateException if the calling thread is not
       *                                 the worker thread.
       */
      GMessageBox::Answer showWorkerMessageBox ( const GString& msg, 
                                                 GMessageBox::Type type = GMessageBox::TYPE_NONE, 
                                                 const GString& flags = GString::Empty, 
                                                 const GString& title = GString::Empty, 
                                                 bool monoFont = false,
                                                 const GString& userText1 = GString::Empty,
                                                 const GString& userText2 = GString::Empty,
                                                 const GString& userText3 = GString::Empty );

      /**
       * This method can be called by the worker thread at any time to
       * implicitly and synchronously call the method 
       * {@link #onWorkerThreadUpdateMonitor} by the GUI thread.
       *
       * @author  Leif Erik Larsen
       * @since   2004.09.15
       */
      void updateMonitor ();

   private:

      /**
       * Called by {@link GWorkerThread::MsgProc#handleDialogMessage} only.
       */
      bool handleMsgProcMessage ( GDialogMessage& msg );

      void sendMessageToGUI ( GDialogMessage& msg );

      void showTheProgressDialog ();

   protected:

      /**
       * All methods below implements the virtuality of our base class
       * {@link GWorkerThreadAdapter}. Their defaul implementation just 
       * forewards the call to the corresponding method on the user defined
       * adapter, if any. The "user defined adapter" here means the argument 
       * as was specified to the "adapter" parameter of our constructor.
       *
       * Thus, if the user defined worker thread code is implemented by
       * being a subclass of this GWorkerThread class (and the user defined
       * adapter therefore usually being null) then any method overriding 
       * these default implementations of GWorkerThreadAdapter does not 
       * need to (and should probably not-) call its corresponding default 
       * super implementation.
       *
       * @author  Leif Erik Larsen
       * @since   2004.01.23
       * @param   worker  If subclassing then this reference will always 
       *                  be the same as the dereferenced this pointer.
       *                  Else it will be a reference to the calling 
       *                  GWorkerThread-instance.
       */
      virtual void runTheWorkerThread ( GWorkerThread& worker );

      /**
       * When the worker thread calls {@link #showWorkerMessageBox} it will 
       * make the GUI thread calling this method to actually show the 
       * message box on screen. 
       *
       * The default implementation of this method uses the standard class
       * {@link GMessageBox} to display the message box, but a subclass can
       * override this method to provide its own GUI for the message box.
       *
       * @author  Leif Erik Larsen
       * @since   2004.11.03
       * @param   ownerWin  The window to use as the owner of the modal 
       *                    message box. This is typically a reference to 
       *                    the progress dialog it self, but in can also be 
       *                    a reference to the owner of the progress dialog
       *                    if the progress dialog is not yet visible due 
       *                    to the <i>stayInvisibleTimerMillis</i> parameter 
       *                    given to {@link #workModal}.
       * @param   mi        Contains the properties of the message of 
       *                    which to show in a modal dialog message box.
       */
      virtual GMessageBox::Answer showGuiMessageBoxImpl ( GWindow& ownerWin, const MessageBoxInfo& mi );

      virtual void onWorkerThreadRequestStop ( GWorkerThread& worker, class GDialogPanel& monitor,  bool success );
      virtual void onWorkerThreadInitDialog ( GWorkerThread& worker, class GDialogPanel& monitor );
      virtual void onWorkerThreadDismissDialog ( GWorkerThread& worker, class GDialogPanel& monitor );
      virtual void onWorkerThreadCommand ( GWorkerThread& worker, class GDialogPanel& monitor, const GString& cmd );
      virtual void onWorkerThreadCtrlChanged ( GWorkerThread& worker, class GDialogPanel& monitor, const GString& compID, const GString& newValue );
      virtual void onWorkerThreadFocusSet ( GWorkerThread& worker, class GDialogPanel& monitor, const GString& focusSetID, const GString& focusLossID );
      virtual void onWorkerThreadFocusLoss ( GWorkerThread& worker, class GDialogPanel& monitor, const GString& focusLossID, const GString& focusSetID );
      virtual void onWorkerThreadUpdateMonitor ( GWorkerThread& worker, class GDialogPanel& monitor );
      virtual void onWorkerThreadHasStarted ( GWorkerThread& worker, class GDialogPanel& monitor );
      virtual void onWorkerThreadHasFinished ( GWorkerThread& worker, class GDialogPanel& monitor, bool success );
      virtual void onWorkerThreadUserEvent ( GWorkerThread& worker, class GDialogPanel& monitor, const GString& msgID, GObject* userParam );
};

#endif
