/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/gui/GTabbedPanel.h"
#include "glib/gui/GDialogPanel.h"
#include "glib/gui/GContainerWindow.h"
#include "glib/gui/GDecoratedWindow.h"
#include "glib/gui/GIcon.h"
#include "glib/gui/GAddComponentException.h"
#include "glib/gui/event/GKeyMessage.h"
#include "glib/gui/layout/GBorderLayout.h"
#include "glib/resource/GNoSuchResourceException.h"
#include "glib/exceptions/GIllegalArgumentException.h"
#include "glib/exceptions/GIllegalStateException.h"
#include "glib/GProgram.h"

GTabbedPanel::GTabbedPanel ( const GString& name,
                             const GString& constraints,
                             GWindow& parentWin,
                             long winStyle,
                             TabsPos tabsPos,
                             int iconWidth,
                             int iconHeight )
             :GWindow(name, 
                      constraints, 
                      &parentWin, 
                      &parentWin, 
                      winStyle, 
                      WS2_OS2Y | WS2_NOT_STATIC),
              peer(*this, tabsPos),
              activeTabIndex(-1),
              tabs(16),
              iconWidth(iconWidth),
              iconHeight(iconHeight),
              tabsAreVisible(true)
{
   setLayoutManager(new GBorderLayout(), true);
}

GTabbedPanel::~GTabbedPanel ()
{
   for (;;)
   {
      int num = getTabCount();
      if (num <= 0)
         break;
      removeTabAt(num - 1);
   }
}

GTabbedPanel::Peer::Peer ( GTabbedPanel& tp, TabsPos tabsPos )
             :GWindow("Peer", 
                      GBorderLayout::CENTER, 
                      &tp, 
                      &tp, 
                      WS_VISIBLE | WS_CLIPCHILDREN | BKS_TABBEDDIALOG | (tabsPos == TOP ? BKS_MAJORTABTOP : BKS_MAJORTABBOTTOM), 
                      WS2_OS2Y | WS2_NOT_STATIC | WS2_DEFAULTPAINT | 
                                 WS2_IGNORE_COLORS_AND_FONT_PROFILE | 
                                 WS2_USE_SAME_PROFILE_SECTION_NAME_AS_PARENT,
                      GWindowClass::TABBEDPANEL),
              tp(tp),
              tabsPos(tabsPos)
{
   int pageColor = GColor::GRAY;
   sendMessage(BKM_SETNOTEBOOKCOLORS, MPARAM(pageColor), MPARAM(BKA_BACKGROUNDPAGECOLOR));
}

GTabbedPanel::Peer::~Peer ()
{
}

GInsets GTabbedPanel::Peer::getPageInsets ()
{
   RECTL pageRect;
   HWND hWnd = getHWND();
   ::WinQueryWindowRect(hWnd, &pageRect);
   RECTL faceRect = pageRect;
   sendMessage(BKM_CALCPAGERECT, GWindowMessage::Param1(&faceRect), GWindowMessage::Param2(TRUE));
   GInsets ins;
   ins.top = pageRect.yTop - faceRect.yTop;
   ins.left = faceRect.xLeft - pageRect.xLeft;
   ins.bottom = faceRect.yBottom - pageRect.yBottom;
   ins.right = pageRect.xRight - faceRect.xRight;
   return ins;
}

void GTabbedPanel::Peer::layout ()
{
   GWindow::layout();
}

void GTabbedPanel::setEnabled ( bool flag, bool repaint )
{
   peer.setEnabled(flag, false);
   GWindow::setEnabled(flag, repaint);
}

void GTabbedPanel::setOily ( bool flag )
{
   peer.setOily(flag);
   GWindow::setOily(flag);
}

bool GTabbedPanel::onNotify ( int ctrlID, int notifyID, int data, int& sysAnswerToReturn )
{
   if (ctrlID != peer.getWindowID())
      return false;

   switch (notifyID)
   {
      case BKN_PAGESELECTED: 
      { 
         // Another page has been activated.
         PAGESELECTNOTIFY* psn = (PAGESELECTNOTIFY*) data;
         int sysID = psn->ulPageIdNew;
         // Find the page which is associated with this system-level ID.
         for (int i=0, num=getTabCount(); i<num; i++)
         {
            Page& page = tabs[i];
            if (page.getSysID() == sysID)
            {
               activeTabIndex = i;
               break;
            }
         }
         fireValueChangeListeners();
         break; 
      }
   }

   return true;
}

GTabbedPanel::Page::Page ( int sysID, 
                           const GString& userID, 
                           const GString& textStr,
                           const GString& hintStr,
                           bool showHintStr,
                           const GString& iconName,
                           GWindow* win, 
                           bool autoDelWin )
                   :sysID(sysID),
                    userID(userID),
                    textStr(textStr),
                    hintStr(hintStr),
                    win(win),
                    autoDelWin(autoDelWin),
                    showHintStr(showHintStr)
{
}

GTabbedPanel::Page::~Page ()
{
   if (autoDelWin)
      delete win;
}

int GTabbedPanel::Page::getSysID () const 
{ 
   return sysID; 
}

const GString& GTabbedPanel::Page::getUserID () const 
{ 
   return userID; 
}

GWindow& GTabbedPanel::Page::getWin () const 
{ 
   return *win; 
}

bool GTabbedPanel::onKeyDown ( const GKeyMessage& key )
{
   switch (key.getCode())
   {
      case GKey::KEY_CTRL_PAGEUP: // Used to turn into previous page.
      {
         int old = getActiveTabIndex();
         int idx = old - 1;
         if (idx < 0)
            idx = getTabCount() - 1;
         if (idx >= 0 && idx != old)
            activateTab(idx);
         return true; 
      }

      case GKey::KEY_CTRL_PAGEDOWN: // Used to turn into next page.
      {
         int old = getActiveTabIndex();
         int idx = old + 1;
         if (idx >= getTabCount())
            idx = 0;
         if (idx >= 0 && idx != old)
            activateTab(idx);
         return true; 
      }

      default:
         return GWindow::onKeyDown(key);
   }
}

void GTabbedPanel::activateTab ( int index )
{
   Page& page = tabs[index];
   peer.sendMessage(BKM_TURNTOPAGE, MPARAM(page.getSysID()), MPARAM(0));
}

void GTabbedPanel::activateTab ( const GString& tabID )
{
   int index = getTabIndex(tabID);
   if (index <= -1)
      gthrow_(GIllegalArgumentException(GString("Unknown tab: \"%s\"", GVArgs(tabID))));

   activateTab(index);
}

int GTabbedPanel::getActiveTabIndex () const
{
   return activeTabIndex;
}

GWindow& GTabbedPanel::getIndexedTab ( int index ) const
{
   Page& page = tabs[index];
   return page.getWin();
}

const GTabbedPanel::Page& GTabbedPanel::getIndexedTabPage ( int index ) const
{
   return tabs[index];
}

GWindow& GTabbedPanel::getTab ( const GString& tabID )
{
   int index = getTabIndex(tabID);
   if (index <= -1)
      gthrow_(GIllegalArgumentException(GString("Unknown tab: \"%s\"", GVArgs(tabID))));

   return getIndexedTab(index);
}

int GTabbedPanel::getTabCount () const
{
   return tabs.getCount();
}

int GTabbedPanel::getTabIndex ( const GString& tabID ) const
{
   const int num = tabs.getCount();
   for (int i=0; i<num; i++)
   {
      if (tabID == tabs[i].getUserID())
         return i;
   }
   return -1;
}

bool GTabbedPanel::removeTab ( const GString& tabID )
{
   int index = getTabIndex(tabID);
   if (index <= -1)
      return false;
   removeTabAt(index);
   return true;
}

void GTabbedPanel::removeTabAt ( int index )
{
   // Remove the tab it self.
   Page& page = tabs[index];
   peer.sendMessage(BKM_DELETEPAGE, MPARAM(page.getSysID()), MPFROM2SHORT(0, BKA_SINGLE));

   // Delete the associated window, if auto-delete is set.
   tabs.remove(index);
}

GDialogPanel& GTabbedPanel::addDialogTab ( const GString& dlgID,
                                           GDialogMessageHandler* msgProc,
                                           const GString& tabID,
                                           const GString& tabText,
                                           const GString& tabHint,
                                           bool showHint )
{
   try {
      GResourceTable& res = getResourceTable();
      GDialogResource* params = res.getDialogResource(dlgID);
      if (params == null)
         gthrow_(GNoSuchResourceException(GString("No such dialog resource: \"%s\"", GVArgs(dlgID))));
      GString name = tabID;
      if (name == "")
         name = dlgID;
      GDialogPanel* odlg = getOwnerDialogPanel();
      GDialogPanel* panel = new GDialogPanel(*odlg, peer, name, GString::Empty, msgProc, *params);
      GString text = (tabText == "" ? params->getTextID() : tabText);
      addTab(name, text, tabHint, showHint, GString::Empty, panel, true);
      return *panel;
   } catch (std::exception& e) {
      GString msg("Error adding tab: \"%s\". Message is: ", GVArgs(dlgID).add(e.what()));
      gthrow_(GAddComponentException(msg));
   }
}

void GTabbedPanel::addTab ( const GString& tabID,
                            const GString& textID,
                            const GString& hintID,
                            bool showHint,
                            const GString& iconName,
                            GWindow* win,
                            bool autoDelWin )
{
   if (getTabIndex(tabID) >= 0)
      gthrow_(GIllegalArgumentException(GString("Tab-ID is already in use: \"%s\"", GVArgs(tabID))));

   GString tabText = GProgram::LoadText(textID);
   GString tabHint = GProgram::LoadText(hintID, GResourceTable::LT_PREFER_HINT);
   if (tabHint == "")
      tabHint = GProgram::LoadText(textID, GResourceTable::LT_PREFER_HINT);

   // Treate {@link GDecoratedWindow}s specially, since it has a virtually
   // non-existant container parent window ({@link GContainerWindow}). It is
   // this parent container window that we must add as a tab, and not the 
   // {@link GDecoratedWindow} it self.
   GWindow* actualWinToAddonTheTab;
   GDecoratedWindow* dwin = dynamic_cast<GDecoratedWindow*>(win);
   if (dwin != null)
   {
      GContainerWindow& cwin = dwin->getContainerWin();
      cwin.setParentWindow(&peer);
      actualWinToAddonTheTab = &cwin;
   }
   else
   {
      win->setParentWindow(&peer);
      actualWinToAddonTheTab = win;
   }

   // ---
   USHORT usPageOrder = BKA_LAST;
   USHORT usPageStyle = BKA_MAJOR | BKA_AUTOPAGESIZE;
   if (showHint)
      usPageStyle |= BKA_STATUSTEXTON;
   int sysID = int(peer.sendMessage(BKM_INSERTPAGE, MPARAM(0), MPFROM2SHORT(usPageStyle, usPageOrder)));
   if (sysID == 0)
      gthrow_(GSystemErrorLevelException("Error creating tab."));

   Page* page = new Page(sysID, tabID, tabText, tabHint, showHint, iconName, actualWinToAddonTheTab, autoDelWin);
   tabs.add(page);

   if (tabHint == tabText || !showHint)
      tabHint = "";
   peer.sendMessage(BKM_SETTABTEXT, MPARAM(sysID), MPFROMP(tabText.cstring()));
   peer.sendMessage(BKM_SETSTATUSLINETEXT, MPARAM(sysID), MPFROMP(tabHint.cstring()));
   peer.sendMessage(BKM_SETPAGEWINDOWHWND, MPARAM(sysID), MPFROMLONG(actualWinToAddonTheTab->getHWND()));

   // Activate the first tab automatically.
   if (getTabCount() == 1)
      activateTab(0);
}

bool GTabbedPanel::isEmpty () const
{
   return getTabCount() <= 0;
}

void GTabbedPanel::changeValue ( const GString& newValue, bool notify )
{
   int idx;
   try {
      idx = GInteger::ParseInt(newValue);
   } catch (GNumberFormatException& /*e*/) {
      gthrow_(GIllegalArgumentException(newValue));
   }

   enterCtrlChangedFilter();
   activateTab(idx);
   exitCtrlChangedFilter();

   if (notify)
      fireValueChangeListeners();
}

GString GTabbedPanel::queryValue () const
{
   int index = getActiveTabIndex();
   return GInteger::ToString(index);
}

GWindow& GTabbedPanel::getComponentByID ( const GString& id )
{
   return getTab(id);
}

GWindow& GTabbedPanel::getComponentByIndex ( int index )
{
   return getIndexedTab(index);
}

int GTabbedPanel::getComponentCount () const
{
   return getTabCount();
}

void GTabbedPanel::setTabsVisible ( bool visible )
{
   if (visible != tabsAreVisible)
   {
      tabsAreVisible = visible;
      layout();
   }
}

void GTabbedPanel::layout ()
{
   if (tabsAreVisible)
   {
      setInsets(null, true);
   }
   else
   {
      GInsets ins = peer.getPageInsets();
      // Adjust insets in order for the face area to occupy all screen space
      // inside the tab page rectangle.
      ins.left = -ins.left;
      ins.top = -ins.top;
      ins.right = -ins.right;
      ins.bottom = -ins.bottom;
      setInsets(&ins, false);
   }
   GWindow::layout();
}
