/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_STATICTEXT
#define __GLIB_STATICTEXT

#include "glib/gui/GWindow.h"

/**
 * This is the low level window class that is used to paint a static
 * text in a container window.
 *
 * @author  Leif Erik Larsen
 * @since   2000.01.29
 */
class GStaticText : public GWindow
{
   public:

      /** The default font for static text windows. */
      static const GString DefaultFont;

      enum ADJUST
      {
         LEFT,
         RIGHT,
         CENTER,

         /** Same as LEFT, but with a few pixels left margine. */
         LEFTMARGINE
      };

      enum VADJUST
      {
         TOP,
         BOTTOM,
         VCENTER
      };

   private:

      ADJUST adjust;
      VADJUST vadjust;
      bool wordWrap;
      bool useLeader;
      bool useHotKey;
      int hotKeyPos;
      GString text;
      mutable GTooltip::Position overriddenTooltipPos;

   public:

      GStaticText ( const GString& name,
                    const GString& constraints,
                    GWindow& parentWin,
                    long winStyle = WS_VISIBLE,
                    long winStyle2 = 0,
                    ADJUST adjust = LEFT,
                    bool leader = false,
                    bool hotKey = false,
                    GWindow::LeaderStyle style = LeaderStyle_DEFAULT,
                    bool wordWrap = false,
                    VADJUST vadjust = VCENTER );

      virtual ~GStaticText ( void );

   private:

      /** Disable the copy constructor. */
      GStaticText ( const GStaticText& src ) {}

      /** Disable the assignment operator. */
      GStaticText& operator= ( const GStaticText& ) { return *this; }

   private:

      /**
       * To be called by {@link #paintText} only.
       *
       * @author  Leif Erik Larsen
       * @since   2000.01.29
       * @param   g         The graphics object of where to draw the text.
       * @param   frgColor  Text color.
       * @param   textLine  The text of which to draw.
       * @param   ypos      Y-position of the lower part of where to draw
       *                    the text. The bottom of the text will occur at
       *                    this position.
       * @param   lineCount Total number of lines that are about to be painted.
       * @param   curLine   Index of which line this call is about.
       * @param   displace  Number of pixels to displace the text position.
       */
      void paintTextLine ( GGraphics& g, const GColor& frgColor, const GString& textLine, int ypos, int lineCount, int curLine, int displace );

      /**
       * To be called by {@link #onPaint} only.
       */
      void paintText ( GGraphics& g, const GColor& frgColor, int displace );

   protected:

      virtual bool onPaint ( GGraphics& g, const GRectangle& rect );
      virtual bool onPaintBackground ( GGraphics& g, const GRectangle& rect );

   public:

      virtual void changeValue ( const GString& newValue, bool notify = true );
      virtual GString getText () const;
      virtual bool isEmpty () const;
      virtual GString queryValue () const;
      virtual void setText ( const GString& text );
      virtual int getPreferredWidth () const;
      virtual GTooltip::Position getTooltipPosition () const;
      virtual GString getTooltipText () const;
};

#endif
