/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_SPINNER
#define __GLIB_SPINNER

#include "glib/gui/GTextEntry.h"
#include "glib/util/GValueChangeListener.h"

/**
 * This is the low level window class that implements a general
 * integer value spinner control.
 *
 * This Control can be nice to use when and where a numeric data or an
 * index is to be entered. For example, a Control of where to enter a date
 * can contain three instances of a SPINNER. The first can contain the
 * day number within month (1-31), the second can contain the month
 * number (1-12) and the third can contain the year.
 *
 * You can easily limit the value range to be edited. You can also link
 * each value to a corresponding text so that the user sees a text value
 * rather than the actual value which is used by the program code. This
 * makes it possible, for instance, to present each of the months of a
 * year in their text form, which probably looks more intuitive to
 * most users.
 *
 * The user editable value of this Component will be an integer
 * and it will contain the current spinner index value.
 *
 * @author  Leif Erik Larsen
 * @since   2000.07.14
 */
class GSpinner : public GWindow
{
   public:

      enum ADJUST
      {
         LEFT,
         RIGHT,
         CENTER
      };

      class Peer : public GWindow
      {
         friend class GSpinner;
         GSpinner& owner;
         Peer ( GSpinner& owner, GSpinner::ADJUST adjust );
         virtual ~Peer ();
      };

      mutable Peer peer;

   private:

      int minValue;
      int maxValue;
      int stepValue;

      /** Strings to display in entry instead of numerical index value, if != null. */
      GArray<GString>* stringMap;

   public:

      /**
       * @param   stringMap Array of strings to use to represent
       *                    each index value. Must be exactly as many items
       *                    in the array as (<i>max</i> - <i>min</i> + 1)
       *                    if vector is not empty, or else
       *                    unpredictable behaviour will take place.
       */
      GSpinner ( const GString& name,
                 const GString& constraints,
                 GWindow& parentWin,
                 long winStyle = WS_VISIBLE,
                 long winStyle2 = WS2_DEFAULTPAINT,
                 ADJUST adjust = RIGHT,
                 int min = 1,
                 int max = 100,
                 int step = 1,
                 const GArray<GString>* stringMap = null );

      virtual ~GSpinner ();

   private:

      /** Disable the copy constructor. */
      GSpinner ( const GSpinner& );

      /** Disable the assignment operator. */
      void operator= ( const GSpinner& );

   private:

      virtual bool onKeyDown ( const GKeyMessage& key );
      virtual void grabFocus ( bool force = false );

   protected:
      virtual bool onNotify ( int ctrlID, int notifyID, int data, int& sysAnswerToReturn );

   public:

      virtual void changeValue ( const GString& newValue, bool notify = true );
      virtual bool isEmpty () const;
      virtual GString queryValue () const;

      virtual void setEnabled ( bool flag = true, bool repaint = true );
      virtual void setOily ( bool flag );

      /**
       * Get the current selected value, with or without range checking.
       */
      int getCurrentValue () const;

      int getMaximumValue () const;
      int getMinimumValue () const;
      void setMaximumValue ( int max );
      void setMinimumValue ( int min );
      void setValue ( int value, bool notify = true );

      /**
       *
       * @author  Leif Erik Larsen
       * @since   2001.03.10
       * @return  A copy of the new value after it has been spinned.
       */
      int spinUp ();

      /**
       *
       * @author  Leif Erik Larsen
       * @since   2001.03.10
       * @return  A copy of the new value after it has been spinned.
       */
      int spinDown ();

      bool valueIsLegal ( int value );
};

#endif

