/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/gui/GPushButton.h"
#include "glib/gui/GDialogFrame.h"
#include "glib/gui/GDialogPanel.h"
#include "glib/gui/GIcon.h"
#include "glib/gui/layout/GBorderLayout.h"
#include "glib/gui/event/GKeyMessage.h"
#include "glib/gui/event/GUserMessage.h"
#include "glib/sys/GSystem.h"
#include "glib/util/GMath.h"

const char GPushButton::MNEMONIC_MARKER_CHAR = '~';

GPushButton::Peer::Peer ( GPushButton& button )
                  :GWindow("Peer",
                           GBorderLayout::CENTER,
                           &button,
                           &button,
                           WS_VISIBLE | BS_PUSHBUTTON | BS_USERBUTTON,
                           WS2_DEFAULTPAINT | WS2_OS2Y | WS2_NOT_STATIC | WS2_AUTO_SHOW_TOOLTIP | WS2_USE_SAME_PROFILE_SECTION_NAME_AS_PARENT,
                           GWindowClass::BUTTON),
                   button(button)
{
   setFocusable(true);
}

GPushButton::Peer::~Peer ()
{
}

void GPushButton::Peer::onFocus ( bool set )
{
   GWindow::onFocus(set);

   bool inclChildren = true;
   if (!button.noFocusPaint)
      postMessage(GM_ASYNC_PAINT, GWindowMessage::Param1(inclChildren));

   // Invalidate the default push button of our containing dialog, if any.
   GWindow& top = getTopLevelWindow();
   GDialogFrame* frame = dynamic_cast<GDialogFrame*>(&top);
   if (frame != null && frame->isExecutingAsDialog())
   {
      GDialogPanel& dlg = frame->getDialogPanel();
      GPushButton* def = dlg.getDefaultPushButton();
      if (def != null && !def->noFocusPaint)
         def->peer.postMessage(GM_ASYNC_PAINT, GWindowMessage::Param1(inclChildren));
   }
}

GWindowMessage::Answer GPushButton::Peer::handleWindowMessage ( GWindowMessage& msg )
{
   switch (msg.getID())
   {
      case WM_FOCUSCHANGE:
      {
         bool set = msg.getParam2LoShort();
         GWindow::callDefaultMsgProc(msg);
         onFocus(set);
         return 0;
      }

      default:
         return GWindow::handleWindowMessage(msg);
   }
}

GPushButton::Command::Command ( const GString& id )
                     :GAbstractCommand(id)
{
}

GPushButton::Command::~Command ()
{
}

GPushButton::GPushButton ( const GString& name,
                           const GString& constraints,
                           GWindow& parentWin,
                           long winStyle,
                           long winStyle2,
                           const GString& iconID,
                           bool isDefault,
                           bool noFocusPaint,
                           ICONPOS iconPos,
                           bool noBoldBorder,
                           bool useHotKey )
            :GWindow(name,
                     constraints,
                     &parentWin,
                     &parentWin,
                     winStyle,
                     winStyle2 | WS2_OS2Y | WS2_NOT_STATIC | WS2_DONT_SEND_CTRL_CHANGE_MSG),
             iconID(iconID),
             iconPos(iconPos),
             isDefault(isDefault),
             hotKeyPos(-1),
             useHotKey(useHotKey),
             noFocusPaint(noFocusPaint),
             noBoldBorder(noBoldBorder),
             peer(*this),
             command(name)
{
   setLayoutManager(new GBorderLayout(), true);
   setBackgroundColor(GSystem::GetSystemColor(GSystem::SCID_BUTTONBCK));
   setForegroundColor(GSystem::GetSystemColor(GSystem::SCID_BUTTONTXT));
}

GPushButton::~GPushButton ()
{
}

bool GPushButton::onNotify ( int ctrlID, int notifyID, int data, int& sysAnswerToReturn )
{
   if (ctrlID != peer.getWindowID())
      return false;

   switch (notifyID)
   {
      case BN_PAINT: // The button needs to be repainted.
      {
         USERBUTTON* ub = (USERBUTTON*) data;
         bool pressed = ((ub->fsState & BDS_HILITED) != 0);
         GGraphics g(ub->hps, peer, false, false);
         repaintImpl(g, pressed);
         break;
      }
   }

   return true;
}

GWindowMessage::Answer GPushButton::handleWindowMessage ( GWindowMessage& msg )
{
   switch (msg.getID())
   {
      case WM_FOCUSCHANGE:
      {
         bool set = msg.getParam2LoShort();
         GWindow::callDefaultMsgProc(msg);
         peer.onFocus(set);
         return 0;
      }

      case GM_CREATE_CARET:
      {
         if (!noFocusPaint && containsFocusedWindow() && isActive())
         {
            GRectangle r = peer.getWindowRect();
            if (r.width <= 20)
               r.inflateRect(-3, -3);
            else
               r.inflateRect(-6, -6);
            HWND hwnd = peer.getHWND();
            ::WinCreateCursor(hwnd, r.x, r.y, r.width, r.height, CURSOR_FRAME | CURSOR_HALFTONE, null);
            ::WinShowCursor(hwnd, true);
         }
         return 0;
      }

      default:
         return GWindow::handleWindowMessage(msg);
   }
}

bool GPushButton::onCommand ( int cmd )
{
   if (cmd == peer.getWindowID())
   {
      const GString& name = getName();
      GAbstractCommand acmd(name);
      if (executeAbstractCommand(&acmd))
         return true;
   }

   return GWindow::onCommand(cmd);
}

bool GPushButton::onKeyDown ( const GKeyMessage& key )
{
   switch (key.getCode())
   {
      case GKey::KEY_ENTER:
         doClick();
         return true;

      default:
         return GWindow::onKeyDown(key);
   }
}

void GPushButton::grabFocus ( bool force )
{
   peer.grabFocus(force);
}

void GPushButton::setDefault ( bool flag )
{
   isDefault = flag;
   invalidateAll(true);
}

void GPushButton::setEnabled ( bool flag, bool repaint )
{
   peer.setEnabled(flag, false);
   GWindow::setEnabled(flag, repaint);
}

void GPushButton::setOily ( bool flag )
{
   peer.setOily(flag);
   GWindow::setOily(flag);
}

void GPushButton::doClick ()
{
   if (!isEnabled())
      return;
   const GString& cmdID = command.getIDString();
   if (cmdID != "") // Don't execute command if it has no ID!
      postCommand(&command);
}

void GPushButton::hotKeyAction ()
{
   grabFocus();
   doClick();
}

int GPushButton::getPreferredHeight () const
{
   return getHeightOfString("X") + 12;
}

int GPushButton::getPreferredWidth () const
{
   GString text = getText();
   return getWidthOfString(text) + 36;
}

void GPushButton::setIconID ( const GString& iconID )
{
   this->iconID = iconID;
   invalidateAll(true);
}

void GPushButton::setText ( const GString& text )
{
   GString prevText = getText();
   if (text == prevText)
      return;

   GWindow::setText(text);
   peer.setText(text);

   if (useHotKey)
      hotKeyPos = autoAssocAHotKey(text);

   invalidateAll(true);
}

void GPushButton::repaintImpl ( GGraphics& g, bool pressed )
{
   bool boldBorder = false;
   if (!noBoldBorder && isEnabled())
   {
      if (peer.hasFocus())
         boldBorder = true;
      else
      if (dynamic_cast<Peer*>(getTopLevelWindow().getFocusedWindow()) != null)
         boldBorder = false;
      else
         boldBorder = isDefault;
   }

   GColor bck = getBackgroundColor();
   GColor box = GSystem::GetSystemColor(GSystem::SCID_BUTTONDBOX);
   GColor light = GSystem::GetSystemColor(GSystem::SCID_3DLIGHT);
   GColor dark = GSystem::GetSystemColor(GSystem::SCID_3DSHADOW);
   GColor upperLeftColor[] =
   {
      boldBorder ? box : dark,
      pressed ? dark : light,
      pressed ? dark : light
   };

   GColor lowerRightColor[] =
   {
      boldBorder ? box : light,
      pressed ? light : dark,
      pressed ? light : dark
   };

   GRectangle r = getWindowRect();

   if (!noBoldBorder)
   {
      g.setColor(boldBorder ? GColor::BLACK : bck);
      g.drawRectangle(r);
      r.inflateRect(-1);
   }

   for (int i=0; i<3; i++)
   {
      g.setColor(upperLeftColor[i]);
      g.setPosition(r.x, r.y);
      g.drawLineTo(r.x, r.y+r.height-1);
      g.drawLineTo(r.x+r.width-1, r.y+r.height-1);
      g.setColor(lowerRightColor[i]);
      g.drawLineTo(r.x+r.width-1, r.y);
      g.drawLineTo(r.x, r.y);
      r.inflateRect(-1);
   }

   g.drawFilledRectangle(r, bck);

   GString txt = getText();
   int displace = (pressed ? 1 : 0);
   bool grayed = !isEnabled();
   const GIcon* icon = GIcon::GetIcon(iconID, grayed);
   if (icon != null)
   {
      if (txt == "" || iconPos == IP_CENTER)
      {
         int xpos = GMath::Max(0, r.x + (r.width/2) - ((icon->getWidth())/2));
         int ypos = GMath::Max(0, r.y + (r.height/2) - (icon->getHeight()/2));
         g.drawIcon(xpos + displace, ypos - displace, *icon);
         return;
      }
      else
      switch (iconPos)
      {
         case IP_BOTTOM:
         {
            int xpos = GMath::Max(0, r.x + (r.width/2) - ((icon->getWidth())/2) - 1);
            r.y += 3;
            g.drawIcon(xpos + displace, r.y - displace, *icon);
            r.y += icon->getHeight() + 1;
            break;
         }

         case IP_TOP:
         {
            int xpos = GMath::Max(0, r.x + (r.width/2) - ((icon->getWidth())/2) - 1);
            r.height -= icon->getHeight() + 3;
            g.drawIcon(xpos + displace, r.y + r.height - 1 - displace, *icon);
            r.height -= 1;
            break;
         }

         case IP_LEFT:
         {
            int txtWidth = (txt == "" ? 0 : g.getWidthOfString(txt) + 4);
            int totWidth = icon->getWidth() + txtWidth;
            r.x = GMath::Max(0, r.width/2 - totWidth/2);
            int ypos = GMath::Max(0, r.y + (r.height/2) - (icon->getHeight()/2) - 1);
            g.drawIcon(r.x + displace, ypos + 1 - displace, *icon);
            r.x += icon->getWidth() + 4;
            break;
         }

         case IP_RIGHT:
         {
            int ypos = GMath::Max(0, r.y + (r.height/2) - (icon->getHeight()/2) - 1);
            r.width -= icon->getWidth() + 5;
            g.drawIcon(r.x + r.width - 1 + displace, ypos + 1 - displace, *icon);
            r.width -= 1;
            break;
         }

         case IP_CENTER:
            break;
      }
   }

   // Paint the text horizontally and vertically centered on the button face
   if (txt != "")
   {
      int height = g.getHeightOfString(txt);
      int ypos = GMath::Max(0, r.y + (r.height/2) - (height/2));
      int xpos;
      if (icon == null)
      {
         int width = g.getWidthOfString(txt);
         xpos = GMath::Max(0, r.x + (r.width/2) - (width/2) - 1);
      }
      else
         xpos = r.x;
      if (grayed)
      {
         // Button is disabled, so paint a grayed text.
         g.setColor(GColor::WHITE);
         g.drawTextMnemonic(xpos + displace, ypos - displace, txt, hotKeyPos);
         g.setColor(GColor::DGRAY);
         g.drawTextMnemonic(xpos - 1 + displace, ypos + 1 - displace, txt, hotKeyPos);
      }
      else
      {
         // Button is enabled, so paint text using normal colors.
         g.setColor(getForegroundColor());
         g.drawTextMnemonic(xpos + displace, ypos - displace, txt, hotKeyPos);
      }
   }

   // Paint the dotted "caret" around the button face, if the button has focus.
   postMessage(GM_CREATE_CARET);
}
