/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_MULTILINEEDITOR
#define __GLIB_MULTILINEEDITOR

#include "glib/gui/GAbstractTextField.h"

/**
 * This is the low level window class that implements a multline text editor.
 *
 * @author  Leif Erik Larsen
 * @since   2000.01.29
 */
class GMultiLineEditor : public GAbstractTextField
{
   private:

      /**
       * @author  Leif Erik Larsen
       * @since   2004.08.27
       */
      class Peer : public GWindow
      {
         friend class GMultiLineEditor;
         GMultiLineEditor& theEntry;
         Peer ( GMultiLineEditor& theEntry );
         virtual ~Peer ();
         virtual bool onKeyDown ( const GKeyMessage& key );
         // Disable copy-constructor and assignment-operator. */
         Peer ( const Peer& );
         void operator= ( const Peer& );
      };

      /**
       * @author  Leif Erik Larsen
       * @since   2006.04.28
       */
      class PeerParent : public GWindow
      {
         friend class GMultiLineEditor;
         GMultiLineEditor& theEntry;
         PeerParent ( GMultiLineEditor& theEntry );
         virtual ~PeerParent ();
         virtual bool onNotify ( int ctrlID, int notifyID, int data, int& sysAnswerToReturn );
         // Disable copy-constructor and assignment-operator. */
         PeerParent ( const PeerParent& );
         void operator= ( const PeerParent& );
      };

      friend class Peer;
      friend class PeerParent;

      /**
       * An intermediate layer window that can be used to workaround some 
       * MLE limitations on some systems, such as e.g. the stupid fixed 
       * MLE-margine in OS/2. This window layer is also practical in order 
       * to hide som implementation, and make it safer to subclass 
       * GMultiLineEditor.
       *
       * @author  Leif Erik Larsen
       * @since   2006.04.28
       */
      mutable PeerParent peerParent;

      /** The system dependent MLE control window. */
      mutable Peer peer;

      bool ignoreTab;
      bool useAsSingleLine;

   public:

      /**
       * @author  Leif Erik Larsen
       * @since   2004.09.18
       * @param   name        Name of the window.
       * @param   constraints Layout constraints.
       * @param   parentWin   The parent window.
       * @param   winStyle    System defined window style flags.
       * @param   winStyle2   GLib-extended window style flags.
       * @param   maxTextLen  Max characters in text, or 0 means "no limit".
       * @param   wordWrap    True to enable auto-wordwrap by default.
       * @param   sysBorder   True if we should use the system default
       *                      3D-border of the entry field. If false is 
       *                      specified you can assign any other border 
       *                      with {@link GWindow#setBorder} later.
       * @param   readOnly    True if text is read-only.
       * @param   ignoreTab   True if tab-character should not be handled.
       * @param   useAsSingleLine True if the multiline editor is to be used
       *                      as an extended single-line entry field.
       */
      GMultiLineEditor ( const GString& name,
                         const GString& constraints,
                         GWindow& parentWin,
                         long winStyle = WS_VISIBLE,
                         long winStyle2 = WS2_DEFAULTPAINT,
                         int maxTextLen = 0,
                         bool wordWrap = false,
                         bool sysBorder = true,
                         bool readOnly = false,
                         bool ignoreTab = false,
                         bool useAsSingleLine = false );

      virtual ~GMultiLineEditor ();

   private:

      /** Disable the copy constructor. */
      GMultiLineEditor ( const GMultiLineEditor& );

      /** Disable the assignment operator. */
      void operator= ( const GMultiLineEditor& );

   public:

      // Methods that implements the pure virtual methods of our parent class.

      virtual void appendText ( const GString& text );
      virtual void copy () const;
      virtual void cut ();
      virtual int getPreferredHeight () const;
      virtual int getPreferredWidth () const;
      virtual int getSelectionEnd () const;
      virtual int getSelectionStart () const;
      virtual int getTextLength () const;
      virtual void paste ();
      virtual void setMaxTextLength ( int maxLength );
      virtual void setSelection ( int selStart, int selEnd );
      virtual void setText ( const GString& text );
      virtual GWindow& getPeer () const;

   protected:

      virtual bool onBackgroundColorChanged ( const GColor& color );
      virtual bool onFontNameSizeChanged ( const GString& fontNameSize );
      virtual bool onForegroundColorChanged ( const GColor& color );

   public:

      /**
       * Get the index of the column in the current line of where the
       * caret is currently located.
       */
      int getCaretColumnPos () const;

      /**
       * Get the index of the line of which currently contains the caret.
       */
      int getCaretLineIndex () const;

      /**
       * Get the current caret position index within the current text.
       */
      int getCaretPosition () const;

      /**
       * Get the current number of lines in the editor.
       *
       * The returned count is the count after the lines have been
       * eventually word-wrapped. Thus, the return is not the CR-LF count,
       * except if word-wrap mode is off.
       */
      int getLineNum () const;

      /**
       * Get the text of the line that currently containes the caret.
       */
      GString getLineText ( int lineIndex ) const;

      /**
       * Get a copy of the part of the word in the text that is before the
       * current caret position.
       */
      GString getPartOfWordBeforeCaret () const;

      /**
       * Return true if and only if the "ignore tab" property is on.
       * The "ignore tab" property is set via the constructor 
       * argument "ignoreTab" only (no explicit setter method).
       *
       * @author  Leif Erik Larsen
       * @since   2004.05.18
       */
      bool isIgnoreTab () const;

      void replaceSelection ( const GString& text );
};

#endif
