/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_MESSAGEBOX
#define __GLIB_MESSAGEBOX

#include "glib/gui/event/GDialogMessageHandler.h"
#include "glib/util/GArray.h"
#include "glib/primitives/GInteger.h"

/**
 * Class used to create and show simple message boxes to interact with the
 * user. Only modal message boxes are supported. This class will typically not
 * be used directly by programs. Most programs are better using the method
 * {@link GWindow#showMessageBox} or {@link GProgram#showMessageBox}.
 *
 * A message box typically consist of four parts:
 *
 * <ol>
 * <li> The window box.
 * <li> An icon.
 * <li> The message text it self.
 *      With or without an additional "details" message string.
 * <li> One or more push buttons.
 * </ol>
 *
 * The point of a message box is to show a useful message to the user, and
 * wait for him or her to respond to the message by clicking or choosing
 * one of the push buttons. The message can be a question, an error message,
 * a general information text or something else. Depending on the type of
 * message you can choose to have one or more push buttons presented in the
 * box. Each of the buttons represents the possible answers that the user can
 * say. The ID of the selected button will be returned back to the program
 * that created the message box.
 *
 * @author  Leif Erik Larsen
 * @since   2000.09.27
 * @see     GWindow#showMessageBox
 * @see     GProgram#showMessageBox
 */
class GMessageBox : public GObject, public GDialogMessageHandler
{
   public:

      enum Answer
      {
         /**
          * This ID is for uninitialized answers only. It will never be
          * returned from the {@link #show} method.
          */
         IDNONE = 0,

         /**
          * Returned from <i>show()</i> if user selects the Abort-button.
          * The <code>IDABORT</code> button is typically used in conjunction with
          * an error message and the user should have the possibility to
          * <code>IDABORT</code>, <code>IDRETRY</code> or <code>IDIGNORE</code>.
          */
         IDABORT,

         /**
          * Returned from <i>show()</i> if user selects the Retry-button.
          * The <code>IDRETRY</code> button is typically used in conjunction with
          * an error message and the user should have the possibility to
          * <code>IDABORT</code>, <code>IDRETRY</code> or <code>IDIGNORE</code>.
          */
         IDRETRY,

         /**
          * Returned from <i>show()</i> if user selects the Ignore-button.
          * The <code>IDIGNORE</code> button is typically used in conjunction with
          * an error message and the user should have the possibility to
          * <code>IDABORT</code>, <code>IDRETRY</code> or <code>IDIGNORE</code>.
          */
         IDIGNORE,

         /**
          * Returned from <i>show()</i> if user selects the Yes-button.
          */
         IDYES,

         /**
          * Returned from <i>show()</i> if user selects the Yes-to-all-button.
          */
         IDYESTOALL,

         /**
          * Returned from <i>show()</i> if user selects the No-button.
          */
         IDNO,

         /**
          * Returned from <i>show()</i> if user selects the Cancel-button, if
          * he press the ESC-key on keyboard or if he clicks on the window-close
          * button of the message box window. The window-close button will be
          * enabled on the window only if the cancel-button actually exist on the
          * message box.
          */
         IDCANCEL,

         /**
          * Returned from <i>show()</i> if user selects the OK-button.
          */
         IDOK,

         /**
          * Returned from <i>show()</i> if user selects the Skip-button.
          */
         IDSKIP,

         /**
          * Returned from <i>show()</i> if user selects the Skip-all-button.
          */
         IDSKIPALL,

         /**
          * Returned from <i>show()</i> if the specified time-out value
          * occured before the user did select any button.
          */
         IDTIMEOUT,

         /**
          * Returned from <code>show</code>() if user selects the Help-button.
          */
         IDHELP,

         /**
          * Returned from <i>show()</i> if user selects the Fail-button.
          */
         IDFAIL,

         /**
          * Returned from <i>show()</i> if user selects the Details-button
          * and the text in <code>userText1</code> is empty.
          */
         IDDETAILS,

         /**
          * Returned from <i>show()</i> if user selects the 
          * first user defined button.
          */
         IDUSER1,

         /**
          * Returned from <i>show()</i> if user selects the 
          * second user defined button.
          */
         IDUSER2,

         /**
          * Returned from <i>show()</i> if user selects the 
          * third user defined button.
          */
         IDUSER3
      };

      enum Type
      {
         /**
          * Flag used to identify message boxes that contains a message of an
          * anoynous type, w/no default icon and no default title string.
          */
         TYPE_NONE = 0,

         /**
          * Flag used to identify message boxes that contains some
          * general message, w/the logo icon and title string by default.
          */
         TYPE_LOGO,

         /**
          * Flag used to identify message boxes that contains some type of a
          * question message, w/the confirmation icon and title string by default.
          */
         TYPE_QUESTION,

         /**
          * Flag used to identify message boxes that contains some type of an
          * error message, w/the error icon and title string by default.
          */
         TYPE_ERROR,

         /**
          * Flag used to identify message boxes that contains some type of a
          * warning message, w/the warning icon and title string by default.
          */
         TYPE_WARNING,

         /**
          * Flag used to identify message boxes that contains some type of
          * information, w/the information icon and title string by default.
          */
         TYPE_INFO,

         /**
          * Flag used to identify message boxes that contains a critical error,
          * w/the critical error icon and title string by default.
          */
         TYPE_CRITICAL_ERROR
      };

   private:

      GString titleStr;
      GString iconID;
      int iNrOfButtons;
      int defaulBt; // Index of the default button, if any.
      int cancelBt; // Index of the Cancel-button, if any.
      GArray<GString> btTexts;
      GArray<GInteger> btIDs; // Each element is an {@link #Answer}.
      GArray<GString> buttons; // Button IDs as strings.
      int timeOutSeconds; // Number of timeout seconds.
      bool doBeep; // True if we shall perform a beep when showing the box.
      class GDialogFrame* dlgFrame;
      bool hasDetails; // True if the "Details" button is defined.
      GString detailsStr; // Details text from <code>userText1</code>.

      /** The time stamp (in milliseconds) of message box was displayed. */
      ulonglong startTimeStamp;

   public:

      /**
       * Create a new message box. To show it and get the user response you must
       * use {@link #show} on the new instance.
       *
       * <b>Supported message box parameter flags/characters are:</b><br>
       * <ul>
       * <li><b><code>a</code>:</b> Abort button.
       * <li><b><code>r</code>:</b> Retry button.
       * <li><b><code>f</code>:</b> Fail button.
       * <li><b><code>i</code>:</b> Ignore button.
       * <li><b><code>h</code>:</b> Help button.
       * <li><b><code>o</code>:</b> OK Button.
       * <li><b><code>c</code>:</b> Cancel button.
       * <li><b><code>s</code>:</b> Skip button.
       * <li><b><code>s!</code>:</b> Skip all button.
       * <li><b><code>y</code>:</b> Yes button.
       * <li><b><code>y!</code>:</b> Yes to all button.
       * <li><b><code>n</code>:</b> No button.
       * <li><b><code>d</code>:</b> Details button. Details text must be given in userText1. If <code>userText1</code> is empty then we will return {@link #IDDETAILS} from {@link #show}.
       * <li><b><code>(u1)</code>:</b> First user defined button.
       * <li><b><code>(u2)</code>:</b> Second user defined button.
       * <li><b><code>(u3)</code>:</b> Third user defined button.
       * <li><b><code>b</code>:</b> Beep.
       * <li><b><code>Tn</code>:</b> Timeout after <code>n</code> seconds.
       * <li><b><code>1</code>:</b> Standard icon #1 (ICON_INFORMATION).
       * <li><b><code>2</code>:</b> Standard icon #2 (ICON_QUESTION).
       * <li><b><code>3</code>:</b> Standard icon #3 (ICON_ERROR).
       * <li><b><code>4</code>:</b> Standard icon #4 (ICON_WARNING).
       * <li><b><code>5</code>:</b> Standard icon #5 (same as #1).
       * </ul>
       *
       * <b>Mark:</b> Use uppercase letters to make the corresponding
       * button the default.
       *
       * @author  Leif Erik Larsen
       * @since   2000.09.27
       * @param   type      The type of the message. This flag is used to
       *                    define the default properties of the message
       *                    box, such as its icon and title string, in case
       *                    they are not defined in the <i>flags</i>
       *                    parameter. Supported values are:
       *                    {@link #TYPE_NONE}, {@link #TYPE_LOGO},
       *                    {@link #TYPE_QUESTION}, {@link #TYPE_ERROR},
       *                    {@link #TYPE_WARNING}, {@link #TYPE_INFO} and
       *                    {@link #TYPE_CRITICAL_ERROR}.
       * @param   msg       The message string it self.
       * @param   title     The text to show in the title bar of the box.
       * @param   flags     A string defining the properties of the message
       *                    box. See the above list of supported characters.
       * @param   monoFont  True to use monospace font to draw the message
       *                    text, or else false. Should usually be false.
       * @param   userText1 The text of the first user defined button. This 
       *                    text is used only if the <i>flags</i> contains 
       *                    the tag for the first user defined button.
       * @param   userText2 The text of the second user defined button. This 
       *                    text is used only if the <i>flags</i> contains 
       *                    the tag for the second user defined button.
       * @param   userText3 The text of the third user defined button. This 
       *                    text is used only if the <i>flags</i> contains 
       *                    the tag for the third user defined button.
       */
      GMessageBox ( Type type,
                    const GString& msg,
                    const GString& title,
                    const GString& flags,
                    bool monoFont,
                    const GString& userText1 = GString::Empty,
                    const GString& userText2 = GString::Empty,
                    const GString& userText3 = GString::Empty );

      virtual ~GMessageBox ();

   public:

      /**
       * Show the message box and wait for the user to respond.
       *
       * <b>Possible return values:</b><br>
       * <ul>
       * <li>{@link #IDABORT} if "Abort" was selected.
       * <li>{@link #IDCANCEL} if "Cancel" was selected or "Escape" was typed.
       * <li>{@link #IDIGNORE} if "Ignore" was selected.
       * <li>{@link #IDNO} if "No" was selected.
       * <li>{@link #IDOK} if "OK" was selected.
       * <li>{@link #IDRETRY} if "Retry" was selected.
       * <li>{@link #IDYES} if "Yes" was selected.
       * <li>{@link #IDYESTOALL} if "Yes to all" was selected.
       * <li>{@link #IDSKIP} if "Skip" was selected.
       * <li>{@link #IDSKIPALL} if "Skip all to all" was selected.
       * <li>{@link #IDTIMEOUT} if a timeout occured before the user responded.
       * <li>{@link #IDHELP} if "Help" was selected.
       * <li>{@link #IDUSER1} if the first user defined button was selected.
       * <li>{@link #IDUSER2} if the second user defined button was selected.
       * <li>{@link #IDUSER3} if the third user defined button was selected.
       * </ul>
       *
       * @author  Leif Erik Larsen
       * @since   2000.09.28
       * @param   ownerWin  Pointer to which window is to be used as the
       *                    owner of the message box, or null if it should
       *                    not display modally on any owner.
       */
      GMessageBox::Answer show ( class GWindow* ownerWin );

   private:

      /**
       * This is the message handler of the message box it self.
       *
       * @author  Leif Erik Larsen
       * @since   2000.09.28
       */
      bool handleDialogMessage ( class GDialogMessage& msg );
};

#endif


